#!/bin/bash
# set -x
###############################################################################
#
# Licensed Materials - Property of IBM
#
# (C) Copyright IBM Corp. 2024. All Rights Reserved.
#
# US Government Users Restricted Rights - Use, duplication or
# disclosure restricted by GSA ADP Schedule Contract with IBM Corp.
#
###############################################################################
CUR_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )" && pwd )"
PARENT_DIR="$( cd "$( dirname "${BASH_SOURCE[0]}" )/.." && pwd )"
# Import common utilities and environment variables
source ${CUR_DIR}/helper/common.sh


function show_help() {
    echo
    echo "Usage:"
    echo
    echo " ${CUR_DIR}/cp4a-deployment.sh -m [modetype] -n <CP4BA_NAMESPACE>"
    echo " ${CUR_DIR}/cp4a-deployment.sh -n <CP4BA_NAMESPACE>"
    echo
    echo "Options:"
    echo
    echo "  -h  Display the help."
    echo
    echo "  -m  Optional: The valid mode types are:[upgradeOperator], [upgradeOperatorStatus], [upgradeDeployment] and [upgradeDeploymentStatus]."
    echo
    echo "  -n  Required: The target namespace of the CP4BA deployment. "
    echo "                If CP4BA is deployed using separate namespaces for operators and operands/services and the script is being used for upgrade, the value is the namespace where the CP4BA operators are deployed"
    echo "                If CP4BA is deployed using separate namespaces for operators and operands/services and the script is being used for generating a custom resource file, the value is the namespace where CP4BA operands/services are to be deployed."
    echo
    echo "  -i  Optional: Operator image name. By default, it is cp.icr.io/cp/cp4a/icp4a-operator:$CP4BA_RELEASE_BASE"
    echo
    echo "  -p  Optional: Pull secret to use to connect to the registry. By default, it is ibm-entitlement-key"
    echo
    echo "  --enable-private-catalog Optional: Set this flag to switch the CatalogSource from global to namespace-scoped. By default it is in openshift-marketplace namespace."
    echo
    echo "Additional Information:"
    echo
    echo "  ${YELLOW_TEXT}* To create a custom resource file for a new CP4BA deployment, follow these step:${RESET_TEXT}"
    echo "      - STEP 1: Run the script with \"-n <CP4BA_NAMESPACE>\"."
    echo "  ${YELLOW_TEXT}* Running the script to upgrade a CP4BA deployment from 24.0.1-IF002 or newer to $CP4BA_RELEASE_BASE GA/$CP4BA_RELEASE_BASE.X. You must run the modes in the following order:${RESET_TEXT}"
    echo "      - STEP 1 (Required): Run the script in [upgradeOperator] mode to upgrade CP4BA operators/migrate (Cluster-scoped -> Cluster-scoped [AllNamespaces] / Namespace-scoped -> Namespace-scoped) the IBM Cloud Pak foundational services and then shutdown all CP4BA operators before upgrade CP4BA deployment."
    echo "      - STEP 2 (Optional): Run the script in [upgradeOperatorStatus] mode to verify the upgrade of CP4BA operators and dependencies."
    echo "      - STEP 3 (Required): Run the script in [upgradeDeployment] mode to upgrade the CP4BA deployment. The script will generate the new version of the custom resource, which can be reviewed and modified offline, or applied directly without modification."
    echo "      - STEP 4 (Required): Run the script in [upgradeDeploymentStatus] mode to start the necessary CP4BA operators to upgrade the dependent service (zenService) first, then start all CP4BA operators to complete the upgrade of the CP4BA deployment, while also verifying the success of the upgrade."
    # echo "      - STEP 5 (Required): Run the script in [upgradePostconfig] mode to show the configuration after the CP4BA deployment upgrade."
    echo "  ${YELLOW_TEXT}* To upgrade a CP4BA deployment from $CP4BA_RELEASE_BASE GA/$CP4BA_RELEASE_BASE.X to $CP4BA_RELEASE_BASE.X, follow these steps in order:${RESET_TEXT}"
    echo "      - STEP 1 (Required): Run the script in [upgradeOperator] mode to upgrade the IBM Cloud Pak foundational services and CP4BA operators, then shutdown all CP4BA operators before upgrading the CP4BA deployment."
    echo "      - STEP 2 (Optional): Run the script in [upgradeOperatorStatus] mode to confirm the upgrade of CP4BA operators and dependencies."
    echo "      - STEP 3 (Required): Run the script in [upgradeDeploymentStatus] mode to start the necessary CP4BA operators to upgrade the dependent service (zenService) first, then restart all CP4BA operators to complete the upgrade, while verifying the success of the upgrade."
}

function parse_arguments() {
    while [[ "$@" != "" ]]; do
        case "$1" in
        -m)
            shift
            if [ -z $1 ]; then
                echo "Invalid option: -m flag requires an argument"
                exit 1
            fi
            RUNTIME_MODE=$1
            if [[ $RUNTIME_MODE == "upgradeOperator" || $RUNTIME_MODE == "upgradeOperatorStatus" || $RUNTIME_MODE == "upgradeDeployment" || $RUNTIME_MODE == "upgradeDeploymentStatus" ]]; then
                echo -n
            else
                echo -e "Provide a valid argument for -m: [upgradeOperator] or [upgradeOperatorStatus] or [upgradeDeployment] or [upgradeDeploymentStatus]"
                exit -1
            fi
            ;;
        -n)
            shift
            if [ -z $1 ]; then
                echo "Invalid option: -n flag requires an argument"
                exit 1
            fi
            TARGET_PROJECT_NAME=$1
            case "$TARGET_PROJECT_NAME" in
            "")
                echo -e "\x1B[1;31mEnter a valid namespace name, namespace name can not be blank\x1B[0m"
                exit -1
                ;;
            "openshift"*)
                echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'openshift' or start with 'openshift' \x1B[0m"
                exit -1
                ;;
            "kube"*)
                echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'kube' or start with 'kube' \x1B[0m"
                exit -1
                ;;
            *)
                # Check cluster login
                check_cluster_login
                # Check project name
                isProjExists=`${CLI_CMD} get project $TARGET_PROJECT_NAME --ignore-not-found | wc -l`  >/dev/null 2>&1
                if [ $isProjExists -ne 2 ] ; then
                    echo -e "\x1B[1;31mInvalid project name \"$TARGET_PROJECT_NAME\", enter a valid name...\x1B[0m"
                    exit 1
                fi
                echo -n
                ;;
            esac
            ;;
        -i)
            shift
            if [ -z $1 ]; then
                echo "Invalid option: -i flag requires an argument"
                exit 1
            fi
            IMAGEREGISTRY=$1
            ;;
        -p)
            shift
            if [ -z $1 ]; then
                echo "Invalid option: -p flag requires an argument"
                exit 1
            fi
            PULLSECRET=$1
            ;;
        -h | --help | \?)
            show_help
            exit 0
            ;;
        --enable-private-catalog)
            ENABLE_PRIVATE_CATALOG=1
            ;;
        --original-cp4ba-csv-ver)
            shift
            CP4BA_ORIGINAL_CSV_VERSION=$1
            ;;
        --cpfs-upgrade-mode)
            shift
            UPGRADE_MODE=$1
            ;;
        dev)
        SCRIPT_MODE="dev"
        ;;
        review)
        SCRIPT_MODE="review"
        ;;
        baw-dev)
        SCRIPT_MODE="baw-dev"
        ;;
        baw)
        SCRIPT_MODE="baw"
        ;;
        #This is an internal flag that allow direct upgrade from the upgrade_blocked_versions list to 24.0.1. This flag can be used to skip logic that related to skip version upgrade.
        allow_direct_upgrade)
        ALLOW_DIRECT_UPGRADE=1
        ;;
        *)
            echo "Invalid option"
            show_help
            exit 1
            ;;
        esac
        shift
    done
}

parse_arguments "$@"
# if [[ -z "$RUNTIME_MODE" ]]; then
#     echo -e "\x1B[1;31mInput value for \"-m <MODE_NAME>\" option.\n\x1B[0m"
#     exit 1
# fi
if [[ -z "$TARGET_PROJECT_NAME" ]]; then
    echo -e "\x1B[1;31mInput value for \"-n <CP4BA_NAMESPACE>\" option.\n\x1B[0m"
    exit 1
fi

# Import common utilities and environment variables
source ${CUR_DIR}/helper/common.sh $TARGET_PROJECT_NAME

# Import variables for property file
source ${CUR_DIR}/helper/cp4ba-property.sh

DOCKER_RES_SECRET_NAME="ibm-entitlement-key"
DOCKER_REG_USER=""

if [[ "$SCRIPT_MODE" == "baw-dev" || "$SCRIPT_MODE" == "dev" || "$SCRIPT_MODE" == "review" ]] # During dev, OLM uses stage image repo
then
    DOCKER_REG_SERVER="cp.stg.icr.io"
    if [[ -z $2 ]]; then
        IMAGE_TAG_DEV="${CP4BA_RELEASE_BASE}"
    else
        IMAGE_TAG_DEV=$2
    fi
    IMAGE_TAG_FINAL="${CP4BA_RELEASE_BASE}"
else
    DOCKER_REG_SERVER="cp.icr.io"
fi
DOCKER_REG_KEY=""
REGISTRY_IN_FILE="cp.icr.io"
# OPERATOR_IMAGE=${DOCKER_REG_SERVER}/cp/cp4a/icp4a-operator:21.0.2

old_db2="docker.io\/ibmcom"
old_db2_alpine="docker.io\/alpine"
old_ldap="docker.io\/osixia"
old_db2_etcd="quay.io\/coreos"
old_busybox="docker.io\/library"

TEMP_FOLDER=${CUR_DIR}/.tmp
BAK_FOLDER=${CUR_DIR}/.bak
FINAL_CR_FOLDER=${CUR_DIR}/generated-cr/project/$TARGET_PROJECT_NAME

DEPLOY_TYPE_IN_FILE_NAME="" # Default value is empty
OPERATOR_FILE=${PARENT_DIR}/descriptors/operator.yaml
OPERATOR_FILE_TMP=$TEMP_FOLDER/.operator_tmp.yaml
OPERATOR_FILE_BAK=$BAK_FOLDER/.operator.yaml

# PREREQUISITES_FOLDER=${CUR_DIR}/cp4ba-prerequisites
# PROPERTY_FILE_FOLDER=${PREREQUISITES_FOLDER}/propertyfile
# TEMPORARY_PROPERTY_FILE=${TEMP_FOLDER}/.TEMPORARY.property
# LDAP_PROPERTY_FILE=${PROPERTY_FILE_FOLDER}/cp4ba_LDAP.property
# EXTERNAL_LDAP_PROPERTY_FILE=${PROPERTY_FILE_FOLDER}/cp4ba_External_LDAP.property

# DB_NAME_USER_PROPERTY_FILE=${PROPERTY_FILE_FOLDER}/cp4ba_db_name_user.property
# DB_SERVER_INFO_PROPERTY_FILE=${PROPERTY_FILE_FOLDER}/cp4ba_db_server.property


# OPERATOR_PVC_FILE=${PARENT_DIR}/descriptors/operator-shared-pvc.yaml
# OPERATOR_PVC_FILE_TMP1=$TEMP_FOLDER/.operator-shared-pvc_tmp1.yaml
# OPERATOR_PVC_FILE_TMP=$TEMP_FOLDER/.operator-shared-pvc_tmp.yaml
# OPERATOR_PVC_FILE_BAK=$BAK_FOLDER/.operator-shared-pvc.yaml


CP4A_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_final_tmp.yaml
CP4A_PATTERN_FILE_BAK=$FINAL_CR_FOLDER/ibm_cp4a_cr_final.yaml
FNCM_SEPARATE_PATTERN_FILE_BAK=$FINAL_CR_FOLDER/ibm_content_cr_final.yaml
CP4A_EXISTING_BAK=$TEMP_FOLDER/.ibm_cp4a_cr_final_existing_bak.yaml
CP4A_EXISTING_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_final_existing_tmp.yaml

JDBC_DRIVER_DIR=${CUR_DIR}/jdbc
SAP_LIB_DIR=${CUR_DIR}/saplibs
ACA_MODEL_FILES_DIR=../ACA/configuration-ha/
PLATFORM_SELECTED=""
PATTERN_SELECTED=""
COMPONENTS_SELECTED=""
OPT_COMPONENTS_CR_SELECTED=""
OPT_COMPONENTS_SELECTED=()
LDAP_TYPE=""
# TARGET_PROJECT_NAME=""

FOUNDATION_CR_SELECTED=""
optional_component_arr=()
optional_component_cr_arr=()
foundation_component_arr=()
FOUNDATION_FULL_ARR=("BAN" "RR" "BAS" "UMS" "AE")
OPTIONAL_COMPONENT_FULL_ARR=("content_integration" "workstreams" "case" "ban" "bai" "css" "cmis" "es" "ier" "iccsap" "tm" "ums" "ads_designer" "ads_runtime" "app_designer" "decisionCenter" "decisionServerRuntime" "decisionRunner" "ae_data_persistence" "baw_authoring" "pfs" "baml" "auto_service" "document_processing_runtime" "document_processing_designer" "wfps_authoring" "kafka" "opensearch")

## VARIABLES TO CHECK IF CPE IS
## CONFIGURED WITH SCIM
CPE_SERVICE_NAME="cpe-stateless-svc"
CPE_SERVICE_PORT="9443"
RUNNABLE_JAR_NAME="cp4ba-scim-upgrade.jar"
CLASS_PATH="/tmp/Jace.jar;/opt/ibm/content_emitter/stax-api.jar;/opt/ibm/content_emitter/xlxpScanner.jar;/opt/ibm/content_emitter/xlxpScannerUtils.jar"
## PODS LABELS
CONTENT_OPERATOR_LABEL="name=ibm-content-operator"
CP4A_OPERATOR_LABEL="name=ibm-cp4a-operator"
CPE_LABEL="cpe-deploy"
## Jar files section
JACE_NAME="Jace.jar"
STAX_NAME="stax-api.jar"
SCANNER_NAME="xlxpScanner.jar"
SCANNER_UTILS_NAME="xlxpScannerUtils.jar"
## Truststore names
TRUSTSTORE_NAME="ibm_customFNCMTrustStore.p12"
POST_UPGRADE_TRUSTSTORE_NAME="trusts.p12"
## paths to files
JACE_PATH="/opt/ibm/wlp/usr/servers/defaultServer/jaceLib/${JACE_NAME}"
STAX_PATH="/opt/ibm/content_emitter/${STAX_NAME}"
SCANNER_PATH="/opt/ibm/content_emitter/${SCANNER_NAME}"
SCANNER_UTILS_PATH="/opt/ibm/content_emitter/${SCANNER_UTILS_NAME}"
CPE_TRUSTSTORE_PATH="/opt/ibm/wlp/usr/servers/defaultServer/resources/security/${TRUSTSTORE_NAME}"
POST_UPGRADE_CPE_TRUSTSTORE_PATH="/shared/tls/truststore/pkcs12/${POST_UPGRADE_TRUSTSTORE_NAME}"

function prompt_license(){
    clear

    get_baw_mode
    retVal_baw=$?
    if [[ $retVal_baw -eq 1 ]]; then
        echo -e "\x1B[1;31mIMPORTANT: Review the IBM Cloud Pak for Business Automation license information here: \n\x1B[0m"
        echo -e "\x1B[1;31mhttps://www.ibm.com/support/customer/csol/terms/?id=L-LDYZ-7V4YJ4&lc=en\n\x1B[0m"
        echo -e "\x1B[1mIf you are selecting BAW capabilities, please refer to the additional license agreement here: \n\x1B[0m"
        echo -e "\x1B[1;31mhttps://www.ibm.com/support/customer/csol/terms/?id=L-FWZS-PUAT9S\n\x1B[0m"
        INSTALL_BAW_ONLY="No"
    fi

    prompt_press_any_key_to_continue

    printf "\n"
    while true; do
        if [[ $retVal_baw -eq 1 ]]; then
            printf "\x1B[1mDo you accept the IBM Cloud Pak for Business Automation license (Yes/No, default: No): \x1B[0m"
        fi
        if  [[ $CP4BA_LICENSE_ACCEPT == "Accept" || $CP4BA_LICENSE_ACCEPT == "accept" || $CP4BA_LICENSE_ACCEPT == "ACCEPT"   ]]; then
            ans='Yes'
            IBM_LICENS='Accept'
        else
            read -rp "" ans
        fi
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES")
                printf "\n"
                while true; do
                    if [[ $retVal_baw -eq 0 ]]; then
                        printf "\n"
                    fi
                    if [[ $retVal_baw -eq 1 ]]; then
                        printf "\x1B[1mDid you deploy Content CR (CRD: contents.icp4a.ibm.com) in current cluster? (Yes/No, default: No): \x1B[0m"
                    fi
                    if  [[ $CONTENT_CR_EXISTS == "" ]]; then
                        read -rp "" ans
                    else
                        ans=$CONTENT_CR_EXISTS
                    fi
                    case "$ans" in
                    "y"|"Y"|"yes"|"Yes"|"YES")
                        echo -e "Continuing...\n"
                        # echo -e "\x1B[1;31mThe cp4a-deployment.sh can not work with existing Content CR together, exiting now...\x1B[0m\n"
                        CONTENT_DEPLOYED="Yes"
                        break
                        ;;
                    "n"|"N"|"no"|"No"|"NO"|"")
                        echo -e "Continuing...\n"
                        CONTENT_DEPLOYED="No"
                        break
                        ;;
                    *)
                        echo -e "Answer must be \"Yes\" or \"No\"\n"
                        ;;
                    esac
                done
            echo -e "Starting to Install the Cloud Pak for Business Automation Operator...\n"
            IBM_LICENS="Accept"
            validate_cli
            break
            ;;
        "n"|"N"|"no"|"No"|"NO"|"")
            echo -e "The license agreement was not accepted. The license agreement must be accepted to continue. The script is exiting...\n"
            exit 0
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}

function create_configmap_os_migration(){
    # info "Creating ibm-cp4ba-os-migration-status configMap for this CP4BA deployment in the project \"$CP4BA_SERVICES_NS\"."

    mkdir -p ${TEMP_FOLDER} >/dev/null 2>&1
cat << EOF > ${TEMP_FOLDER}/ibm-cp4ba-os-migration-status-configmap.yaml
# YAML template for ibm-cp4ba-os-migration-status
---
kind: ConfigMap
apiVersion: v1
metadata:
  name: ibm-cp4ba-os-migration-status
  namespace: "$CP4BA_SERVICES_NS"
data:
  opensearch_migration_done: "$MIGRATE_ES_TO_OS_DONE"
EOF
    ${CLI_CMD} delete -f ${TEMP_FOLDER}/ibm-cp4ba-os-migration-status-configmap.yaml >/dev/null 2>&1
    ${CLI_CMD} apply -f ${TEMP_FOLDER}/ibm-cp4ba-os-migration-status-configmap.yaml >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        # success "Created ibm-cp4ba-os-migration-status configMap for this CP4BA deployment in the project \"$CP4BA_SERVICES_NS\"."
        sleep 1
    else
        warning "Failed to create ibm-cp4ba-os-migration-status configMap for this CP4BA deployment in the project \"$CP4BA_SERVICES_NS\"!"
        exit 1
    fi
}

function show_tips_es_to_os_migration(){
    which jq &>/dev/null
    [[ $? -ne 0 ]] && \
    echo -e  "\x1B[1;31mUnable to locate the jq CLI. You must install it to run this migration script.\x1B[0m"

    # For step1
    ELASTICSEARCH_CR_NAME=$(${CLI_CMD} get elasticsearch.elastic.automation.ibm.com -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o name)
    if [[ ! -z $ELASTICSEARCH_CR_NAME  ]]; then
        ELASTICSEARCH_CR_NAME=$(echo $ELASTICSEARCH_CR_NAME | cut -d'/' -f2)
        ELASTICSEARCH_URL=$(${CLI_CMD} get route ${ELASTICSEARCH_CR_NAME}-es -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o jsonpath='{.spec.host}')
        ELASTIC_PASSWORD=$(${CLI_CMD} get secret ${ELASTICSEARCH_CR_NAME}-elasticsearch-es-default-user -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o jsonpath='{.data.password}' | base64 -d)
        OPENSEARCH_URL=$(${CLI_CMD} get route opensearch-route -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o jsonpath='{.spec.host}')
        OPENSEARCH_PASSWORD=$(${CLI_CMD} get secret opensearch-ibm-elasticsearch-cred-secret -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o jsonpath='{.data.elastic}' | base64 -d)
    else
        fail "Elasticsearch custom resource not found in the project \"$CP4BA_SERVICES_NS\"."
    fi
    printf "\n"
    echo -e "\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31mYou need to complete migration from Elasticsearch to Opensearch first before upgrade CP4BA.\n\x1B[0m"
    echo "${YELLOW_TEXT}* How to migrate from Elasticsearch to Opensearch${RESET_TEXT}"
    # For step1
    printf "\n"
    echo "  ${YELLOW_TEXT}- STEP 1 (Optional)${RESET_TEXT}: Run pre-migration"
    echo "    For high volume data, you can reduce Business Automation Insights downtime by running pre-migration. This step must be executed following the instructions documented in the Knowledge Center: \"Premigrating Business Automation Insights data to OpenSearch\" topic:"
    echo "    - if upgrading from 21.0.3 or 22.0.2: [https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/24.0.0?topic=data-premigrating-business-automation-insights-opensearch]"
    echo "    - if upgrading from 23.0.2: [https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/24.0.0?topic=iomed-premigrating-business-automation-insights-data-opensearch]"

    echo "    ${YELLOW_TEXT}GOAL${RESET_TEXT}: Migrate immutable indices first to reduce downtime. You can bypass this step if you prefer migrating all data at once during the downtime."
    echo "    ${YELLOW_TEXT}* Before executing the migration script, run the prerequisites to export variables in your terminal${RESET_TEXT}"
    echo "      Execute the following commands:"
    echo -e '\033[0;32m      # export ELASTICSEARCH_URL=https://'"${ELASTICSEARCH_URL}"':443\033[0m'
    echo -e '\033[0;32m      # export ELASTIC_USERNAME="elasticsearch-admin"\033[0m'
    echo -e '\033[0;32m      # export ELASTIC_PASSWORD=$('"${CLI_CMD}"' get secret iaf-system-elasticsearch-es-default-user -n '"$CP4BA_SERVICES_NS"' --no-headers --ignore-not-found -o jsonpath='{.data.password}' | base64 -d)\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_URL=https://'"${OPENSEARCH_URL}"'\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_USERNAME="elastic"\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_PASSWORD=$('"${CLI_CMD}"' get secret opensearch-ibm-elasticsearch-cred-secret -n '"$CP4BA_SERVICES_NS"' --no-headers --ignore-not-found -o jsonpath='{.data.elastic}' | base64 -d)\033[0m'
    echo "    ${YELLOW_TEXT}* Migration script Usage:${RESET_TEXT}"
    echo "      For more information, Refer to Knowledge Center: \"Installing OpenSearch and migrating Elasticsearch data\" topic:"
    echo "      - if upgrading from 21.0.3 or 22.0.2: [https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/24.0.0?topic=upgrade-installing-opensearch-migrating-elasticsearch-data]"
    echo "      - if upgrading from 23.0.2: [https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/24.0.0?topic=2302-installing-opensearch-migrating-elasticsearch-data]"
    echo "      Execute the following commands:"
    echo "      ${GREEN_TEXT}# $OPENSEARCH_MIGRATION_SCRIPT${RESET_TEXT} [-dryrun] [-doc_count] [-include=<comma separated indices>] [-exclude=<comma separated indices>] [-include_regex=<comma separated regex patterns>] [-exclude_regex=<comma separated regex patterns>] [-startdate=<start date>] [-enddate=<end date>] [-timestamp_key=<date field key>] [-delete] [logfile] [--help]"
    echo "      Options:"
    echo "        -dryrun                              List of indices of elastic and displaying dry run steps"
    echo "        -doc_count                           List all indices of elastic and opensearch with document count, and exit."
    echo "        -include=<comma separated indices>   List of indices to include"
    echo "        -exclude=<comma separated indices>   List of indices to exclude"
    echo "        -include_regex=<regex pattern>       List of regex pattern to include indices"
    echo "        -exclude_regex=<regex pattern>       List of regex pattern to exclude indices"
    echo "        -startdate=<start date>              Start date for data migration (format: 'YYYY-MM-DDTHH:MM:SS')"
    echo "        -enddate=<end date>                  End date for data migration (format: 'YYYY-MM-DDTHH:MM:SS')"
    echo "        -timestamp_key=<key>                 Key for date values (default: 'timestamp')"
    echo "        -delete                              delete Opensearch indices"
    echo "        logfile                              Optional: Log file to save migration summary"
    echo "        --help                               Display usage details"
    echo "    ${YELLOW_TEXT}* Sample execution commands:${RESET_TEXT}"
    echo "      - To migrate only included indices data which is speicified in include option"
    echo "        Execute the following command:"
    echo "        ${GREEN_TEXT}# $OPENSEARCH_MIGRATION_SCRIPT -include=iaf_elastic_index_mapping,test_iaf_elastic_index_mapping${RESET_TEXT}"
    echo "      - To migrate all indices based on the date range"
    echo "        Execute the following command:"
    echo "        ${GREEN_TEXT}# $OPENSEARCH_MIGRATION_SCRIPT -startdate=\"2023-01-28T12:00:00\" -enddate=\"2024-02-07T00:00:00\"${RESET_TEXT}"
    echo "      - To migrate all PFS indices"
    echo "        Execute the following command:"
    echo "        ${GREEN_TEXT}# $OPENSEARCH_MIGRATION_SCRIPT -include_regex=ibm-pfs@*${RESET_TEXT}"
    echo "      - To migrate PFS index one by one"
    echo "        Execute the following command:"
    echo "        ${GREEN_TEXT}# $OPENSEARCH_MIGRATION_SCRIPT -include=icp4ba-pfs@94c01f27-716f-4289-92b9-f596b540cdc6${RESET_TEXT}"

    # For step2
    step_num=2
    if [[ ( $CONTENT_CR_EXIST="Yes" || " ${EXISTING_PATTERN_ARR[@]}" =~ "workflow-runtime" || " ${EXISTING_PATTERN_ARR[@]}" =~ "workflow-authoring" || " ${EXISTING_PATTERN_ARR[@]}" =~ "workstreams" || " ${EXISTING_PATTERN_ARR[@]}" =~ "content" || " ${EXISTING_PATTERN_ARR[@]}" =~ "document_processing" || "${EXISTING_OPT_COMPONENT_ARR[@]}" =~ "ae_data_persistence") && (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "bai" || $bai_flag == "true") ]]; then
        printf "\n"
        echo "  ${YELLOW_TEXT}- STEP ${step_num} (Required)${RESET_TEXT}: Stop the Content Emitter if the Content capabilty is installed in this CP4BA deployment, and if the \"oc_cpe_obj_store_enable_content_event_emitter\" is set to \"true\" in the initialize_configuration section of the custom resource for any object store."
        echo "    ${YELLOW_TEXT}* Before upgrading the Cloud Pak, ensure to disable the Content Event Emitter if it is configured on an object store for Content Platform Engine.${RESET_TEXT}"
        echo "      1. Log in to the Administration Console for Content Platform Engine."
        echo "      2. Navigate to Object Stores > object store name > Events, Actions, Processes > Subscriptions."
        echo "      3. Select ContentEventEmitterSubscription or the name of the existing subscription used by the Content Event Emitter."
        echo "      4. Go to the Properties tab."
        echo "      5. In the row labeled 'Is Enabled,' click the Property Value dropdown and select 'False.'"
        echo "      6. Click Save."
        step_num=$((step_num + 1))
    fi

    # For step3
    if [[ (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "bai") || $bai_flag == "true" ]]; then
        printf "\n"
        echo "  ${YELLOW_TEXT}- STEP ${step_num} (Required)${RESET_TEXT}: Create Flink savepoints and stop Flink jobs.${RED_TEXT} (NOTE: This is a one-time operation. The command will return nothing if run again.)${RESET_TEXT}"
        step_num=$((step_num + 1))
        ${CLI_CMD} get crd |grep insightsengines.icp4a.ibm.com >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            INSIGHTS_ENGINE_CR=$(${CLI_CMD} get insightsengines.icp4a.ibm.com --no-headers --ignore-not-found -n ${TARGET_PROJECT_NAME} -o name)
        fi
        if [[ -z $INSIGHTS_ENGINE_CR ]]; then
            INSIGHTS_ENGINE_CR=$(${CLI_CMD} get insightsengines.insightsengine.automation.ibm.com --no-headers --ignore-not-found -n ${TARGET_PROJECT_NAME} -o name)
            if [[ -z $INSIGHTS_ENGINE_CR ]]; then
                error "insightsengines custom resource instance not found in the project \"${TARGET_PROJECT_NAME}\"."
            fi
            # exit 1
        fi
        if [[ ! -z $INSIGHTS_ENGINE_CR ]]; then
            MANAGEMENT_URL=$(${CLI_CMD} get ${INSIGHTS_ENGINE_CR} --no-headers --ignore-not-found -n ${TARGET_PROJECT_NAME} -o jsonpath='{.status.components.management.endpoints[?(@.scope=="External")].uri}')
            MANAGEMENT_AUTH_SECRET=$(${CLI_CMD} get ${INSIGHTS_ENGINE_CR} --no-headers --ignore-not-found -n ${TARGET_PROJECT_NAME} -o jsonpath='{.status.components.management.endpoints[?(@.scope=="External")].authentication.secret.secretName}')
            MANAGEMENT_USERNAME=$(${CLI_CMD} get secret ${MANAGEMENT_AUTH_SECRET} --no-headers --ignore-not-found -n ${TARGET_PROJECT_NAME} -o jsonpath='{.data.username}' | base64 -d)
            MANAGEMENT_PASSWORD=$(${CLI_CMD} get secret ${MANAGEMENT_AUTH_SECRET} --no-headers --ignore-not-found -n ${TARGET_PROJECT_NAME} -o jsonpath='{.data.password}' | base64 -d)
            if [[ -z "$MANAGEMENT_URL" || -z "$MANAGEMENT_AUTH_SECRET" || -z "$MANAGEMENT_USERNAME" || -z "$MANAGEMENT_PASSWORD" ]]; then
                error "Can not create the BAI savepoints for recovery path."
                # exit 1
            else
                if [[ -e ${UPGRADE_DEPLOYMENT_CR}/bai.json ]]; then
                    [ "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" != "[]" ] && mkdir -p ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup && cp ${UPGRADE_DEPLOYMENT_CR}/bai.json ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup/bai_$(date +'%Y%m%d%H%M%S').json
                fi
                # touch ${UPGRADE_DEPLOYMENT_BAI_TMP} >/dev/null 2>&1
                echo "    ${YELLOW_TEXT}* Create a Flink savepoint and stop the Flink job simultaneously.${RESET_TEXT}"
                echo "      Execute the following command:"
                # echo "      ${GREEN_TEXT}# curl -X POST -k -u ${MANAGEMENT_USERNAME}:${MANAGEMENT_PASSWORD} \"${MANAGEMENT_URL}/api/v1/processing/jobs/savepoints?cancel-job=true\" -o ${UPGRADE_DEPLOYMENT_CR}/bai.json >/dev/null 2>&1 && cat ${UPGRADE_DEPLOYMENT_CR}/bai.json && echo ${RESET_TEXT}"
                echo -e '\033[0;32m      '"${GREEN_TEXT}"'# curl -X POST -k -u '"${MANAGEMENT_USERNAME}"':'"${MANAGEMENT_PASSWORD}"' "'"${MANAGEMENT_URL}"'/api/v1/processing/jobs/savepoints?cancel-job=true" -o /tmp/tmp_bai.json && [ "$(cat /tmp/tmp_bai.json)" != "[]" ] && '"${COPY_CMD}"' -rf /tmp/tmp_bai.json '"${UPGRADE_DEPLOYMENT_CR}"'/bai.json || rm -rf /tmp/tmp_bai.json\033[0m'
            fi
        fi

        bpc_deployment_name=$(${CLI_CMD} get deployment --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS  -o name | grep insights-engine-cockpit)
        if [[ ! -z $bpc_deployment_name ]]; then
            printf "\n"
            echo "  ${YELLOW_TEXT}- STEP ${step_num} (Required)${RESET_TEXT}: Scale down the Business Performance Center (BPC) to prevent the generation of dirty data during migration${RESET_TEXT}"
            echo "    Execute the following commands:"
            if [[ $ALL_NAMESPACE_FLAG == "No" ]]; then
                bai_operator_name=$(${CLI_CMD} get deployment --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS  -o name | grep insights-engine-operator-controller-manager)
                if [[ ! -z $bai_operator_name ]]; then
                    echo -e '\033[0;32m    '"${GREEN_TEXT}"'# '"${CLI_CMD}"' scale --replicas=0 '"${bai_operator_name}"' -n '"${TARGET_PROJECT_NAME}"'\033[0m'
                fi
                bai_operator_name=$(${CLI_CMD} get deployment --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS  -o name | grep ibm-insights-engine-operator)
                if [[ ! -z $bai_operator_name ]]; then
                    echo -e '\033[0;32m    '"${GREEN_TEXT}"'# '"${CLI_CMD}"' scale --replicas=0 '"${bai_operator_name}"' -n '"${TARGET_PROJECT_NAME}"'\033[0m'
                fi
            elif [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
                bai_operator_name=$(${CLI_CMD} get deployment --no-headers --ignore-not-found -n openshift-operators  -o name | grep insights-engine-operator-controller-manager)
                if [[ ! -z $bai_operator_name ]]; then
                    echo -e '\033[0;32m    '"${GREEN_TEXT}"'# '"${CLI_CMD}"' scale --replicas=0 '"${bai_operator_name}"' -n openshift-operators\033[0m'
                fi
                bai_operator_name=$(${CLI_CMD} get deployment --no-headers --ignore-not-found -n openshift-operators  -o name | grep ibm-insights-engine-operator)
                if [[ ! -z $bai_operator_name ]]; then
                    echo -e '\033[0;32m    '"${GREEN_TEXT}"'# '"${CLI_CMD}"' scale --replicas=0 '"${bai_operator_name}"' -n openshift-operators\033[0m'
                fi
            fi

            echo -e '\033[0;32m    '"${GREEN_TEXT}"'# '"${CLI_CMD}"' scale --replicas=0 '"${bpc_deployment_name}"' -n '"${TARGET_PROJECT_NAME}"'\033[0m'
            step_num=$((step_num + 1))
        fi
    fi

    # For step4
    printf "\n"
    echo "  ${YELLOW_TEXT}- STEP ${step_num} (Required)${RESET_TEXT}: Run migration script ${RED_TEXT}(WARNING: If pre-migration [STEP 1] has been run, STEP ${step_num} must be executed following instructions in the Knowledge Center: \"Installing OpenSearch and migrating Elasticsearch data\" topic:)"
    echo "    - If upgrading from 21.0.3 or 22.0.2: [https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/24.0.0?topic=upgrade-installing-opensearch-migrating-elasticsearch-data]"
    echo "    - If upgrading from 23.0.2: [https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/24.0.0?topic=2302-installing-opensearch-migrating-elasticsearch-data]${RESET_TEXT}"
    step_num=$((step_num + 1))
    echo "    ${YELLOW_TEXT}GOAL${RESET_TEXT}: To migrate all data from Elasticsearch to OpenSearch as soon as possible during downtime before upgrading the CP4BA deployment."
    echo "    ${YELLOW_TEXT}* Before executing the migration script, run the prerequisites to export variables in your terminal${RESET_TEXT}"
    echo "      Execute the following commands:"
    echo -e '\033[0;32m      # export ELASTICSEARCH_URL=https://'"${ELASTICSEARCH_URL}"':443\033[0m'
    echo -e '\033[0;32m      # export ELASTIC_USERNAME="elasticsearch-admin"\033[0m'
    echo -e '\033[0;32m      # export ELASTIC_PASSWORD=$('"${CLI_CMD}"' get secret iaf-system-elasticsearch-es-default-user -n '"$CP4BA_SERVICES_NS"' --no-headers --ignore-not-found -o jsonpath='{.data.password}' | base64 -d)\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_URL=https://'"${OPENSEARCH_URL}"'\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_USERNAME="elastic"\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_PASSWORD=$('"${CLI_CMD}"' get secret opensearch-ibm-elasticsearch-cred-secret -n '"$CP4BA_SERVICES_NS"' --no-headers --ignore-not-found -o jsonpath='{.data.elastic}' | base64 -d)\033[0m'
    echo "    ${YELLOW_TEXT}* Run the following command to migrate all the indexes, except for the indexes related to the Process Federation Server (if any) which must not be migrated with the migration script.${RESET_TEXT}"
    echo "      Execute the following command:"
    echo "      ${GREEN_TEXT}# $OPENSEARCH_MIGRATION_SCRIPT -exclude_regex=icp4ba-pfs@*${RESET_TEXT}"
    echo "    ${YELLOW_TEXT}* If your Cloud Pak for Business Automation includes IBM Process Federation Server, or if your Cloud Pak for Business Automation includes at least one Business Automation Workflow or Workflow Process Service instance with full-text search enabled, run the following command to migrate IBM Process Federation Server saved search indexes from Elasticsearch to OpenSearch.${RESET_TEXT}"
    echo "      Execute the following command:"
    echo "      ${GREEN_TEXT}# $OPENSEARCH_MIGRATION_SCRIPT -include_regex=ibmpfssavedsearches*${RESET_TEXT}"

    # For step5
    printf "\n"
    echo "  ${YELLOW_TEXT}- STEP ${step_num} (Required)${RESET_TEXT}: Verify all data migration done."
    step_num=$((step_num + 1))
    echo "    ${YELLOW_TEXT}* Before execute verification, run the prerequesties to export varaiables in your terminal${RESET_TEXT}"
    echo "      Execute the following commands:"
    echo -e '\033[0;32m      # export ELASTICSEARCH_URL=https://'"${ELASTICSEARCH_URL}"':443\033[0m'
    echo -e '\033[0;32m      # export ELASTIC_USERNAME="elasticsearch-admin"\033[0m'
    echo -e '\033[0;32m      # export ELASTIC_PASSWORD=$('"${CLI_CMD}"' get secret iaf-system-elasticsearch-es-default-user -n '"$CP4BA_SERVICES_NS"' --no-headers --ignore-not-found -o jsonpath='{.data.password}' | base64 -d)\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_URL=https://'"${OPENSEARCH_URL}"'\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_USERNAME="elastic"\033[0m'
    echo -e '\033[0;32m      # export OPENSEARCH_PASSWORD=$('"${CLI_CMD}"' get secret opensearch-ibm-elasticsearch-cred-secret -n '"$CP4BA_SERVICES_NS"' --no-headers --ignore-not-found -o jsonpath='{.data.elastic}' | base64 -d)\033[0m'
    echo "    ${YELLOW_TEXT}* To list the indices in Elasticsearch${RESET_TEXT}"
    echo "      Execute the following command:"
    echo -e '\033[0;32m      # curl -X GET -u ${ELASTIC_USERNAME}:${ELASTIC_PASSWORD} --insecure "${ELASTICSEARCH_URL}/_cat/indices?v&s=docs.count:desc,index"\033[0m'
    echo "    ${YELLOW_TEXT}* To list the indices in OpenSearch${RESET_TEXT}"
    echo "      Execute the following command:"
    echo -e '\033[0;32m      # curl -X GET -u ${OPENSEARCH_USERNAME}:${OPENSEARCH_PASSWORD} --insecure "${OPENSEARCH_URL}/_cat/indices?v&s=docs.count:desc,index"\033[0m'
    echo "    ${YELLOW_TEXT}* To compare above two outputs and check whether any missing indice between Elasticsearch and OpenSearch.${RESET_TEXT}"
    # For step6
    printf "\n"
    echo "  ${YELLOW_TEXT}- STEP ${step_num} (Required)${RESET_TEXT}: Upgrade CP4BA operators."
    echo "    Execute the following command:"
    echo "    ${GREEN_TEXT}# ${CUR_DIR}/cp4a-deployment.sh -m upgradeOperator -n $TARGET_PROJECT_NAME${RESET_TEXT}"
}

function select_private_catalog_cp4ba(){
    printf "\n"
    echo "${YELLOW_TEXT}[NOTES] You can choose to deploy CP4BA as a private catalog (namespace scope) or retain the global catalog namespace (GCN). The private catalog (recommended) uses the same target namespace as the CP4BA deployment, while the GCN uses the openshift-marketplace namespace.${RESET_TEXT}"

    while true; do
        printf "\x1B[1mDo you want to switch CP4BA deployment using private catalog? (Yes/No, default: Yes): \x1B[0m"
        read -rp "" ans
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES"|"")
            if [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
                info "The CP4BA operator is deployed in the "All" namespace, so the private catalog (namespace-scoped) cannot be used. You must continue using the global catalog namespace (GCN)."
                ENABLE_PRIVATE_CATALOG=0
            else
                ENABLE_PRIVATE_CATALOG=1
            fi
            break
            ;;
        "n"|"N"|"no"|"No"|"NO")
            ENABLE_PRIVATE_CATALOG=0
            break
            ;;
        *)
            PRIVATE_CATALOG=""
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}

function select_private_catalog_opensearch(){
    printf "\n"
    echo "${YELLOW_TEXT}[NOTES] You can install OpenSearch as either a private catalog (namespace-scoped) or in the global catalog namespace (GCN). The private option uses the same target namespace as the CP4BA deployment, while the GCN uses the openshift-marketplace namespace.${RESET_TEXT}"
    while true; do
        printf "\x1B[1mDo you want to deploy Opensearch using private catalog?\x1B[0m (Yes/No, default: Yes): "
        read -rp "" ans

        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES"|"")
            if [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
                info "The CP4BA operator is deployed in the "All" namespace, so the private catalog (namespace-scoped) cannot be used. You must continue using the global catalog namespace (GCN)."
                ENABLE_PRIVATE_CATALOG=0
                sleep 3
            else
                ENABLE_PRIVATE_CATALOG=1
            fi
            break
            ;;
        "n"|"N"|"no"|"No"|"NO")
            ENABLE_PRIVATE_CATALOG=0
            break
            ;;
        *)
            PRIVATE_CATALOG=""
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}

function setup_opensearch_catalog(){
# Create catalog source for Opensearch
    info "Creating ibm-cs-opensearch-catalog CatalogSource for Opensearch in the project \"$OPENSEARCH_CATALOG_NS\"."
    OLM_CATALOG=${PARENT_DIR}/descriptors/op-olm/catalog_source.yaml
    local LINE_NUM=$(grep -E '^  image: icr\.io/cpopen/opencontent-elasticsearch-operator-catalog@sha256:[0-9a-f]+' $OLM_CATALOG)
    local CURRENT_DIGEST=$(echo $LINE_NUM | sed -E 's/^.*@sha256:([0-9a-f]+).*$/\1/')

cat << EOF > ${TEMP_FOLDER}/ibm-cp4ba-os-catalog-source.yaml
# IBM CS Opensearch Operator Catalog 1.1.2131
apiVersion: operators.coreos.com/v1alpha1
kind: CatalogSource
metadata:
  name: ibm-cs-opensearch-catalog
  namespace: "${OPENSEARCH_CATALOG_NS}"
spec:
  displayName: IBM CS Opencontent Opensearch Catalog
  publisher: IBM
  sourceType: grpc
  image: icr.io/cpopen/opencontent-elasticsearch-operator-catalog@sha256:${CURRENT_DIGEST}
  updateStrategy:
    registryPoll:
      interval: 45m
  priority: 100
EOF
    # ${CLI_CMD} delete -f ${TEMP_FOLDER}/ibm-cp4ba-os-catalog-source.yaml >/dev/null 2>&1
    ${CLI_CMD} apply -f ${TEMP_FOLDER}/ibm-cp4ba-os-catalog-source.yaml >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        success "Created ibm-cs-opensearch-catalog CatalogSource for Opensearch in the project \"$OPENSEARCH_CATALOG_NS\"."
        sleep 3
    else
        warning "Failed to create ibm-cs-opensearch-catalog CatalogSource for Opensearch in the project \"$OPENSEARCH_CATALOG_NS\"!"
        exit 1
    fi

    info "Checking Opensearch operator catalog pod ready or not in the project \"$OPENSEARCH_CATALOG_NS\""
    maxRetry=10
    for ((retry=0;retry<=${maxRetry};retry++)); do
        opensearch_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=ibm-cs-opensearch-catalog -n $OPENSEARCH_CATALOG_NS -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')

        if [[  -z $opensearch_catalog_pod_name ]]; then
            if [[ $retry -eq ${maxRetry} ]]; then
                printf "\n"
                if [[ -z $opensearch_catalog_pod_name ]]; then
                    warning "Timeout waiting for ibm-cs-opensearch-catalog catalog pod to be ready in the project \"$OPENSEARCH_CATALOG_NS\""
                 fi
                exit 1
            else
                sleep 30
                echo -n "..."
                continue
            fi
        else
            success "Opensearch operator catalog pod to be ready in the project \"$OPENSEARCH_CATALOG_NS\"!"
            break
        fi
    done
}

function setup_opensearch_subscription(){
    local operator_project_name=$1
    operator_project_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$operator_project_name")
# Create Opensearch subscription
    info "Creating ibm-elasticsearch-operator Subscription for Opensearch in the project \"$operator_project_name\"."

cat << EOF > ${TEMP_FOLDER}/ibm-cp4ba-os-subscription.yaml
apiVersion: operators.coreos.com/v1alpha1
kind: Subscription
metadata:
  annotations:
    $operator_project_name.common-service/config: "true"
    $operator_project_name.common-service/registry: "true"
    $operator_project_name.opensearch-request.ibm-elasticsearch-operator/operatorNamespace: $operator_project_name
    $operator_project_name.opensearch-request.ibm-elasticsearch-operator/request: v1.1
  labels:
    operator.ibm.com/opreq-control: "true"
    operators.coreos.com/ibm-elasticsearch-operator.$operator_project_name: ""
  name: ibm-elasticsearch-operator
  namespace: "$operator_project_name"
spec:
  channel: v1.1
  installPlanApproval: Automatic
  name: ibm-elasticsearch-operator
  source: ibm-cs-opensearch-catalog
  sourceNamespace: "$OPENSEARCH_CATALOG_NS"
EOF
    # ${CLI_CMD} delete -f ${TEMP_FOLDER}/ibm-cp4ba-os-subscription.yaml >/dev/null 2>&1
    ${CLI_CMD} apply -f ${TEMP_FOLDER}/ibm-cp4ba-os-subscription.yaml >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        success "Created ibm-elasticsearch-operator Subscription for Opensearch in the project \"$operator_project_name\"."
        sleep 3
    else
        warning "Failed to create ibm-elasticsearch-operator Subscription for Opensearch in the project \"$operator_project_name\"!"
        exit 1
    fi


    info "Checking Opensearch operator pod ready or not in the project \"$operator_project_name\""
    maxRetry=50
    for ((retry=0;retry<=${maxRetry};retry++)); do
        opensearch_pod_name=$(${CLI_CMD} get pod -l=name=ibm-cloudpakopen-elasticsearch-operator -n $operator_project_name -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')

        if [[  -z $opensearch_pod_name ]]; then
            if [[ $retry -eq ${maxRetry} ]]; then
                printf "\n"
                if [[ -z $opensearch_pod_name ]]; then
                    warning "Timeout waiting for Opensearch operator pod to be ready in the project \"$operator_project_name\""
                 fi
                exit 1
            else
                sleep 30
                echo -n "..."
                continue
            fi
        else
            success "Opensearch operator pod to be ready in the project \"$operator_project_name\"!"
            msg "Pod: $opensearch_pod_name"
            break
        fi
    done
}

function setup_opensearch_issuer(){
# Create Issuer for Opensearch
    info "Creating opensearch-tls-issuer Issuer for Opensearch in the project \"$CP4BA_SERVICES_NS\"."
    if [[ -z $cp4ba_root_ca_secret_name || $cp4ba_root_ca_secret_name == *"meta.name"* ]]; then
        cp4ba_root_ca_secret_name="${cp4ba_cr_metaname}-root-ca"
    fi

cat << EOF > ${TEMP_FOLDER}/ibm-cp4ba-os-issuer.yaml
apiVersion: cert-manager.io/v1
kind: Issuer
metadata:
  name: opensearch-tls-issuer
  namespace: "$CP4BA_SERVICES_NS"
spec:
  ca:
    secretName: $cp4ba_root_ca_secret_name
EOF
    # ${CLI_CMD} delete -f ${TEMP_FOLDER}/ibm-cp4ba-os-issuer.yaml >/dev/null 2>&1
    ${CLI_CMD} apply -f ${TEMP_FOLDER}/ibm-cp4ba-os-issuer.yaml >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        success "Created opensearch-tls-issuer Issuer for Opensearch in the project \"$CP4BA_SERVICES_NS\"."
        sleep 3
    else
        warning "Failed to create opensearch-tls-issuer Issuer for Opensearch in the project \"$CP4BA_SERVICES_NS\"!"
        exit 1
    fi
}

function setup_opensearch_cr(){
# Create ElasticsearchCluster
    info "Creating Opensearch cluster in the project \"$CP4BA_SERVICES_NS\"."
    sa_scc_mcs=$(${CLI_CMD} get project $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o jsonpath='{.metadata.annotations.openshift\.io/sa\.scc\.mcs}')
    if [ -z $sa_scc_mcs ]; then
        fail "Can NOT get value for \"sa.scc.mcs\" from the attribute of project \"$CP4BA_SERVICES_NS\"."
        exit 1
    fi

    # retrieve the elasticsearch cr metaname
    elasticsearch_cr_name=$(${CLI_CMD} get elasticsearch.elastic.automation.ibm.com -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
    if [[ ! -z $elasticsearch_cr_name ]]; then
        cr_metaname=$(${CLI_CMD} get elasticsearch.elastic.automation.ibm.com $elasticsearch_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)

        es_storage_class_node=$(${CLI_CMD} get elasticsearch.elastic.automation.ibm.com $elasticsearch_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.nodegroupspecs.[0].storage.class)
        if [ -z $es_storage_class_node ]; then
            fail "Can NOT get value for \"storage.class\" from the existing Elasticsearch custom resource \"$elasticsearch_cr_name\" in the project \"$CP4BA_SERVICES_NS\"."
            exit 1
        fi

        es_storage_size_node=$(${CLI_CMD} get elasticsearch.elastic.automation.ibm.com $elasticsearch_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.nodegroupspecs.[0].storage.size)
        if [ -z $es_storage_size_node ]; then
            fail "Can NOT get value for \"storage.size\" from the existing Elasticsearch custom resource \"$elasticsearch_cr_name\" in the project \"$CP4BA_SERVICES_NS\"."
            exit 1
        fi
        es_storage_class_snapshot=$(${CLI_CMD} get elasticsearch.elastic.automation.ibm.com $elasticsearch_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.snapshotStores.[0].storage.class)
        if [ -z $es_storage_class_snapshot ]; then
            fail "Can NOT get value for \"storage.class\" from the existing Elasticsearch custom resource \"$elasticsearch_cr_name\" in the project \"$CP4BA_SERVICES_NS\"."
            exit 1
        fi

        es_storage_size_snapshot=$(${CLI_CMD} get elasticsearch.elastic.automation.ibm.com $elasticsearch_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.snapshotStores.[0].storage.size)
        if [ -z $es_storage_size_snapshot ]; then
            fail "Can NOT get value for \"storage.size\" from the existing Elasticsearch custom resource \"$elasticsearch_cr_name\" in the project \"$CP4BA_SERVICES_NS\"."
            exit 1
        fi

        es_secret_name="${cr_metaname}-elasticsearch-es-client-cert-kp"

        es_route_hostname=$(${CLI_CMD} get Route ${elasticsearch_cr_name}-es -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.host)
    else
        fail "Elasticsearch custom resource not found in the project \"$CP4BA_SERVICES_NS\"."
        exit 1
    fi
    # setting profile type
    if [[ $PROFILE_TYPE == "small" ]]; then
        es_resources_limits_cpu="800m"
        es_resources_limits_memory="2Gi"

        es_resources_requests_cpu="500m"
        es_resources_requests_memory="820Mi"

        ha_resources_limits_cpu="800m"
        ha_resources_limits_memory="2Gi"

        ha_resources_requests_cpu="500m"
        ha_resources_requests_memory="820Mi"
    elif [[ $PROFILE_TYPE == "medium" ]]; then
        es_resources_limits_cpu="1000m"
        es_resources_limits_memory="5Gi"

        es_resources_requests_cpu="500m"
        es_resources_requests_memory="3512Mi"

        ha_resources_limits_cpu="1000m"
        ha_resources_limits_memory="5Gi"

        ha_resources_requests_cpu="500m"
        ha_resources_requests_memory="3512Mi"
    elif [[ $PROFILE_TYPE == "large" ]]; then
        es_resources_limits_cpu="2000m"
        es_resources_limits_memory="5Gi"

        es_resources_requests_cpu="1000m"
        es_resources_requests_memory="3512Mi"

        ha_resources_limits_cpu="2000m"
        ha_resources_limits_memory="5Gi"

        ha_resources_requests_cpu="1000m"
        ha_resources_requests_memory="3512Mi"
    fi
cat << EOF > ${TEMP_FOLDER}/ibm-cp4ba-os-cluster.yaml
apiVersion: elasticsearch.opencontent.ibm.com/v1
kind: ElasticsearchCluster
metadata:
  annotations:
    productID: 068a62892a1e4db39641342e592daa25
    productMetric: FREE
    productName: IBM Cloud Platform Common Services
  name: opensearch
  namespace: "$CP4BA_SERVICES_NS"
spec:
  addDefaultPlugins: true
  minimumMasterNodes: 2
  allowRebuildOnChange: false
  enableDataNodeService: false
  sharedStoragePVC: ''
  credentialSecret: ''
  odlmRegistryNamespace: ibm-common-services
  tlsSecret: ''
  maxUnavailable: 1
  imagePullSecret: ''
  nodes:
    - nodeSelector: {}
      jvmOpts: ''
      master: true
      environmentVariables: {}
      ingest: true
      schedulerName: ''
      nodeAffinity: {}
      mountSecrets:
        - name: $es_secret_name
          path: /workdir/apps/elasticsearch/config/esos
      data: true
      name: all
      haResources:
        limits:
          cpu: $ha_resources_limits_cpu
          ephemeral-storage: 6Gi
          memory: $ha_resources_limits_memory
        requests:
          cpu: $ha_resources_requests_cpu
          ephemeral-storage: 10Mi
          memory: $ha_resources_requests_memory
      podAffinity: {}
      additionalConfiguration:
        - name: gateway.expected_data_nodes
          value: '0'
        - name: reindex.remote.allowlist
          value: '["$es_route_hostname:443"]'
        - name: reindex.ssl.certificate_authorities
          value: '/workdir/apps/elasticsearch/config/esos/ca.crt'
        - name: reindex.ssl.certificate
          value: '/workdir/apps/elasticsearch/config/esos/tls.crt'
        - name: reindex.ssl.key
          value: '/workdir/apps/elasticsearch/config/esos/tls.key'
      podAntiAffinity: {}
      topologySpreadConstraints: []
      addKeys: []
      esResources:
        limits:
          cpu: $es_resources_limits_cpu
          ephemeral-storage: 5Gi
          memory: $es_resources_limits_memory
        requests:
          cpu: $es_resources_requests_cpu
          ephemeral-storage: 10Mi
          memory: $es_resources_requests_memory
      secureEnvironmentVariables: []
      replicas: 3
      storageSize: '$es_storage_size_node'
      description: All in one nodes
      nodeGroupLabels:
        app: ibm-opensearch
      roles:
        - data
        - ingest
        - cluster_manager
      storageClass: $es_storage_class_node
  imagePullPolicy: IfNotPresent
  odlmRegistry: common-service
  tlsIssuer: 'opensearch-tls-issuer'
  useResourceRequestLimitsForJVMHeapRatio: 0.5
  snapshotRepo:
    enabled: true
    seLinuxOptionsLevels: '$sa_scc_mcs'
    size: 10Gi
    snapshotActivity: none
    storageClass: $es_storage_class_snapshot
  useResourceRequestLimitsForJVMHeap: true
  proxyHttpPort: 9200
  version: '2'
  serviceAccount: ''
  image: ''
  ignoreForMaintenance: false
  updateStrategy: RollingUpdate
  proxyTransportPort: 9300
  useODLM: false
  enableNetworkPolicy: true
  useCertificateManager: true
  sessionAffinity: ClientIP
EOF
    ${CLI_CMD} apply -f ${TEMP_FOLDER}/ibm-cp4ba-os-cluster.yaml >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        success "Created Opensearch cluster in the project \"$CP4BA_SERVICES_NS\"."
        sleep 3
    else
        warning "Failed to create Opensearch cluster in the project \"$CP4BA_SERVICES_NS\"!"
        exit 1
    fi

    info "Checking Opensearch cluster pod ready or not in the project \"$CP4BA_SERVICES_NS\""
    maxRetry=50
    for ((podnum=0;podnum<=2;podnum++)); do
        for ((retry=0;retry<=${maxRetry};retry++)); do
            opensearch_pod_name=$(${CLI_CMD} get pod -n $CP4BA_SERVICES_NS -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep "opensearch-ib-.*-es-server-all-${podnum}" | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
            if [[ -z $opensearch_pod_name ]]; then
                if [[ $retry -eq ${maxRetry} ]]; then
                    printf "\n"
                    if [[ -z $opensearch_pod_name ]]; then
                        warning "Timeout waiting for Opensearch cluster pod to be ready in the project \"$CP4BA_SERVICES_NS\""
                    fi
                    exit 1
                else
                    sleep 30
                    echo -n "..."
                    continue
                fi
            else
                break
            fi
        done
        success "Opensearch cluster pod-${podnum} ready in the project \"$CP4BA_SERVICES_NS\"!"
        msg "Pod: $opensearch_pod_name"
    done
}

function setup_opensearch_route(){
# Creat route for Opensearch
    info "Creating opensearch-route Route for Opensearch in the project \"$CP4BA_SERVICES_NS\"."
    openshift_hostname=$(${CLI_CMD} get IngressController default -n openshift-ingress-operator --no-headers --ignore-not-found -o jsonpath='{.status.domain}')
    if [ -z $openshift_hostname ]; then
        fail "Can NOT get value for \"openshift_ingress_domain.status.domain\" for project \"openshift-ingress-operator\"."
        exit 1
    fi
cat << EOF > ${TEMP_FOLDER}/ibm-cp4ba-os-route.yaml
kind: Route
apiVersion: route.openshift.io/v1
metadata:
  annotations:
    haproxy.router.openshift.io/balance: roundrobin
    haproxy.router.openshift.io/disable_cookies: 'true'
    haproxy.router.openshift.io/timeout: 3600s
  name: opensearch-route
  namespace: "$CP4BA_SERVICES_NS"
  labels:
    app.kubernetes.io/component: OpenSearch
    app.kubernetes.io/instance: ibm-opensearch
    app.kubernetes.io/managed-by: Operator
    app.kubernetes.io/name: ibm-opensearch
spec:
  host: opensearch-$CP4BA_SERVICES_NS.$openshift_hostname
  to:
    kind: Service
    name: opensearch-ibm-elasticsearch-srv
    weight: 100
  port:
    targetPort: https
  tls:
    termination: passthrough
    insecureEdgeTerminationPolicy: None
  wildcardPolicy: None
EOF
    ${CLI_CMD} delete -f ${TEMP_FOLDER}/ibm-cp4ba-os-route.yaml >/dev/null 2>&1
    ${CLI_CMD} apply -f ${TEMP_FOLDER}/ibm-cp4ba-os-route.yaml >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        success "Created opensearch-route Route for Opensearch in the project \"$CP4BA_SERVICES_NS\"."
        sleep 3
    else
        warning "Failed to create opensearch-route Route for Opensearch in the project \"$CP4BA_SERVICES_NS\"!"
        exit 1
    fi
}

function setup_opensearch_networkpolicy(){
# Creat networkPolicy for Opensearch for CPfs issue https://github.ibm.com/IBMPrivateCloud/roadmap/issues/63514 as workaround
    info "Creating \"opensearch-network-policy-allow-all\" network policy for Opensearch in the project \"$CP4BA_SERVICES_NS\"."
    # opensearch_ss_name=$(${CLI_CMD} get StatefulSet -n $CP4BA_SERVICES_NS -o 'custom-columns=NAME:.metadata.name' --no-headers | grep "opensearch-ib-.*-es-server-all" )
    # if [[ -z ${opensearch_ss_name} ]]; then
    #     fail "No found opensearch StatefulSet in the project \"$CP4BA_SERVICES_NS\"."
    # else
    # opensearch_label=$(${CLI_CMD} get StatefulSet $opensearch_ss_name -n $CP4BA_SERVICES_NS -o jsonpath='{.metadata.labels.ibm-es-server}')
cat << EOF > ${TEMP_FOLDER}/ibm-cp4ba-os-network-policy.yaml
kind: NetworkPolicy
apiVersion: networking.k8s.io/v1
metadata:
  name: ${cp4ba_cr_metaname}-cp4a-ingress-allow-opensearch
  namespace: "$CP4BA_SERVICES_NS"
spec:
  podSelector:
    matchLabels:
      app: ibm-opensearch
  ingress:
    - ports:
        - protocol: TCP
          port: 9200
        - protocol: TCP
          port: 443
        - protocol: TCP
          port: 2112
  policyTypes:
    - Ingress
EOF
        ${CLI_CMD} delete -f ${TEMP_FOLDER}/ibm-cp4ba-os-network-policy.yaml >/dev/null 2>&1
        ${CLI_CMD} apply -f ${TEMP_FOLDER}/ibm-cp4ba-os-network-policy.yaml >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            success "Created \"${cp4ba_cr_metaname}-cp4a-ingress-allow-opensearch\" network policy for Opensearch in the project \"$CP4BA_SERVICES_NS\"."
            sleep 3
        else
            warning "Failed to create \"${cp4ba_cr_metaname}-cp4a-ingress-allow-opensearch\" network policy for Opensearch in the project \"$CP4BA_SERVICES_NS\"!"
            exit 1
        fi
}

function select_upgrade_mode_simple(){
    UPGRADE_MODE=""
    while [[ $UPGRADE_MODE == "" ]]; do
        printf "\n"
        options=("Cluster-scoped to Cluster-scoped" "Cluster-scoped to Namespace-scoped" "Namespace-scoped to Namespace-scoped")
        tips=("This migration mode will first isolate the shared 3.x version of foundational services on the cluster, then install a new instance of foundational services version 4.6.2. Any data from the original foundational services will be copied to the new instance." "This migration mode will migrate a single shared instance of foundational services that is shared by all IBM Cloud Paks in the entire cluster, all of its core foundational services operators to the cluster-scope namespace will be upgraded v4.6.2 without creating new instances." "This migration mode will migrate a dedicated foundational services without creating any new instances of foundational services. This method should only be used if the foundational services you are migrating is only for IBM Cloud Pak for Business Automation.")
        pros_tips=("The namespace-scoped foundational services for each Cloud Pak or CP4BA component." "The fewer resource required for a single shared instance of foundational services.")
        cons_tips=("The more resource required for each namespace-scoped instance of IBM Cloud Pak foundational services." "It is not flexible for all Cloud Pak or CP4BA component to share same one instance of IBM Cloud Pak foundational services.")

        echo -e "\x1B[1mWhich migration mode for the IBM Cloud Pak foundational services do you plan to migrate to? \x1B[0m${YELLOW_TEXT}(NOTES: This choice will decide to either use a private catalog (namespace-scoped) or a global catalog namespace (GCN) for Opensearch installation. Make sure you choose SAME MIGRATION MODE when rerun [upgradeOperator] for upgrade IBM Cloud Pak foundational services next.)${RESET_TEXT}\x1B[0m"

        echo -e "${YELLOW_TEXT}1) Cluster-scoped to Cluster-scoped${RESET_TEXT}"
        if [[ $RUNTIME_MODE == "upgradeOperator" ]]; then
            echo -e "   ${YELLOW_TEXT}[Tips]${RESET_TEXT}: ${tips[1]}"
            # echo -e "   ${GREEN_TEXT}[Pros]${RESET_TEXT}: ${pros_tips[1]}"
            # echo -e "   ${RED_TEXT}[Cons]${RESET_TEXT}: ${cons_tips[1]}"
        fi
        # echo -e "${YELLOW_TEXT}2) Namespace-scoped to Namespace-scoped${RESET_TEXT}"
        # if [[ $RUNTIME_MODE == "upgradeOperator" ]]; then
        #     echo -e "   ${YELLOW_TEXT}[Tips]${RESET_TEXT}: ${tips[2]}"
        #     # echo -e "   ${GREEN_TEXT}[Pros]${RESET_TEXT}: ${pros_tips[0]}"
        #     # echo -e "   ${RED_TEXT}[Cons]${RESET_TEXT}: ${cons_tips[0]}"
        # fi
        echo -e "${YELLOW_TEXT}2) Cluster-scoped to Namespace-scoped${RESET_TEXT}"
        if [[ $RUNTIME_MODE == "upgradeOperator" ]]; then
            echo -e "   ${YELLOW_TEXT}[Tips]${RESET_TEXT}: ${tips[0]}"
            # echo -e "   ${GREEN_TEXT}[Pros]${RESET_TEXT}: ${pros_tips[0]}"
            # echo -e "   ${RED_TEXT}[Cons]${RESET_TEXT}: ${cons_tips[0]}"
        fi

        read -p "Enter your choice [1 or 2]: " choice
        case $choice in
            1)
                UPGRADE_MODE="shared2shared"
                ;;
            # 2)
            #     UPGRADE_MODE="dedicated2dedicated"
            #     ;;
            2)
                UPGRADE_MODE="shared2dedicated"
                ;;
            *)
                echo "Invalid choice. Select either 1 or 2."
                sleep 2
                ;;
        esac
        if [[ ! -z $UPGRADE_MODE ]]; then
            echo "You selected: ${options[$choice-1]}"
            printf "\n"
            sleep 2
        fi
    done
}

function setup_opensearch(){
    mkdir -p ${TEMP_FOLDER} >/dev/null 2>&1

    determine_upgrade_mode

    if [[ $UPGRADE_MODE == "shared2dedicated" || $UPGRADE_MODE == "dedicated2dedicated" ]]; then
        if ${CLI_CMD} get catalogsource -n $TARGET_PROJECT_NAME | grep ibm-cp4a-operator-catalog >/dev/null 2>&1; then
            PRIVATE_CATALOG_FOUND="Yes"
            ENABLE_PRIVATE_CATALOG=1
            info "Found this CP4BA deployment is installed using private catalog \"ibm-cp4a-operator-catalog\" under project \"$TARGET_PROJECT_NAME\", Opensearch will be installed using private catalog."
        else
            select_private_catalog_opensearch
        fi
    else
        info "If you plan to migrate IBM Cloud Pak foundational services \"cluster-scoped to cluster-scoped\", Opensearch will keep to use global catalog in the project \"openshift-marketplace\" same as other IBM Cloud Pak foundational services/CP4BA operators."
        sleep 3
    fi

    # if ${CLI_CMD} get catalogsource --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS | grep ibm-cp4a-operator-catalog >/dev/null 2>&1; then
    #     PRIVATE_CATALOG_FOUND="Yes"
    #     ENABLE_PRIVATE_CATALOG=1
    #     info "Found this CP4BA deployment is installed using private catalog in the project \"$CP4BA_SERVICES_NS\", the Opensearch will be installed using private catalog also"
    # elif ${CLI_CMD} get catalogsource --no-headers --ignore-not-found -n openshift-marketplace | grep ibm-cp4a-operator-catalog >/dev/null 2>&1; then
    #     PRIVATE_CATALOG_FOUND="No"
    #     # info "Found this CP4BA deployment is installed using global catalog in the project \"openshift-marketplace\""
    #     if [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
    #         ENABLE_PRIVATE_CATALOG=0
    #         info "Found CP4BA deployment is installed into All Namespaces, will keep to use global catalog in the project \"openshift-marketplace\" for Opensearch same as other CP4BA operators."
    #     else
    #         info "Found CP4BA deployment is installed into specific Namespaces."
    #         select_upgrade_mode_simple
    #         if [[ $UPGRADE_MODE == "shared2dedicated" || $UPGRADE_MODE == "dedicated2dedicated" ]]; then
    #             select_private_catalog_opensearch
    #         else
    #             info "The migration mode is \"cluster-scoped to cluster-scoped\", will keep to use global catalog in the project \"openshift-marketplace\" for Opensearch same as other CP4BA operators."
    #             sleep 3
    #         fi
    #     fi
    # fi

    if [[ ($ENABLE_PRIVATE_CATALOG -eq 1 && $ALL_NAMESPACE_FLAG == "Yes") || $ENABLE_PRIVATE_CATALOG -eq 0 ]]; then
        OPENSEARCH_CATALOG_NS="openshift-marketplace"
    elif [[ $ENABLE_PRIVATE_CATALOG -eq 1 ]]; then
        OPENSEARCH_CATALOG_NS="$CP4BA_SERVICES_NS"
    fi

    select_profile_type
    setup_opensearch_catalog
    # support All namespace scenario
    if [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
        setup_opensearch_subscription "openshift-operators"
    else
        setup_opensearch_subscription "$CP4BA_SERVICES_NS"
    fi

    setup_opensearch_issuer
    setup_opensearch_cr
    setup_opensearch_route
    setup_opensearch_networkpolicy

    openshift_user_pwd=$(${CLI_CMD} get secret opensearch-ibm-elasticsearch-cred-secret -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o jsonpath='{.data.elastic}')
    if [ -z $openshift_user_pwd ]; then
        fail "Can NOT get password from secret \"opensearch-ibm-elasticsearch-cred-secret\" in the project \"$CP4BA_SERVICES_NS\"."
        exit 1
    else
        tmp_val=$(echo "$openshift_user_pwd" | base64 -d)
        info "Access URL for Opensearch Cluster: https://opensearch-$CP4BA_SERVICES_NS.$openshift_hostname"
        msg "       username: elastic"
        msg "       password: ***************"
        echo -e '       NOTES: To get the password, use this command: \033[0;32m'"${CLI_CMD}"' get secret opensearch-ibm-elasticsearch-cred-secret -n '"$CP4BA_SERVICES_NS"' --no-headers --ignore-not-found -o jsonpath='{.data.elastic}' | base64 -d && echo\033[0m'
    fi
}

function create_project() {
    local project_name=$1
    project_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$project_name")

    isProjExists=`${CLI_CMD} get project $project_name --ignore-not-found | wc -l`  >/dev/null 2>&1

    if [ $isProjExists -ne 2 ] ; then
        oc new-project ${project_name} >/dev/null 2>&1
        returnValue=$?
        if [ "$returnValue" == 1 ]; then
            if [ -z "$CP4BA_AUTO_NAMESPACE" ]; then
                echo -e "\x1B[1;31mInvalid project name, enter a valid name...\x1B[0m"
                project_name=""
                return 1
            else
                echo -e "\x1B[1;31mInvalid project name \"$CP4BA_AUTO_NAMESPACE\". Set a valid name...\x1B[0m"
                project_name=""
                exit 1
            fi
        else
            echo -e "\x1B[1mUsing project ${project_name}...\x1B[0m"
            return 0
        fi
    else
        echo -e "\x1B[1mProject \"${project_name}\" already exists! Continue...\x1B[0m"
        return 0
    fi
}

function check_selection_migration(){
    es_to_os_migration_flag=$(${CLI_CMD} get configmap ibm-cp4ba-os-migration-status --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath={.data.opensearch_migration_done})
    # es_to_os_migration_flag=$(echo $es_to_os_migration_flag | tr '[:upper:]' '[:lower:]')
    if [[ $es_to_os_migration_flag == "NONEED" ]]; then
        ES_TO_OS_MIGRATION_SELECTED="No"
        MIGRATE_ES_TO_OS_DONE="NONEED"
        create_configmap_os_migration
        success "\"opensearch_migration_done\" was found to be \"$es_to_os_migration_flag\" in ibm-cp4ba-os-migration-status configMap in the project \"$CP4BA_SERVICES_NS\""
        info "Script will bypass the migration of Elasticsearch to OpenSearch and continue to upgrade CP4BA operators and IBM Cloud Pak foundational services."
        sleep 5
    elif [[ ! -z $es_to_os_migration_flag ]]; then
        ES_TO_OS_MIGRATION_SELECTED="Yes"
        success "\"opensearch_migration_done\" was found to be \"$es_to_os_migration_flag\" in ibm-cp4ba-os-migration-status configMap in the project \"$CP4BA_SERVICES_NS\""
    elif [[ -z $es_to_os_migration_flag ]]; then
        # For PFS
        while true; do
            printf "\n"
            printf "\x1B[1mAre you planning to use IBM Process Federation Server (PFS) after the upgrade AND if you want to perform the migration of Elasticsearch to OpenSearch?\x1B[0m (Yes/No): "
            read -rp "" ans
            case "$ans" in
            "y"|"Y"|"yes"|"Yes"|"YES")
                ES_TO_OS_MIGRATION_SELECTED="Yes"
                break
                ;;
            "n"|"N"|"no"|"No"|"NO")
                ES_TO_OS_MIGRATION_SELECTED="No"
                break
                ;;
            *)
                echo -e "Answer must be \"Yes\" or \"No\"\n"
                ;;
            esac
        done
    fi
}

function check_es_to_os_migration(){
    which jq &>/dev/null
    [[ $? -ne 0 ]] && \
    fail  "Unable to locate \"jq\" CLI. You must install it first for migration from Elasticsearch to Opensearch, for example \"yum -y install jq\" on Linux" && \
    exit 1

    printf "\n"
    while true; do
        printf "\x1B[1mDid you complete migration from Elasticsearch to Opensearch? \x1B[0m${YELLOW_TEXT}(NOTES: If you select \"No\", the script will guide you to complete migration.)${RESET_TEXT} (Yes/No, default: No): "
        read -rp "" ans
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES")
            info "Checking whether the ElasticsearchCluster custom resource exists in the project \"$CP4BA_SERVICES_NS\""
            ${CLI_CMD} get crd |grep elasticsearch.opencontent.ibm.com >/dev/null 2>&1
            if [ $? -eq 0 ]; then
                opensearch_cr_name=$(${CLI_CMD} get ElasticsearchCluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
            fi
            if [[ ! -z $opensearch_cr_name ]]; then
                os_cr_metaname=$(${CLI_CMD} get ElasticsearchCluster $opensearch_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
                if [[ ! -z $os_cr_metaname ]]; then
                    success "ElasticsearchCluster custom resource \"$os_cr_metaname\" was found in the project \"$CP4BA_SERVICES_NS\""
                fi

                info "Checking Opensearch cluster pods ready or not in the project \"$CP4BA_SERVICES_NS\""
                maxRetry=3
                declare -A opensearch_pod_name_array
                for ((podnum=0;podnum<=2;podnum++)); do
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        opensearch_pod_name_array[$podnum]=$(${CLI_CMD} get pod -n $CP4BA_SERVICES_NS -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep "opensearch-ib-.*-es-server-all-${podnum}" | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                        if [[ -z ${opensearch_pod_name_array[$podnum]} ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                printf "\n"
                                if [[ -z ${opensearch_pod_name_array[$podnum]} ]]; then
                                    fail "Not found Opensearch cluster all pods ready in the project \"$CP4BA_SERVICES_NS\""
                                    printf "\n"
                                    echo "${YELLOW_TEXT}[ATTENTION]: ${RESET_TEXT}${RED_TEXT}You need to install Elasticsearch cluster and complete migration from Elasticsearch to Opensearch first.${RESET_TEXT}"
                                fi
                                MIGRATE_ES_TO_OS_DONE="No"
                                create_configmap_os_migration
                                setup_opensearch
                                show_tips_es_to_os_migration
                                exit 1
                            else
                                sleep 1
                                echo -n "..."
                                continue
                            fi
                        else
                            break
                        fi
                    done
                    success "Opensearch cluster pod-${podnum} ready in the project \"$CP4BA_SERVICES_NS\"!"
                    msg "Pod: ${opensearch_pod_name_array[$podnum]}"
                done
                ## Add more logic to check migration done if CPfs migration script supports checking migration status.
                MIGRATE_ES_TO_OS_DONE="Yes"
                create_configmap_os_migration
            else
                fail "ElasticsearchCluster custom resource was not found in the project \"$CP4BA_SERVICES_NS\"."
                printf "\n"
                echo "${YELLOW_TEXT}[ATTENTION]: ${RESET_TEXT}${RED_TEXT}You need to install Opensearch cluster and complete migration from Elasticsearch to Opensearch first.${RESET_TEXT}"
                MIGRATE_ES_TO_OS_DONE="No"
                create_configmap_os_migration
                setup_opensearch
                show_tips_es_to_os_migration
                exit 1
            fi
            break
            ;;
        "n"|"N"|"no"|"No"|"NO"|"")
            MIGRATE_ES_TO_OS_DONE="No"
            create_configmap_os_migration
            ${CLI_CMD} get crd |grep elasticsearch.opencontent.ibm.com >/dev/null 2>&1
            if [ $? -eq 0 ]; then
                opensearch_cr_name=$(${CLI_CMD} get ElasticsearchCluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
            fi
            if [ -z $opensearch_cr_name ]; then
                while true; do
                    printf "\n"
                    printf "\x1B[1mDo you want to install Opensearch by script for migration Elasticsearch to Opensearch?\x1B[0m (Yes/No, default: Yes): "
                    read -rp "" ans
                    case "$ans" in
                    "y"|"Y"|"yes"|"Yes"|"YES"|"")
                        # MIGRATE_ES_TO_OS_DONE="No"
                        # create_configmap_os_migration
                        setup_opensearch
                        show_tips_es_to_os_migration
                        exit 1
                        ;;
                    "n"|"N"|"no"|"No"|"NO")
                        printf "\n"
                        echo -e "\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31mYou need to install Opensearch first for migration from Elasticsearch to Opensearch.\n\x1B[0m"
                        # MIGRATE_ES_TO_OS_DONE="No"
                        # create_configmap_os_migration
                        echo "Exiting..."
                        exit 1
                        ;;
                    *)
                        echo -e "Answer must be \"Yes\" or \"No\"\n"
                        ;;
                    esac
                done
            else
                success "One existing ElasticsearchCluster \"$opensearch_cr_name\" custom resource for Opensearch cluster was found in the project \"$CP4BA_SERVICES_NS\"."
                sleep 3

                # the script will check Opensearch pod ready or not even ElasticsearchCluster CR existing.
                info "Checking Opensearch cluster pod ready or not in the project \"$CP4BA_SERVICES_NS\""
                maxRetry=50
                for ((podnum=0;podnum<=2;podnum++)); do
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        opensearch_pod_name=$(${CLI_CMD} get pod -n $CP4BA_SERVICES_NS -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep "opensearch-ib-.*-es-server-all-${podnum}" | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                        if [[ -z $opensearch_pod_name ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                printf "\n"
                                if [[ -z $opensearch_pod_name ]]; then
                                    warning "Timeout waiting for Opensearch cluster pod to be ready in the project \"$CP4BA_SERVICES_NS\""
                                fi
                                exit 1
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            break
                        fi
                    done
                    success "Opensearch cluster pod-${podnum} ready in the project \"$CP4BA_SERVICES_NS\"!"
                    msg "Pod: $opensearch_pod_name"
                done

                openshift_route=$(${CLI_CMD} get route opensearch-route -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o name)
                if [[ -z $openshift_route ]]; then
                    warning "Not found route for opensearch cluster in the project \"$CP4BA_SERVICES_NS\"."
                    setup_opensearch_route
                    setup_opensearch_networkpolicy
                fi
            fi
            show_tips_es_to_os_migration
            exit 1
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}
function retrieve_dependencies(){
  # get pods -l 'name=ibm-content-operators' --no-headers --ignore-not-found
  # Check if the content operator exists
  # kubectl get pods | grep  cpe-deploy | head -n 1 | awk '{print $1}'
  local content_operator=$( $CLI_CMD get pods -l "${CONTENT_OPERATOR_LABEL}" --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}' )
  # Check if the cp4a operator exists for upgrade from 21.0.3+
  # ibm-cp4a-operator
  local cp4a_operator=$($CLI_CMD get pods -l "${CP4A_OPERATOR_LABEL}" --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}' )

  local cpe_pod=$($CLI_CMD get pods -n $CP4BA_SERVICES_NS -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep "${CPE_LABEL}" | grep 'Running' | grep 'true' | grep '<none>' | head -n 1 | awk '{print $1}')
  if [[ -z $cpe_pod ]]; then
    ${CLI_CMD} scale --replicas=1 deployment ${cr_metaname}-cpe-deploy -n $CP4BA_SERVICES_NS >/dev/null 2>&1
    info "Waiting for cpe-deploy pod to be ready in the project \"$CP4BA_SERVICES_NS\"."
    maxRetry=25
    for ((retry=0;retry<=${maxRetry};retry++)); do
        cpe_pod=$($CLI_CMD get pods -n $CP4BA_SERVICES_NS -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep "${CPE_LABEL}" | grep 'Running' | grep 'true' | grep '<none>' | head -n 1 | awk '{print $1}')
        if [[ -z $cpe_pod ]]; then
            if [[ $retry -eq ${maxRetry} ]]; then
                printf "\n"
                if [[ -z $cpe_pod ]]; then
                    warning "Timeout waiting for cpe-deploy pod to be ready in the project \"$CP4BA_SERVICES_NS\"."
                fi
                exit 1
            else
                sleep 30
                echo -n "..."
                continue
            fi
        else
            break
        fi
    done
  fi

  local option=$1

  if [[ ${cpe_pod} != ""  ]]; then
    $CLI_CMD cp $cpe_pod:${JACE_PATH} ${TEMP_FOLDER}/${JACE_NAME} -n $CP4BA_SERVICES_NS  >/dev/null 2>&1
    $CLI_CMD cp $cpe_pod:${CPE_TRUSTSTORE_PATH} ${TEMP_FOLDER}/${TRUSTSTORE_NAME} -n $CP4BA_SERVICES_NS  >/dev/null 2>&1
    OLD_TRUSTSTORE_EXISTS=$( $CLI_CMD exec -i -n $CP4BA_SERVICES_NS  $cpe_pod -- bash -c "(ls ${CPE_TRUSTSTORE_PATH} >> /dev/null 2>&1 && echo yes) || echo no")
    if [[ $option == "POSTUPGRADE"  || $OLD_TRUSTSTORE_EXISTS == "no" ]]; then
      $CLI_CMD cp $cpe_pod:${POST_UPGRADE_CPE_TRUSTSTORE_PATH} ${TEMP_FOLDER}/${POST_UPGRADE_TRUSTSTORE_NAME} -n $CP4BA_SERVICES_NS  >/dev/null 2>&1
    fi
  else
    return 1
  fi


  if [[  ${content_operator} != "" ]]; then
    EXEC_OPERATOR=$content_operator

    $CLI_CMD cp ${TEMP_FOLDER}/${JACE_NAME}  $content_operator:/tmp/${JACE_NAME} -n $TEMP_OPERATOR_PROJECT_NAME  >/dev/null 2>&1
    $CLI_CMD cp ${TEMP_FOLDER}/${TRUSTSTORE_NAME} $content_operator:/tmp/${TRUSTSTORE_NAME} -n $TEMP_OPERATOR_PROJECT_NAME   >/dev/null 2>&1
    $CLI_CMD cp ${CUR_DIR}/helper/$RUNNABLE_JAR_NAME  $content_operator:/tmp/${RUNNABLE_JAR_NAME} -n $TEMP_OPERATOR_PROJECT_NAME  >/dev/null 2>&1

    if [[ $option == "POSTUPGRADE" || $OLD_TRUSTSTORE_EXISTS == "no" ]]; then
      $CLI_CMD cp ${TEMP_FOLDER}/${POST_UPGRADE_TRUSTSTORE_NAME} $content_operator:/tmp/${POST_UPGRADE_TRUSTSTORE_NAME} -n $TEMP_OPERATOR_PROJECT_NAME  >/dev/null 2>&1
    fi

  elif [[ ${cp4a_operator} != "" ]]; then
    EXEC_OPERATOR=$cp4a_operator
    $CLI_CMD cp ${TEMP_FOLDER}/${JACE_NAME}  $cp4a_operator:/tmp/${JACE_NAME} -n $TEMP_OPERATOR_PROJECT_NAME  >/dev/null 2>&1
    $CLI_CMD cp ${TEMP_FOLDER}/${TRUSTSTORE_NAME} $cp4a_operator:/tmp/${TRUSTSTORE_NAME} -n $TEMP_OPERATOR_PROJECT_NAME  >/dev/null 2>&1
    $CLI_CMD cp ${CUR_DIR}/helper/$RUNNABLE_JAR_NAME  $cp4a_operator:/tmp/${RUNNABLE_JAR_NAME} -n $TEMP_OPERATOR_PROJECT_NAME   >/dev/null 2>&1
    if [[ $option == "POSTUPGRADE" || $OLD_TRUSTSTORE_EXISTS == "no" ]]; then
     $CLI_CMD cp ${TEMP_FOLDER}/${POST_UPGRADE_TRUSTSTORE_NAME} $cp4a_operator:/tmp/${POST_UPGRADE_TRUSTSTORE_NAME} -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
   fi

  else
    return 1
  fi
  return 0
}

function is_scim_enabled(){
  retrieve_dependencies
  ecode=$?
  if [[ $ecode -ne 0 ]]; then
    error "Failed to retrieve dependencies to determine if Content Process Engine is configured with SCIM as the directory provider type."
    warning "Assuming that Content Process Engine directory provider type is set to SCIM."
    echo "    ${YELLOW_TEXT}* We advice you to verify that Content Process Engine directory provider type is truly SCIM to determine if you need to execute the script cp4a-pre-upgrade-and-post-upgrade-optional.sh${RESET_TEXT}"
    echo "      1. Log in to the Administration Console for Content Platform Engine."
    echo "      2. Go to Directory Configuration tab."
    echo "      3. Verify your Directory Server Type."
    IS_SCIM_ENABLED="True"

  else

    if [[ $CP4BA_IBM_FNCM_SECRET_NAME == "" ]]; then
      IBM_FNCM_SECRET_NAME='ibm-fncm-secret'
    else
      IBM_FNCM_SECRET_NAME="$CP4BA_IBM_FNCM_SECRET_NAME"
    fi

    local app_login_user=$( decode_xor_password $( $CLI_CMD get secret $IBM_FNCM_SECRET_NAME -n $CP4BA_SERVICES_NS -o jsonpath='{ .data.appLoginUsername }' | base64 -d ) $TEMP_OPERATOR_PROJECT_NAME $EXEC_OPERATOR )
    local app_login_pwd=$( decode_xor_password $( $CLI_CMD get secret $IBM_FNCM_SECRET_NAME -n $CP4BA_SERVICES_NS -o jsonpath='{ .data.appLoginPassword }' | base64 -d ) $TEMP_OPERATOR_PROJECT_NAME $EXEC_OPERATOR )
    local key_store_pass=$( decode_xor_password $( $CLI_CMD get secret $IBM_FNCM_SECRET_NAME -n $CP4BA_SERVICES_NS -o jsonpath='{ .data.keystorePassword }' | base64 -d ) $TEMP_OPERATOR_PROJECT_NAME $EXEC_OPERATOR | sed  's/\$/\\$/g' )
    local cpe_svc_name=$( $CLI_CMD get svc -n $CP4BA_SERVICES_NS | grep $CPE_SERVICE_NAME | awk '{print $1}' )
    local class_path="/tmp/Jace.jar;/opt/ibm/content_emitter/stax-api.jar;/opt/ibm/content_emitter/xlxpScanner.jar;/opt/ibm/content_emitter/xlxpScannerUtils.jar"
    local truststore="/tmp/ibm_customFNCMTrustStore.p12"
    if [[ $OLD_TRUSTSTORE_EXISTS == "no" ]]; then
        truststore="/tmp/trusts.p12"
    fi
    if [[ $UPGRADE_MODE == "dedicated2dedicated" ]]; then
      TARGET_PROJECT_NAME_CS=$TARGET_PROJECT_NAME
    elif [[ $UPGRADE_MODE == "shared2shared" || $UPGRADE_MODE == "shared2dedicated" ]]; then
      TARGET_PROJECT_NAME_CS="ibm-common-services"
    fi

    IS_SCIM_ENABLED_RESPONSE=$( ${CLI_CMD} exec -i -n $TEMP_OPERATOR_PROJECT_NAME $EXEC_OPERATOR -- bash -c "java -cp \"${CLASS_PATH}\" -jar -Duser.language=en -Duser.country=US -Djavax.net.ssl.trustStore=$truststore -Djavax.net.ssl.trustStoreType=pkcs12  -Djavax.net.ssl.trustStorePassword=\"${key_store_pass}\" /tmp/${RUNNABLE_JAR_NAME} SCIMENABLED $cpe_svc_name.$CP4BA_SERVICES_NS.svc $CPE_SERVICE_PORT $app_login_user  $app_login_pwd  $TARGET_PROJECT_NAME_CS" )
    ## "TRUE : The p8domain is configured with a scim directory"
    if [[ $IS_SCIM_ENABLED_RESPONSE =~ "TRUE : The p8domain is configured with a scim directory" ]]; then
      IS_SCIM_ENABLED="True"
      info "Content Process Engine directory provider type is set to SCIM."
    else
      IS_SCIM_ENABLED="False"
      info "Content Process Engine directory provider type is not set to SCIM."
    fi
  fi
}

function set_script_mode(){
    if [[ -f $TEMPORARY_PROPERTY_FILE && -f $DB_NAME_USER_PROPERTY_FILE && -f $DB_SERVER_INFO_PROPERTY_FILE && -f $LDAP_PROPERTY_FILE ]]; then
        DEPLOYMENT_WITH_PROPERTY="Yes"
    else
        DEPLOYMENT_WITH_PROPERTY="No"
    fi
}

function validate_kube_oc_cli(){
    if  [[ $PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS" ]]; then
        which oc &>/dev/null
        [[ $? -ne 0 ]] && \
        echo -e  "\x1B[1;31mUnable to locate the OpenShift CLI. You must install it to run this script.\x1B[0m" && \
        exit 1
    fi
    if  [[ $PLATFORM_SELECTED == "other" ]]; then
        which kubectl &>/dev/null
        [[ $? -ne 0 ]] && \
        echo -e  "\x1B[1;31mUnable to locate the Kubernetes CLI, You must install it to run this script.\x1B[0m" && \
        exit 1
    fi
}

function prop_tmp_property_file() {
    grep "\b${1}\b" ${TEMPORARY_PROPERTY_FILE}|cut -d'=' -f2
}

function load_property_before_generate(){
    if [[ ! -f $TEMPORARY_PROPERTY_FILE || ! -f $DB_NAME_USER_PROPERTY_FILE || ! -f $DB_SERVER_INFO_PROPERTY_FILE || ! -f $LDAP_PROPERTY_FILE ]]; then
        fail "Property file not found under \"$PROPERTY_FILE_FOLDER\". Run the \"cp4a-prerequisites.sh\" script to complete the prerequisites."
        exit 1
    fi

    # load pattern into pattern_cr_arr
    pattern_list="$(prop_tmp_property_file PATTERN_LIST)"
    pattern_name_list="$(prop_tmp_property_file PATTERN_NAME_LIST)"
    optional_component_list="$(prop_tmp_property_file OPTION_COMPONENT_LIST)"
    optional_component_name_list="$(prop_tmp_property_file OPTION_COMPONENT_NAME_LIST)"
    foundation_list="$(prop_tmp_property_file FOUNDATION_LIST)"

    OIFS=$IFS
    IFS=',' read -ra pattern_cr_arr <<< "$pattern_list"
    IFS=',' read -ra PATTERNS_CR_SELECTED <<< "$pattern_list"

    IFS=',' read -ra pattern_arr <<< "$pattern_name_list"
    IFS=',' read -ra optional_component_cr_arr <<< "$optional_component_list"
    IFS=',' read -ra optional_component_arr <<< "$optional_component_name_list"
    IFS=',' read -ra foundation_component_arr <<< "$foundation_list"
    IFS=$OIFS

    # load db_name_full_array and db_user_full_array
    db_name_list="$(prop_tmp_property_file DB_NAME_LIST)"
    db_user_list="$(prop_tmp_property_file DB_USER_LIST)"
    db_user_pwd_list="$(prop_tmp_property_file DB_USER_PWD_LIST)"

    OIFS=$IFS
    IFS=',' read -ra db_name_full_array <<< "$db_name_list"
    IFS=',' read -ra db_user_full_array <<< "$db_user_list"
    IFS=',' read -ra db_user_pwd_full_array <<< "$db_user_pwd_list"
    IFS=$OIFS

    # load db ldap type
    LDAP_TYPE="$(prop_tmp_property_file LDAP_TYPE)"
    DB_TYPE="$(prop_tmp_property_file DB_TYPE)"
    
    # making sure the DB type is in lowercase
    DB_TYPE=$(echo "$DB_TYPE" | tr '[:upper:]' '[:lower:]')
    # For Database type DB2 DB2HADR and DB2 RDS the generate mode and validate mode are all identical and in the script taken care off using $DB_TYPE == "db2"
    if [[ $DB_TYPE == "db2"* ]]; then
        DB_TYPE="db2"
    fi

    # load CONTENT_OS_NUMBER
    content_os_number=$(prop_tmp_property_file CONTENT_OS_NUMBER)

    # load DB_SERVER_NUMBER
    db_server_number=$(prop_tmp_property_file DB_SERVER_NUMBER)

    # load external ldap flag
    SET_EXT_LDAP=$(prop_tmp_property_file EXTERNAL_LDAP_ENABLED)

    # load limited CPE storage support flag
    CPE_FULL_STORAGE=$(prop_tmp_property_file CPE_FULL_STORAGE_ENABLED)

    # load GPU enabled worker nodes flag
    ENABLE_GPU_ARIA=$(prop_tmp_property_file ENABLE_GPU_ARIA_ENABLED)
    nodelabel_key=$(prop_tmp_property_file NODE_LABEL_KEY)
    nodelabel_value=$(prop_tmp_property_file NODE_LABEL_VALUE)

    # load LDAP/DB required flag for wfps
    LDAP_WFPS_AUTHORING=$(prop_tmp_property_file LDAP_WFPS_AUTHORING_FLAG)
    EXTERNAL_DB_WFPS_AUTHORING=$(prop_tmp_property_file EXTERNAL_DB_WFPS_AUTHORING_FLAG)

    # load fips enabled flag
    FIPS_ENABLED=$(prop_tmp_property_file FIPS_ENABLED_FLAG)

    # load profile size  flag
    PROFILE_TYPE=$(prop_tmp_property_file PROFILE_SIZE_FLAG)
}

function validate_docker_podman_cli(){
    if [[ $OCP_VERSION == "3.11" || "$machine" == "Mac" ]];then
        which podman &>/dev/null
        if [[ $? -ne 0 ]]; then
            PODMAN_FOUND="No"

            which docker &>/dev/null
            [[ $? -ne 0 ]] && \
                DOCKER_FOUND="No"
            if [[ $DOCKER_FOUND == "No" && $PODMAN_FOUND == "No" ]]; then
                echo -e "\x1B[1;31mUnable to locate docker and podman. Install either of them first.\x1B[0m" && \
                exit 1
            fi
        fi
    elif [[ $OCP_VERSION == "4.4OrLater" ]]
    then
        which podman &>/dev/null
        [[ $? -ne 0 ]] && \
            echo -e "\x1B[1;31mUnable to locate podman. Install it first.\x1B[0m" && \
            exit 1
    fi
}

function select_project() {
    while [[ $TARGET_PROJECT_NAME == "" ]];
    do
        printf "\n"
        echo -e "\x1B[1mWhere do you want to deploy Cloud Pak for Business Automation?\x1B[0m"
        read -p "Enter the name for an existing project (namespace): " TARGET_PROJECT_NAME
        if [ -z "$TARGET_PROJECT_NAME" ]; then
            echo -e "\x1B[1;31mEnter a valid project name, project name can not be blank\x1B[0m"
        elif [[ "$TARGET_PROJECT_NAME" == openshift* ]]; then
            echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'openshift' or start with 'openshift' \x1B[0m"
            TARGET_PROJECT_NAME=""
        elif [[ "$TARGET_PROJECT_NAME" == kube* ]]; then
            echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'kube' or start with 'kube' \x1B[0m"
            TARGET_PROJECT_NAME=""
        else
            isProjExists=`${CLI_CMD} get project $TARGET_PROJECT_NAME --ignore-not-found | wc -l`  >/dev/null 2>&1

            if [ "$isProjExists" -ne 2 ] ; then
                echo -e "\x1B[1;31mInvalid project name. Enter an existing project name ...\x1B[0m"
                TARGET_PROJECT_NAME=""
            else
                echo -e "\x1B[1mUsing project ${TARGET_PROJECT_NAME}...\x1B[0m"
            fi
        fi
    done
}

function containsElement(){
    local e match="$1"
    shift
    for e; do [[ "$e" == "$match" ]] && return 0; done
    return 1
}

function containsObjectStore(){
    OBJECT_NAME=$1
    FILE=$2
    os_num=0
    os_index_array=()
    while true; do
        object_name_tmp=`cat $FILE | ${YQ_CMD} r - spec.datasource_configuration.dc_os_datasources.[$os_num].dc_common_os_datasource_name`

        if [ -z "$object_name_tmp" ]; then
            break
        else
            if [[ "$OBJECT_NAME" == "$object_name_tmp" ]]; then
                os_index_array=( "${os_index_array[@]}" "${os_num}" )
            fi
        fi
        ((os_num++))
    done
}

function getTotalFNCMObjectStore(){
    object_array=("FNOS1DS" "FNOS2DS" "FNOS3DS" "FNOS4DS" "FNOS5DS" "FNOS6DS" "FNOS7DS" "FNOS8DS" "FNOS9DS" "FNOS10DS")
    FILE=$1
    os_index_array=()
    total_os=0
    for object_name in "${object_array[@]}"
    do
        os_num=0
        while true; do
            object_name_tmp=`cat $FILE | ${YQ_CMD} r - spec.datasource_configuration.dc_os_datasources.[$os_num].dc_common_os_datasource_name`
            if [ -z "$object_name_tmp" ]; then
                break
            else
                if [[ "$object_name" == "$object_name_tmp" ]]; then
                    os_index_array=( "${os_index_array[@]}" "${os_num}" )
                fi
            fi
            ((os_num++))
        done
    done
    total_os=${#os_index_array[@]}
}

function containsInitObjectStore(){
    OBJECT_NAME=$1
    FILE=$2
    os_num=0
    os_index_array=()
    while true; do
        object_name_tmp=`cat $FILE | ${YQ_CMD} r - spec.initialize_configuration.ic_obj_store_creation.object_stores.[$os_num].oc_cpe_obj_store_display_name`
        if [ -z "$object_name_tmp" ]; then
            break
        else
            if [[ "$OBJECT_NAME" == "$object_name_tmp" ]]; then
                os_index_array=( "${os_index_array[@]}" "${os_num}" )
            fi
        fi
        ((os_num++))
    done
}

function containsInitLDAPGroups(){
    FILE=$1
    ldap_num=0
    ldap_groups_index_array=()
    while true; do
        name_tmp=`cat $FILE | ${YQ_CMD} r - spec.initialize_configuration.ic_ldap_creation.ic_ldap_admins_groups_name.[$ldap_num]`
        if [ -z "$name_tmp" ]; then
            break
        else
            ldap_groups_index_array=( "${ldap_groups_index_array[@]}" "${ldap_num}" )
        fi
        ((ldap_num++))
    done
}

function containsInitLDAPUsers(){
    FILE=$1
    ldap_num=0
    ldap_users_index_array=()
    while true; do
        name_tmp=`cat $FILE | ${YQ_CMD} r - spec.initialize_configuration.ic_ldap_creation.ic_ldap_admin_user_name.[$ldap_num]`
        if [ -z "$name_tmp" ]; then
            break
        else
            ldap_users_index_array=( "${ldap_users_index_array[@]}" "${ldap_num}" )
        fi
        ((ldap_num++))
    done
}

function containsBAWInstance(){
    BAW_INS_NAME=$1
    FILE=$2
    baw_instance_num=0
    baw_index_array=()
    while true; do
        name_tmp=`cat $FILE | ${YQ_CMD} r - spec.baw_configuration.[$baw_instance_num].name`
        if [ -z "$name_tmp" ]; then
            break
        else
            if [[ "$BAW_INS_NAME" == "$name_tmp" ]]; then
                baw_index_array=( "${baw_index_array[@]}" "${baw_instance_num}" )
            fi
        fi
        ((baw_instance_num++))
    done
}

function containsAEInstance(){
    FILE=$1
    ae_instance_num=0
    ae_index_array=()
    while true; do
        name_tmp=`cat $FILE | ${YQ_CMD} r - spec.application_engine_configuration.[$ae_instance_num].name`
        if [ -z "$name_tmp" ]; then
            break
        else
            ae_index_array=( "${ae_index_array[@]}" "${ae_instance_num}" )
        fi
        ((ae_instance_num++))
    done
}

function containsICNRepos(){
    FILE=$1
    icn_repo_instance_num=0
    icn_repo_index_array=()
    while true; do
        name_tmp=`cat $FILE | ${YQ_CMD} r - spec.initialize_configuration.ic_icn_init_info.icn_repos.[$icn_repo_instance_num].add_repo_id`
        if [ -z "$name_tmp" ]; then
            break
        else
            icn_repo_index_array=( "${icn_repo_index_array[@]}" "${icn_repo_instance_num}" )
        fi
        ((icn_repo_instance_num++))
    done
}

function containsICNDesktop(){
    FILE=$1
    icn_desktop_instance_num=0
    icn_desktop_index_array=()
    while true; do
        name_tmp=`cat $FILE | ${YQ_CMD} r - spec.initialize_configuration.ic_icn_init_info.icn_desktop.[$icn_desktop_instance_num].add_desktop_id`
        if [ -z "$name_tmp" ]; then
            break
        else
            icn_desktop_index_array=( "${icn_desktop_index_array[@]}" "${icn_desktop_instance_num}" )
        fi
        ((icn_desktop_instance_num++))
    done
}

function containsTenantDB(){
    FILE=$1
    tenant_db_instance_num=0
    tenant_db_index_array=()
    while true; do
        name_tmp=`cat $FILE | ${YQ_CMD} r - spec.datasource_configuration.dc_ca_datasource.tenant_databases.[$tenant_db_instance_num]`
        if [ -z "$name_tmp" ]; then
            break
        else
            tenant_db_index_array=( "${tenant_db_index_array[@]}" "${tenant_db_instance_num}" )
        fi
        ((tenant_db_instance_num++))
    done
}

function get_baw_mode(){
    if [[ "$SCRIPT_MODE" == "baw" || "$SCRIPT_MODE" == "baw-dev" ]]; then
       return 0
    else
       return 1
    fi
}

function select_platform(){
    printf "\n"
    echo -e "\x1B[1mSelect the cloud platform to deploy: \x1B[0m"
    COLUMNS=12
    if [ -z "$existing_platform_type" ]; then
        if [[ $DEPLOYMENT_TYPE == "starter" ]];then
            options=("RedHat OpenShift Kubernetes Service (ROKS) - Public Cloud" "Openshift Container Platform (OCP) - Private Cloud")
            PS3='Enter a valid option [1 to 2]: '
        elif [[ $DEPLOYMENT_TYPE == "production" ]]
        then
            if [[ "${SCRIPT_MODE}" == "OLM" ]]; then
                options=("RedHat OpenShift Kubernetes Service (ROKS) - Public Cloud" "Openshift Container Platform (OCP) - Private Cloud")
                PS3='Enter a valid option [1 to 2]: '
            else
                options=("RedHat OpenShift Kubernetes Service (ROKS) - Public Cloud" "Openshift Container Platform (OCP) - Private Cloud")
                PS3='Enter a valid option [1 to 2]: '
            fi
        fi
        if [[ -z "${CP4BA_AUTO_PLATFORM}" ]]; then
            select opt in "${options[@]}"
            do
                case $opt in
                    "RedHat OpenShift Kubernetes Service (ROKS) - Public Cloud")
                        PLATFORM_SELECTED="ROKS"
                        use_entitlement="yes"
                        break
                        ;;
                    "Openshift Container Platform (OCP) - Private Cloud")
                        PLATFORM_SELECTED="OCP"
                        use_entitlement="yes"
                        break
                        ;;
                    *) echo "invalid option $REPLY";;
                esac
            done
        else
           PLATFORM_SELECTED="${CP4BA_AUTO_PLATFORM}"
           use_entitlement="yes"
        fi
    else
        if [[ $DEPLOYMENT_TYPE == "starter" ]];then
            options=("RedHat OpenShift Kubernetes Service (ROKS) - Public Cloud" "Openshift Container Platform (OCP) - Private Cloud")
            options_var=("ROKS" "OCP")
        elif [[ $DEPLOYMENT_TYPE == "production" ]]
        then
            if [[ "${SCRIPT_MODE}" == "OLM" ]]; then
                options=("RedHat OpenShift Kubernetes Service (ROKS) - Public Cloud" "Openshift Container Platform (OCP) - Private Cloud")
                options_var=("ROKS" "OCP")
            else
                options=("RedHat OpenShift Kubernetes Service (ROKS) - Public Cloud" "Openshift Container Platform (OCP) - Private Cloud")
                options_var=("ROKS" "OCP")
            fi
        fi
        for i in ${!options_var[@]}; do
            if [[ "${options_var[i]}" == "$existing_platform_type" ]]; then
                printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "(Selected)"
            else
                printf "%1d) %s\n" $((i+1)) "${options[i]}"
            fi
        done
        echo -e "\x1B[1;31mExisting platform type found in CR: \"$existing_platform_type\"\x1B[0m"
        # echo -e "\x1B[1;31mDo not need to select again.\n\x1B[0m"
        prompt_press_any_key_to_continue
    fi

    if [[ "$PLATFORM_SELECTED" == "OCP" || "$PLATFORM_SELECTED" == "ROKS" ]]; then
        CLI_CMD=oc
    elif [[ "$PLATFORM_SELECTED" == "other" ]]
    then
        CLI_CMD=kubectl
    fi

    validate_kube_oc_cli

    # For Azure Red Hat OpenShift (ARO)/Red Hat OpenShift Service on AWS (ROSA)
    if [[ "$PLATFORM_SELECTED" == "OCP" && "${DEPLOYMENT_TYPE}" == "starter" ]] || [[ "$PLATFORM_SELECTED" == "OCP" && "${DEPLOYMENT_TYPE}" == "production" ]] ; then   #DBACLD-166320 This code changes addressing the issue while the customer deploying CP4BA into ARO or AWS
        while true; do
            printf "\n"
            printf "\x1B[1mIs your OCP deployed on AWS or Azure? (Yes/No, default: No): \x1B[0m"
            read -rp "" ans
            case "$ans" in
            "y"|"Y"|"yes"|"Yes"|"YES")
                printf "\n"
                echo -e "\x1B[1mWhich platform is OCP deployed on? \x1B[0m"
                COLUMNS=12
                options=("AWS (ROSA: Red Hat OpenShift Service on AWS)" "Azure (ARO: Azure Red Hat OpenShift)")
                PS3='Enter a valid option [1 to 2]: '
                select opt in "${options[@]}"
                do
                    case $opt in
                        "Azure"*)
                            OCP_PLATFORM="ARO"
                            break
                            ;;
                        "AWS"*)
                            OCP_PLATFORM="ROSA"
                            break
                            ;;
                        *) echo "invalid option $REPLY";;
                    esac
                done
                break
                ;;
            "n"|"N"|"no"|"No"|"NO"|"")
                OCP_PLATFORM=""
                break
                ;;
            *)
                echo -e "Answer must be \"Yes\" or \"No\"\n"
                ;;
            esac
        done
    fi
}

function check_ocp_version(){
    if [[ ${PLATFORM_SELECTED} == "OCP" || ${PLATFORM_SELECTED} == "ROKS" ]];then
        temp_ver=`${CLI_CMD} version | grep v[1-9]\.[1-9][0-9] | tail -n1`
        if [[ $temp_ver == *"Kubernetes Version"* ]]; then
            currentver="${temp_ver:20:7}"
        else
            currentver="${temp_ver:11:7}"
        fi
        requiredver="v1.17.1"
        if [ "$(printf '%s\n' "$requiredver" "$currentver" | sort -V | head -n1)" = "$requiredver" ]; then
            OCP_VERSION="4.4OrLater"
        else
            # OCP_VERSION="3.11"
            OCP_VERSION="4.4OrLater"
            echo -e "\x1B[1;31mIMPORTANT: The apiextensions.k8s.io/v1beta API has been deprecated from k8s 1.16+, OCP 4.3 is using k8s 1.16.x. recommend you to upgrade your OCP to version 4.4 or later\n\x1B[0m"
            prompt_press_any_key_to_continue
            # exit 0
        fi
    fi
}

function select_pattern(){
# This function support mutiple checkbox, if do not select anything, it will return None

    PATTERNS_SELECTED=""
    choices_pattern=()
    pattern_arr=()
    pattern_cr_arr=()
    AUTOMATION_SERVICE_ENABLE=""
    AE_DATA_PERSISTENCE_ENABLE=""
    CPE_FULL_STORAGE=""


    if [[ "${PLATFORM_SELECTED}" == "other" ]]; then
        if [[ "${DEPLOYMENT_TYPE}" == "starter" ]];
        then
            options=("FileNet Content Manager" "Operational Decision Manager" "Automation Decision Services" "Business Automation Application" "Business Automation Workflow Authoring and Automation Workstream Services" "IBM Automation Document Processing")
            options_cr_val=("content" "decisions" "decisions_ads" "application" "workflow-workstreams" "document_processing")
            foundation_0=("BAN" "RR")                 # Foundation for FileNet Content Manager
            foundation_1=("BAN" "RR")                # Foundation for Operational Decision Manager
            foundation_2=("BAN" "RR" "UMS")     # Foundation for Automation Decision Services
            foundation_3=("RR" "UMS" "BAS")     # Foundation for Business Automation Applications (full)
            foundation_4=("RR" "UMS" "AE" "BAS")           # Foundation for Business Automation Workflow and workstreams(Demo)
            foundation_5=("BAN" "RR" "AE" "BAS" "UMS")  # Foundation for IBM Automation Document Processing
        else
            options=("FileNet Content Manager" "Operational Decision Manager" "Automation Decision Services" "Business Automation Application" "Business Automation Workflow" "(a) Workflow Authoring" "(b) Workflow Runtime" "Automation Workstream Services" "IBM Automation Document Processing" "(a) Development Environment" "(b) Runtime Environment" "Workflow Process Service Authoring")
            options_cr_val=("content" "decisions" "decisions_ads" "application" "workflow" "workflow-authoring" "workflow-runtime" "workstreams" "document_processing" "document_processing_designer" "document_processing_runtime" "workflow-process-service")
            foundation_0=("BAN" "RR")                 # Foundation for FileNet Content Manager
            foundation_1=("BAN" "RR")                 # Foundation for Operational Decision Manager
            foundation_2=("BAN" "RR" "UMS")     # Foundation for Automation Decision Services
            foundation_3=("BAN" "RR" "UMS" "AE")     # Foundation for Business Automation Applications (full)
            foundation_4=("BAN" "RR")           # Foundation for dummy
            foundation_5=("BAN" "RR" "UMS" "BAS")          # Foundation for Business Automation Workflow - Workflow Authoring (5a)
            foundation_6=("BAN" "RR" "UMS" "AE")           # Foundation for Business Automation Workflow - Workflow Runtime (5b)
            foundation_7=("BAN" "RR" "UMS" "AE")           # Foundation for Automation Workstream Services (6)
            foundation_8=("BAN" "RR")  # Foundation for IBM Automation Document Processing
            foundation_9=("BAN" "RR" "AE" "BAS" "UMS")  # Foundation for IBM Automation Document Processing - 7a Development Environment
            foundation_10=("BAN" "RR" "AE" "UMS")  # Foundation for IBM Automation Document Processing - 7b Runtime Environment
            foundation_11=("BAS")           # Foundation for Workflow Process Service Authoring
            foundation_12=("BAN" "RR" "UMS" "AE")           # Foundation for Business Automation Workflow and workstreams(5b+6)
        fi
    else
        if [[ "${DEPLOYMENT_TYPE}" == "starter" ]];
        then
            options=("FileNet Content Manager" "Operational Decision Manager" "Automation Decision Services" "Business Automation Application" "Business Automation Workflow Authoring and Automation Workstream Services" "IBM Automation Document Processing")
            options_cr_val=("content" "decisions" "decisions_ads" "application" "workflow-workstreams" "document_processing")
            foundation_0=("BAN" "RR")                 # Foundation for FileNet Content Manager
            foundation_1=("BAN" "RR")                # Foundation for Operational Decision Manager
            foundation_2=("BAN" "RR")     # Foundation for Automation Decision Services
            foundation_3=("RR" "BAS")     # Foundation for Business Automation Applications (full)
            foundation_4=("RR" "AE" "BAS")           # Foundation for Business Automation Workflow and workstreams(Demo)
            foundation_5=("BAN" "RR" "AE" "BAS")  # Foundation for IBM Automation Document Processing
        else
            options=("FileNet Content Manager" "Operational Decision Manager" "Automation Decision Services" "Business Automation Application" "Business Automation Workflow" "(a) Workflow Authoring" "(b) Workflow Runtime" "Automation Workstream Services" "IBM Automation Document Processing" "(a) Development Environment" "(b) Runtime Environment" "Workflow Process Service Authoring")
            options_cr_val=("content" "decisions" "decisions_ads" "application" "workflow" "workflow-authoring" "workflow-runtime" "workstreams" "document_processing" "document_processing_designer" "document_processing_runtime" "workflow-process-service")
            foundation_0=("BAN" "RR")                 # Foundation for FileNet Content Manager
            foundation_1=("BAN" "RR")                 # Foundation for Operational Decision Manager
            foundation_2=("BAN" "RR")     # Foundation for Automation Decision Services
            foundation_3=("BAN" "RR" "AE")     # Foundation for Business Automation Applications (full)
            foundation_4=("BAN" "RR")           # Foundation for dummy
            foundation_5=("BAN" "RR" "BAS")           # Foundation for Business Automation Workflow - Workflow Authoring (5a)
            foundation_6=("BAN" "RR" "AE")           # Foundation for Business Automation Workflow - Workflow Runtime (5b)
            foundation_7=("BAN" "RR" "AE")           # Foundation for Automation Workstream Services (6)
            foundation_8=("BAN" "RR")  # Foundation for IBM Automation Document Processing
            foundation_9=("BAN" "RR" "AE" "BAS")  # Foundation for IBM Automation Document Processing - 7a Development Environment
            foundation_10=("BAN" "RR" "AE")  # Foundation for IBM Automation Document Processing - 7b Runtime Environment
            foundation_11=("BAS")           # Foundation for Workflow Process Service Authoring
            foundation_12=("BAN" "RR" "AE")           # Foundation for Business Automation Workflow and workstreams(5b+6)
        fi
    fi
    patter_ent_input_array=("1" "2" "3" "4" "5a" "5b" "5A" "5B" "6" "7a" "7b" "7A" "7B" "8" "5b,6" "5B,6" "5b, 6" "5B, 6" "5b 6" "5B 6")
    tips1="\x1B[1;31mTips\x1B[0m:\x1B[1m Press [ENTER] to accept the default (None of the capabilities is selected). If none of the capabilities is chosen, the script will exit.\x1B[0m"
    tips2="\x1B[1;31mTips\x1B[0m:\x1B[1m Press [ENTER] when you are done\x1B[0m"
    pattern_starter_tips="\x1B[1mInfo: Business Automation Navigator will be automatically installed in the environment as it is part of the Cloud Pak for Business Automation foundation platform. \n\nTips: After you make your first selection you will be able to make additional selections since you can combine multiple selections.\n\x1B[0m"
    pattern_production_tips="\x1B[1mInfo: Business Automation Navigator will be automatically installed in the environment as it is part of the Cloud Pak for Business Automation foundation platform. \n\nTips: After you make your first selection you will be able to make additional selections since you can combine multiple selections.\n\x1B[0m"
    baw_iaws_tips="\x1B[1mInfo: Note that Business Automation Workflow Authoring (5a) cannot be installed together with Automation Workstream Services (6). However, Business Automation Workflow Runtime (5b) can be installed together with Automation Workstream Services (6).\n\x1B[0m"
    linux_starter_tips="\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31mIBM Automation Document Processing (6) does NOT support a cluster running a Linux on Z (s390x) or Power (ppc64le) architecture.\n\x1B[0m"
    linux_production_tips="\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31mIBM Automation Document Processing (7a/7b) does NOT support a cluster running a Linux on Z (s390x) or Power (ppc64le) architecture.\n\x1B[0m"
    content_deployed_tips="\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31m\"FileNet Content Manager\" can not be selected because one Content (Kind: content.icp4a.ibm.com) custom resource was deployed.\n\x1B[0m"
    indexof() {
        i=-1
        for ((j=0;j<${#options_cr_val[@]};j++));
        do [ "${options_cr_val[$j]}" = "$1" ] && { i=$j; break; }
        done
        echo $i
    }
    menu() {
        clear
        echo -e "\x1B[1mSelect the Cloud Pak for Business Automation capability to install: \x1B[0m"
        for i in ${!options[@]}; do
            if [[ $DEPLOYMENT_TYPE == "starter" ]];then
                containsElement "${options_cr_val[i]}" "${EXISTING_PATTERN_ARR[@]}"
                retVal=$?
                if [ $retVal -ne 0 ]; then
                    printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "${choices_pattern[i]}"
                else
                    if [[ "${choices_pattern[i]}" == "(To Be Uninstalled)" ]]; then
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "${choices_pattern[i]}"
                    else
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "(Installed)"
                    fi
                fi
            elif [[ $DEPLOYMENT_TYPE == "production" ]]
            then
                containsElement "${options_cr_val[i]}" "${EXISTING_PATTERN_ARR[@]}"
                retVal=$?
                if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime") && !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") ]]; then
                    wwVal=0
                elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime" && " ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams" ]]; then
                    wwVal=1
                fi
                containsElement "baw_authoring" "${EXISTING_OPT_COMPONENT_ARR[@]}"
                baw_authoring_Val=$?
                containsElement "document_processing_designer" "${EXISTING_OPT_COMPONENT_ARR[@]}"
                document_processing_designer_Val=$?
                containsElement "document_processing_runtime" "${EXISTING_OPT_COMPONENT_ARR[@]}"
                document_processing_runtime_Val=$?
                if [[ $retVal -ne 0 ]]; then
                    case "$i" in
                    "7") # for Automation Workstream Services
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" 6 "${options[i]}"  "${choices_pattern[i]}"
                        ;;
                    "8")
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" 7 "${options[i]}"  "${choices_pattern[i]}"
                        printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                        printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                        ;;
                    "9") # for wfps
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" 8 "${options[i+2]}"  "${choices_pattern[i+2]}"
                        ;;
                    "4") # 5 for Workflow Authoring, 6 for Workflow Runtime
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "${choices_pattern[i]}"
                        printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                        printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                        ;;
                    "0"|"1"|"2"|"3")
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "${choices_pattern[i]}"
                        ;;
                    esac
                else
                    if [[ "${choices_pattern[i]}" == "(To Be Uninstalled)" ]]; then
                        case "$i" in
                        "7") # for Automation Workstream Services
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" 6 "${options[i]}"  "${choices_pattern[i]}"
                            ;;
                        "4") # 5 for Workflow Authoring, 6 for Workflow Runtime
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "${choices_pattern[i]}"
                            printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                            printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                            ;;
                        "0"|"1"|"2"|"3"|"4")
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "${choices_pattern[i]}"
                            ;;
                        "8") # 9 for Development Environment, 10 for Runtime Environment,
                            # if [[ "${choices_pattern[i+1]}" == "(Selected)" || "${choices_pattern[i+2]}" == "(Selected)" ]]; then
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" 7 "${options[i]}"  "${choices_pattern[i]}"
                            printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                            printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                            ;;
                        "9") # for wfps
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" 8 "${options[i+2]}"  "${choices_pattern[i+2]}"
                            ;;
                        esac
                    else
                        case "$i" in
                        "7") # for Automation Workstream Services
                            if [[ (${choices_pattern[6]} == "(To Be Uninstalled)" && ${choices_pattern[7]} == "(To Be Uninstalled)") ]]; then
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" 6 "${options[i]}"  "${choices_pattern[i]}"
                            else
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" 6 "${options[i]}"  "(Installed)"
                            fi
                            ;;
                        "4") # 5 for Workflow Authoring, 6 for Workflow Runtime
                            if [[ ${choices_pattern[6]} == "(To Be Uninstalled)" && ${choices_pattern[7]} == "(To Be Uninstalled)" && ${choices_pattern[5]} == "" ]]; then
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "(To Be Uninstalled)"
                                if [[ $baw_authoring_Val -eq 0 ]]; then
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                                else
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                                fi
                            elif [[ ${choices_pattern[6]} == "(To Be Uninstalled)" && ${choices_pattern[7]} == "(To Be Uninstalled)" && ${choices_pattern[5]} == "(Selected)" ]]; then
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "${choices_pattern[i]}"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                            elif [[ $baw_authoring_Val -eq 0 && ${choices_pattern[5]} == "(To Be Uninstalled)" && ${choices_pattern[6]} != "" ]]; then
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"   "${choices_pattern[i]}"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"   "${choices_pattern[i+1]}"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                            elif [[ $baw_authoring_Val -eq 0 && ${choices_pattern[5]} == "(To Be Uninstalled)" && ${choices_pattern[6]} == "" ]]; then
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"   "(To Be Uninstalled)"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"   "${choices_pattern[i+1]}"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                            else
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"   "(Installed)"
                                if [[ $baw_authoring_Val -eq 0 ]]; then
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "(Installed)"
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                                else
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "(Installed)"
                                fi
                            fi
                            ;;
                        "0"|"1"|"2"|"3")
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "(Installed)"
                            ;;
                        "8")
                            if [[ ${choices_pattern[9]} == "" && ${choices_pattern[10]} == "" ]]; then
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" 7 "${options[i]}"  "(Installed)"
                                if [[ $document_processing_designer_Val -eq 0 ]]; then
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "(Installed)"
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i]}"
                                elif [[ $document_processing_runtime_Val -eq 0 ]]
                                then
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i]}"
                                    printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "(Installed)"
                                fi
                            elif [[ (${choices_pattern[9]} == "(To Be Uninstalled)" && ${choices_pattern[10]} == "(Selected)") || (${choices_pattern[9]} == "(Selected)" && ${choices_pattern[10]} == "(To Be Uninstalled)") ]]; then
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" 7 "${options[i]}"  "(Selected)"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                            elif [[ (${choices_pattern[8]} == "(Selected)" && ${choices_pattern[9]} == "(To Be Uninstalled)") || (${choices_pattern[8]} == "(Selected)" && ${choices_pattern[10]} == "(To Be Uninstalled)") ]]; then
                                printf "%1d) %s \x1B[1m%s\x1B[0m\n" 7 "${options[i]}"  "(To Be Uninstalled)"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+1]}"  "${choices_pattern[i+1]}"
                                printf "%s \x1B[1m%s\x1B[0m\n" "   ${options[i+2]}"  "${choices_pattern[i+2]}"
                            fi
                            ;;
                        "9") # for wfps
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" 8 "${options[i+2]}"  "(Installed)"
                            ;;
                        esac
                   fi
                fi
            fi
        done
        if [[ "$msg" ]]; then echo "$msg"; fi
        printf "\n"
        if [[ $DEPLOYMENT_TYPE == "production" ]]; then
            echo -e "${baw_iaws_tips}"
        fi

        if [[ $DEPLOYMENT_TYPE == "production" ]]; then
            echo -e "${pattern_production_tips}"
            echo -e "${linux_production_tips}"
            if [[ $CONTENT_DEPLOYED == "Yes" ]]; then
                echo -e "${content_deployed_tips}"
            fi
        else
            echo -e "${pattern_starter_tips}"
            echo -e "${linux_starter_tips}"
        fi
        # Show different tips according components select or unselect
        containsElement "(Selected)" "${choices_pattern[@]}"
        retVal=$?
        if [ $retVal -ne 0 ]; then
            echo -e "${tips1}"
        else
            echo -e "${tips2}"
        fi
# ##########################DEBUG############################
#     for i in "${!choices_pattern[@]}"; do
#         printf "%s\t%s\n" "$i" "${choices_pattern[$i]}"
#     done
# ##########################DEBUG############################
    }

    if [[ $DEPLOYMENT_TYPE == "starter" ]]; then
        prompt="Enter a valid option [1 to ${#options[@]}]: "
    elif [[ $DEPLOYMENT_TYPE == "production" ]]
    then
        prompt="Enter a valid option [1 to 4, 5a, 5b, 6, 7a, 7b, 8]: "
    fi

    while menu && read -rp "$prompt" num && [[ "$num" ]]; do
        if [[ $DEPLOYMENT_TYPE == "starter" ]]; then
            [[ "$num" != *[![:digit:]]* ]] &&
            (( num > 0 && num <= ${#options[@]} )) ||
            { msg="Invalid option: $num"; continue; }
            ((num--));
        elif [[ $DEPLOYMENT_TYPE == "production" ]]
        then
            containsElement "${num}" "${patter_ent_input_array[@]}"
            inputretVal=$?
            [[ "${inputretVal}" -eq 0 ]] ||
            { msg="Invalid option: $num"; continue; }
            case "$num" in
            "5a"|"5A")
                num=5
                if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") ]]; then
                    choices_pattern[6]=""
                    choices_pattern[7]=""
                elif [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime" || " ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && (${choices_pattern[6]} == "" || ${choices_pattern[7]} == "") ]]; then
                    choices_pattern[5]="(Selected)"
                elif [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime") && (" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && (${choices_pattern[6]} == "(To Be Uninstalled)") && (${choices_pattern[7]} == "(To Be Uninstalled)") ]]; then
                    num=5
                elif [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-authoring") && ${choices_pattern[5]} == "(To Be Uninstalled)" && (${choices_pattern[6]} == "(Selected)" || ${choices_pattern[7]} == "(Selected)") ]]; then
                    num=5
                fi
                ;;
            "5b"|"5B")
                num=6
                if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") ]]; then

                    choices_pattern[5]=""
                elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime" && " ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams" && ${choices_pattern[5]} == "(Selected)" ]]; then

                    choices_pattern[6]=""
                elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-authoring" ]]; then
                        if [[ ${choices_pattern[5]} == "(To Be Uninstalled)" ]]; then

                            num=6
                        elif [[ ${choices_pattern[5]} == "(Selected)" || ${choices_pattern[5]} == "" ]]; then
                            choices_pattern[6]="(Selected)"
                            # choices_pattern[7]=""
                        fi
                fi
                ;;
            6)
                num=7
                if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") ]]; then
                    choices_pattern[5]=""
                elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-authoring" ]]; then
                        if [[ ${choices_pattern[5]} == "(To Be Uninstalled)" ]]; then
                            num=7
                        elif [[ ${choices_pattern[5]} == "(Selected)" || ${choices_pattern[5]} == "" ]]; then
                            choices_pattern[7]="(Selected)"
                            # choices_pattern[7]=""
                        fi
                fi
                ;;
            "5b,6"|"5B,6"|"5b, 6"|"5B, 6"|"5b 6"|"5B 6")
                num=12
                if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") ]]; then
                    choices_pattern[5]=""
                else
                    if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-authoring" || ${choices_pattern[5]} == "" ]]; then
                        choices_pattern[6]="(Selected)"
                        choices_pattern[7]="(Selected)"
                    fi
                fi
                ;;
            "1"|"2"|"3"|"4")
                ((num--))
                ;;
            "7a"|"7A")
                num=9
                if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing") ]]; then
                    choices_pattern[10]=""
                else
                    if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing" && " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "document_processing_designer" && ${choices_pattern[10]} == "" ]]; then
                        num=9
                    elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing" && " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "document_processing_designer" && ${choices_pattern[10]} == "(Selected)" ]]; then
                        choices_pattern[9]=""
                        choices_pattern[8]=""
                    elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing" && " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "document_processing_runtime" && ${choices_pattern[10]} == "" ]]; then
                        choices_pattern[9]="(Selected)"
                    fi
                fi
                ;;
            "7b"|"7B")
                num=10
                if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing") ]]; then
                    choices_pattern[9]=""
                else
                    if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing" && " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "document_processing_runtime" && ${choices_pattern[9]} == "" ]]; then
                        num=10
                    elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing" && " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "document_processing_runtime" && ${choices_pattern[9]} == "(Selected)" ]]; then
                        choices_pattern[10]=""
                        choices_pattern[8]=""
                    elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing" && " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "document_processing_designer" && ${choices_pattern[9]} == "" ]]; then
                        choices_pattern[10]="(Selected)"
                    fi
                fi
                ;;
            "8")
                num=11
                ;;
            esac
        else
            echo "Deployment type is invalid"
            exit 0
        fi
        containsElement "${options_cr_val[num]}" "${EXISTING_PATTERN_ARR[@]}"
        retVal=$?
        containsElement "baw_authoring" "${EXISTING_OPT_COMPONENT_ARR[@]}"
        baw_authoring_Val=$?
        if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime") && !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") ]]; then
            wwVal=0
        elif [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime" && " ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams" ]]; then
            wwVal=1
        fi

        if [[ $retVal -ne 0 ]]; then
            if [[ ($num -eq 12) && ($wwVal -eq 0) ]]; then
                [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(Selected)"
                [[ "${choices_pattern[num]}" ]] && choices_pattern[4]="(Selected)" || choices_pattern[4]=""
                [[ "${choices_pattern[num]}" ]] && choices_pattern[6]="(Selected)" || choices_pattern[6]=""
                [[ "${choices_pattern[num]}" ]] && choices_pattern[7]="(Selected)" || choices_pattern[7]=""
            elif [[ ($num -eq 12) && ($wwVal -eq 1) ]]; then
                if [[ ${choices_pattern[4]} == "(Selected)" && ${choices_pattern[5]} == "(Selected)" ]]; then
                    choices_pattern[6]="(To Be Uninstalled)"
                    choices_pattern[7]="(To Be Uninstalled)"
                    choices_pattern[12]="(To Be Uninstalled)"
                else
                    [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(To Be Uninstalled)"
                    [[ "${choices_pattern[num]}" ]] && choices_pattern[4]="(To Be Uninstalled)" || choices_pattern[4]=""
                    [[ "${choices_pattern[num]}" ]] && choices_pattern[6]="(To Be Uninstalled)" || choices_pattern[6]=""
                    [[ "${choices_pattern[num]}" ]] && choices_pattern[7]="(To Be Uninstalled)" || choices_pattern[7]=""
                fi
            else
                [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(Selected)"
            fi
            if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                if [[ ${choices_pattern[5]} == "(Selected)" || ${choices_pattern[6]} == "(Selected)"  ]]; then
                    choices_pattern[4]="(Selected)"
                fi
                if  [[ "${choices_pattern[5]}" == "" && "${choices_pattern[6]}" == "" ]]; then
                    choices_pattern[4]=""
                fi
                if [[ ${choices_pattern[9]} == "(Selected)" || ${choices_pattern[10]} == "(Selected)"  ]]; then
                    choices_pattern[8]="(Selected)"
                fi
                if  [[ "${choices_pattern[9]}" == "" && "${choices_pattern[10]}" == "" ]]; then
                    choices_pattern[8]=""
                fi
                if [[ ${choices_pattern[0]} == "(Selected)" && "$CONTENT_DEPLOYED" == "Yes" && "$INSTALLATION_TYPE" == "new" ]]; then
                    choices_pattern[0]=""
                fi
            fi
        else
            if [[ $DEPLOYMENT_TYPE == "starter" ]]; then
                [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(To Be Uninstalled)"
            elif [[ $DEPLOYMENT_TYPE == "production" ]]
            then
                case "$num" in
                "5")
                    if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-authoring" && ("${choices_pattern[6]}" == "(Selected)" || "${choices_pattern[7]}" == "(Selected)") ]]; then
                        choices_pattern[num]="(To Be Uninstalled)"
                    else
                        [[ "${choices_pattern[num]}" ]] && choices_pattern[num-1]="" || choices_pattern[num-1]="(To Be Uninstalled)"
                        [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(To Be Uninstalled)"
                    fi
                    ;;
                "6")
                    if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams" && "${choices_pattern[7]}" == "(To Be Uninstalled)" ]]; then
                        if [[ "${choices_pattern[5]}" == "" ]]; then
                            if [[ choices_pattern[num]="(To Be Uninstalled)" ]]; then
                                choices_pattern[num]="(To Be Uninstalled)"
                            else
                                choices_pattern[num]=""
                            fi
                        elif [[ "${choices_pattern[5]}" == "(Selected)" ]]; then
                            choices_pattern[num]="(To Be Uninstalled)"
                        fi

                        # choices_pattern[num-2]="(Installed)"
                    elif  [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams" && "${choices_pattern[7]}" == "" && " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow" && "${choices_pattern[6]}" == "" ]]; then
                        choices_pattern[num]=""
                    else
                        [[ "${choices_pattern[num]}" ]] && choices_pattern[num-2]="" || choices_pattern[num-2]="(To Be Uninstalled)"
                        [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(To Be Uninstalled)"
                    fi
                    ;;
                "7")
                    if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime" && "${choices_pattern[6]}" == "(To Be Uninstalled)" ]]; then
                        if [[ "${choices_pattern[5]}" == "" ]]; then
                            if [[ choices_pattern[num]="(To Be Uninstalled)" ]]; then
                                choices_pattern[num]="(To Be Uninstalled)"
                            else
                                choices_pattern[num]=""
                            fi
                        elif [[ "${choices_pattern[5]}" == "(Selected)" ]]; then
                            choices_pattern[num]="(To Be Uninstalled)"
                        fi

                        # choices_pattern[num-2]="(Installed)"
                    elif  [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime" && "${choices_pattern[7]}" == "" && " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow" && "${choices_pattern[6]}" == "" ]]; then
                        choices_pattern[num]=""
                    else
                        [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(To Be Uninstalled)"
                    fi
                    ;;
                "9")
                    if [[ ${choices_pattern[10]} == "(Selected)" ]]; then
                        choices_pattern[8]="(Selected)"
                    else
                        [[ "${choices_pattern[num]}" ]] && choices_pattern[num-1]="" || choices_pattern[num-1]="(To Be Uninstalled)"
                    fi
                    [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(To Be Uninstalled)"
                    ;;
                "10")
                    if [[ ${choices_pattern[9]} == "(Selected)" ]]; then
                        choices_pattern[8]="(Selected)"
                    else
                        [[ "${choices_pattern[num]}" ]] && choices_pattern[num-2]="" || choices_pattern[num-2]="(To Be Uninstalled)"
                    fi
                    [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(To Be Uninstalled)"
                    ;;
                "0"|"1"|"2"|"3")
                    [[ "${choices_pattern[num]}" ]] && choices_pattern[num]="" || choices_pattern[num]="(To Be Uninstalled)"
                    ;;
                esac
            fi
        fi
    done

    # echo "choices_pattern: ${choices_pattern[*]}"
    # read -rsn1 -p"Press Enter/Return to continue (DEBUG MODEL)";echo
    # Generate list of the pattern which will be installed or To Be Uninstalled
    for i in ${!options[@]}; do
        array_varname=foundation_$i[@]
        containsElement "${options_cr_val[i]}" "${EXISTING_PATTERN_ARR[@]}"
        retVal=$?
        if [ $retVal -ne 0 ]; then
            [[ "${choices_pattern[i]}" ]] && { pattern_arr=( "${pattern_arr[@]}" "${options[i]}" ); pattern_cr_arr=( "${pattern_cr_arr[@]}" "${options_cr_val[i]}" ); msg=""; }
            [[ "${choices_pattern[i]}" ]] && { foundation_component_arr=( "${foundation_component_arr[@]}" "${!array_varname}" ); }
        else
            if [[ "${choices_pattern[i]}" == "(To Be Uninstalled)" ]]; then
                pos=`indexof "${pattern_cr_arr[i]}"`
                if [[ "$pos" != "-1" ]]; then
                { pattern_cr_arr=(${pattern_cr_arr[@]:0:$pos} ${pattern_cr_arr[@]:$(($pos + 1))}); pattern_arr=(${pattern_arr[@]:0:$pos} ${pattern_arr[@]:$(($pos + 1))}); }

                fi
            else
                { pattern_arr=( "${pattern_arr[@]}" "${options[i]}" ); pattern_cr_arr=( "${pattern_cr_arr[@]}" "${options_cr_val[i]}" ); msg=""; }
                { foundation_component_arr=( "${foundation_component_arr[@]}" "${!array_varname}" ); }
            fi
        fi
    done
    echo -e "$msg"

    # 4Q: add workflow-workstream into pattern list when select both workflow-runtime and workstream
    # https://jsw.ibm.com/browse/DBACLD-174822 (modified if condition by changing workflow to workflow-runtime)
    if [[ " ${pattern_cr_arr[@]} " =~ "workflow-runtime" && " ${pattern_cr_arr[@]} " =~ "workstreams" && "${DEPLOYMENT_TYPE}" == "production" ]]; then
        pattern_cr_arr=( "${pattern_cr_arr[@]}" "workflow-workstreams" )
        if [[ $PLATFORM_SELECTED == "other" ]]; then
            foundation_ww=("BAN" "RR" "UMS" "AE")
        else
            foundation_ww=("BAN" "RR" "AE")
        fi
        foundation_component_arr=( "${foundation_component_arr[@]}" "${foundation_ww[@]}" )
    fi

    if [ "${#pattern_arr[@]}" -eq "0" ]; then
        PATTERNS_SELECTED="None"
        printf "\x1B[1;31mNo selection was made. At least one capability must be chosen to proceed. Exiting. \n\x1B[0m"
        exit 1
    else
        PATTERNS_SELECTED=$( IFS=$','; echo "${pattern_arr[*]}" )
        PATTERNS_CR_SELECTED=$( IFS=$','; echo "${pattern_cr_arr[*]}" )

    fi
    if [[ "$DEPLOYMENT_TYPE" == "production" ]]; then
        select_ae_data_persistence
        AUTOMATION_SERVICE_ENABLE="No"
    fi
    # select_cpe_full_storage
    FOUNDATION_CR_SELECTED=($(echo "${foundation_component_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))

    x=0;while [ ${x} -lt ${#FOUNDATION_CR_SELECTED[*]} ] ; do FOUNDATION_CR_SELECTED_LOWCASE[$x]=$(tr [A-Z] [a-z] <<< ${FOUNDATION_CR_SELECTED[$x]}); let x++; done
    FOUNDATION_DELETE_LIST=($(echo "${FOUNDATION_CR_SELECTED[@]}" "${FOUNDATION_FULL_ARR[@]}" | tr ' ' '\n' | sort | uniq -u))

    PATTERNS_CR_SELECTED=($(echo "${pattern_cr_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
}

function select_optional_component(){
# This function support mutiple checkbox, if do not select anything, it will return
    OPT_COMPONENTS_CR_SELECTED=()
    OPTIONAL_COMPONENT_DELETE_LIST=()
    KEEP_COMPOMENTS=()
    OPT_COMPONENTS_SELECTED=()
    optional_component_arr=()
    optional_component_cr_arr=()
    BAI_SELECTED=""
    show_optional_components(){
        COMPONENTS_SELECTED=""
        choices_component=()
        component_arr=()

        tips1="\x1B[1;31mTips\x1B[0m:\x1B[1m Press [ENTER] if you do not want any optional components or when you are finished selecting your optional components\x1B[0m"
        tips2="\x1B[1;31mTips\x1B[0m:\x1B[1m Press [ENTER] when you are done\x1B[0m"
        fncm_tips="\x1B[1mNote: IBM Enterprise Records (IER) and IBM Content Collector for SAP (ICCSAP) do not integrate with User Management Service (UMS).\n"
        linux_starter_tips="\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31mIBM Content Collector for SAP (4) does NOT support a cluster running a Linux on Power architecture.\n\x1B[0m"
        linux_production_tips="\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31mIBM Content Collector for SAP (5) does NOT support a cluster running a Linux on Power architecture.\n\x1B[0m"
        ads_tips="\x1B[1mTips:\x1B[0m Decision Designer is typically required if you are deploying a development or test environment.\nThis feature will automatically install Business Automation Studio, if not already present. \n\nDecision Runtime is typically recommended if you are deploying a test or production environment.\n\nDecision Runtime is required when Decision Designer is selected.\n\nYou should choose at least one these features to have a minimum environment configuration.\n"
        if [[ $DEPLOYMENT_TYPE == "starter" ]];then
            decision_tips="\x1B[1mTips:\x1B[0m Decision Center, Rule Execution Server and Decision Runner will be installed by default.\n"
        else
            decision_tips="\x1B[1mTips:\x1B[0m Decision Center is typically required for development and testing environments. \nRule Execution Server is typically required for testing and production environments and for using Business Automation Insights. \nYou should choose at least one these 2 features to have a minimum environment configuration. \n"
        fi
        application_tips_demo="\x1B[1mTips:\x1B[0m Application Designer is typically required if you are deploying a development or test environment.\nThis feature will automatically install Business Automation Studio, if not already present.  \n\nMake your selection or press enter to proceed. \n"
        application_tips_ent="\x1B[1mTips:\x1B[0m Application Designer is typically required if you are deploying a development or test environment.\nThis feature will automatically install Business Automation Studio, if not already present. \n\nApplication Engine is automatically installed in the environment.  \n\nMake your selection or press enter to proceed. \n"

        indexof() {
            i=-1
            for ((j=0;j<${#optional_component_cr_arr[@]};j++));
            do [ "${optional_component_cr_arr[$j]}" = "$1" ] && { i=$j; break; }
            done
            echo $i
        }
        menu() {
            clear
            echo -e "\x1B[1;31mCapability \"$item_pattern\": \x1B[0m\x1B[1mSelect optional components: \x1B[0m"
            # echo -e "\x1B[1mSelect optional components: \x1B[0m"
            containsElement "bai" "${EXISTING_OPT_COMPONENT_ARR[@]}"
            bai_cr_retVal=$?
            for i in ${!optional_components_list[@]}; do
                if [[ ("${choices_component[i]}" == "(Selected)" || "${choices_component[i]}" == "(Installed)") && "${optional_components_list[i]}" == "Business Automation Insights" ]];then
                    BAI_SELECTED="Yes"
                elif [[ ( $bai_cr_retVal -ne 0 || "${choices_component[i]}" == "(To Be Uninstalled)") && "${optional_components_list[i]}" == "Business Automation Insights" ]]
                then
                    BAI_SELECTED="No"
                fi
            done

            for i in ${!optional_components_list[@]}; do
                containsElement "${optional_components_cr_list[i]}" "${EXISTING_OPT_COMPONENT_ARR[@]}"
                retVal=$?
                containsElement "${optional_components_cr_list[i]}" "${optional_component_cr_arr[@]}"
                selectedVal=$?
                if [ $retVal -ne 0 ]; then
                    if [[ "${item_pattern}" == "FileNet Content Manager" || ( "${item_pattern}" == "Operational Decision Manager" && "${DEPLOYMENT_TYPE}" == "production" ) ]];then
                        if [[ "${optional_components_list[i]}" == "User Management Service" && "${BAI_SELECTED}" == "Yes" ]];then
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "(Selected)"
                        elif [ $selectedVal -ne 0 ]
                        then
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "${choices_component[i]}"
                        else
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "(Selected)"
                        fi
                    else
                        if [ $selectedVal -ne 0 ]; then
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "${choices_component[i]}"
                        else
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "(Selected)"
                        fi
                    fi
                else
                    if [[ "${optional_components_list[i]}" == "User Management Service" ]];then
                        if [[ "${choices_component[i]}" == "(To Be Uninstalled)" ]]; then
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "${choices_component[i]}"
                        else
                            printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "(Installed)"
                        fi
                    elif [[ "${choices_component[i]}" == "(To Be Uninstalled)" ]]
                    then
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "${choices_component[i]}"
                    else
                        printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${optional_components_list[i]}"  "(Installed)"
                        if [[ "${optional_components_cr_list[i]}" == "bai" ]];then
                            BAI_SELECTED="Yes"
                        fi
                    fi
                fi
            done
            if [[ "$msg" ]]; then echo "$msg"; fi
            printf "\n"

            if [[ "${item_pattern}" == "Automation Decision Services" ]]; then
                echo -e "${ads_tips}"
            fi
            if [[ "${item_pattern}" == "Operational Decision Manager" ]]; then
                echo -e "\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31m You must select at least one of ODM components.\x1B[0m\n"
                echo -e "${decision_tips}"
            fi
            if [[ "${item_pattern}" == "Business Automation Application" ]]; then

                echo -e "${application_tips}"
                if [[ $DEPLOYMENT_TYPE == "starter" ]];then
                    echo -e "${application_tips_demo}"
                elif [[ $DEPLOYMENT_TYPE == "production" ]]
                then
                    echo -e "${application_tips_ent}"
                fi
            fi

            if [[ "${item_pattern}" == "FileNet Content Manager" ]]; then
                if [[ $DEPLOYMENT_TYPE == "starter" ]];then
                    echo -e "${linux_starter_tips}"
                elif [[ $DEPLOYMENT_TYPE == "production" ]]
                then
                    echo -e "${linux_production_tips}"
                fi
            fi
            # Show different tips according components select or unselect
            containsElement "(Selected)" "${choices_component[@]}"
            retVal=$?
            if [ $retVal -eq 0 ]; then
                echo -e "${tips2}"
            elif [ $selectedVal -eq 0 ]
            then
                echo -e "${tips2}"
            else
                echo -e "${tips1}"
            fi
# ##########################DEBUG############################
#         for i in "${!choices_component[@]}"; do
#             printf "%s\t%s\n" "$i" "${choices_component[$i]}"
#         done
# ##########################DEBUG############################
        }

        prompt="Enter a valid option [1 to ${#optional_components_list[@]} or ENTER]: "
        while menu && read -rp "$prompt" num && [[ "$num" ]]; do
            [[ "$num" != *[![:digit:]]* ]] &&
            (( num > 0 && num <= ${#optional_components_list[@]} )) ||
            { msg="Invalid option: $num"; continue; }
            if [[ "${item_pattern}" == "FileNet Content Manager" && "$DEPLOYMENT_TYPE" == "production" ]]; then
                case "$num" in
                "1"|"2"|"3"|"4"|"5"|"6"|"7"|"8")
                    ((num--))
                    ;;
                esac
            elif [[ "${item_pattern}" == "FileNet Content Manager" && "$DEPLOYMENT_TYPE" == "starter" ]]; then
                case "$num" in
                "1"|"2"|"3"|"4"|"5"|"6"|"7")
                    ((num--))
                    ;;
                esac
            else
                ((num--))
            fi
            containsElement "${optional_components_cr_list[num]}" "${EXISTING_OPT_COMPONENT_ARR[@]}"
            retVal=$?
            if [ $retVal -ne 0 ]; then
                [[ "${choices_component[num]}" ]] && choices_component[num]="" || choices_component[num]="(Selected)"
                if [[ "${optional_components_list[num]}" == "Decision Designer" ]]; then
                    choices_component[num+1]="(Selected)"
                fi
                if [[ $PLATFORM_SELECTED == "other" && ("${item_pattern}" == "FileNet Content Manager" || ("${item_pattern}" == "Operational Decision Manager" && "${DEPLOYMENT_TYPE}" == "production")) ]]; then
                    if [[ "${optional_components_cr_list[num]}" == "bai" && ${choices_component[num]} == "(Selected)" ]]; then
                        choices_component[num-1]="(Selected)"
                    fi
                    if [[ "${optional_components_cr_list[num]}" == "ums" && ${choices_component[num+1]} == "(Selected)" ]]; then
                        choices_component[num]="(Selected)"
                    fi
                fi
            else
                containsElement "ums" "${EXISTING_OPT_COMPONENT_ARR[@]}"
                ums_retVal=$?
                containsElement "bai" "${EXISTING_OPT_COMPONENT_ARR[@]}"
                bai_retVal=$?
                if [[ "${optional_components_cr_list[num]}" == "bai" && $ums_retVal -eq 0 ]];then
                    ((ums_check_num=num-1))
                    if [[ "${choices_component[num]}" == "(To Be Uninstalled)" ]];then
                        [[ "${choices_component[num]}" ]] && choices_component[num]="" || choices_component[num]=""
                        [[ "${choices_component[num]}" ]] && choices_component[num]="" || choices_component[ums_check_num]=""
                    else
                        [[ "${choices_component[num]}" ]] && choices_component[num]="" || choices_component[num]="(To Be Uninstalled)"
                    fi
                elif [[ "${optional_components_cr_list[num]}" == "ums" && $bai_retVal -eq 0 && ("${choices_component[num+1]}" == "" || "${choices_component[num+1]}" == "(Installed)") ]]
                then
                    [[ "${choices_component[num]}" ]] && choices_component[num]="" || choices_component[num]=""
                else
                    [[ "${choices_component[num]}" ]] && choices_component[num]="" || choices_component[num]="(To Be Uninstalled)"
                fi
            fi
        done

        # printf "\x1B[1mCOMPONENTS selected: \x1B[0m"; msg=" None"
        for i in ${!optional_components_list[@]}; do
            # [[ "${choices_component[i]}" ]] && { printf " \"%s\"" "${optional_components_list[i]}"; msg=""; }

            containsElement "${optional_components_cr_list[i]}" "${EXISTING_OPT_COMPONENT_ARR[@]}"
            retVal=$?
            if [ $retVal -ne 0 ]; then
                # [[ "${choices_component[i]}" ]] && { pattern_arr=( "${pattern_arr[@]}" "${options[i]}" ); pattern_cr_arr=( "${pattern_cr_arr[@]}" "${options_cr_val[i]}" ); msg=""; }
                if [[ "${optional_components_list[i]}" == "External Share" ]]; then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ExternalShare" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Task Manager" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "TaskManager" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Content Search Services" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ContentSearchServices" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Decision Center" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "DecisionCenter" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Rule Execution Server" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "RuleExecutionServer" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Decision Runner" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "DecisionRunner" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Decision Designer" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "DecisionDesigner" "DecisionRuntime" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Decision Runtime" ]]
                then
                    if [[ " ${optional_component_arr[*]} " =~ " DecisionDesigner " ]]; then
                       msg="";
                    else
                       [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "DecisionRuntime" ); msg=""; }
                    fi
                elif [[ "${optional_components_list[i]}" == "Content Management Interoperability Services" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ContentManagementInteroperabilityServices" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "User Management Service" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "UserManagementService" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Business Automation Insights" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "BusinessAutomationInsights" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Process Federation Server" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ProcessFederationServer" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Data Collector and Data Indexer" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "DataCollectorandDataIndexer" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Exposed Kafka Services" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ExposedKafkaServices" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Exposed OpenSearch" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ExposedOpenSearch" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Business Automation Machine Learning" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "BusinessAutomationMachineLearning" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Application Designer" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ApplicationDesigner" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Business Automation Application Data Persistence" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "BusinessAutomationApplicationDataPersistence" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "IBM Enterprise Records" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "IBMEnterpriseRecords" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "IBM Content Collector for SAP" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "IBMContentCollectorforSAP" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "Content Integration" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ContentIntegration" ); msg=""; }
                elif [[ "${optional_components_list[i]}" == "IBM Content Navigator" ]]
                then
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "IBMContentNavigator" ); msg=""; }
                else
                    [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "${optional_components_list[i]}" ); msg=""; }
                fi
                [[ "${choices_component[i]}" ]] && { optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "${optional_components_cr_list[i]}" ); msg=""; }
            else
                if [[ "${choices_component[i]}" == "(To Be Uninstalled)" ]]; then
                    pos=`indexof "${optional_component_cr_arr[i]}"`
                    if [[ "$pos" != "-1" ]]; then
                    { optional_component_cr_arr=(${optional_component_cr_arr[@]:0:$pos} ${optional_component_cr_arr[@]:$(($pos + 1))}); optional_component_arr=(${optional_component_arr[@]:0:$pos} ${optional_component_arr[@]:$(($pos + 1))}); }
                    fi
                else
                    if [[ "${optional_components_list[i]}" == "External Share" ]]; then
                        optional_component_arr=( "${optional_component_arr[@]}" "ExternalShare" )
                    elif [[ "${optional_components_list[i]}" == "Task Manager" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "TaskManager" )
                    elif [[ "${optional_components_list[i]}" == "Content Search Services" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "ContentSearchServices" )
                    elif [[ "${optional_components_list[i]}" == "Decision Center" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "DecisionCenter" )
                    elif [[ "${optional_components_list[i]}" == "Rule Execution Server" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "RuleExecutionServer" )
                    elif [[ "${optional_components_list[i]}" == "Decision Runner" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "DecisionRunner" )
                    elif [[ "${optional_components_list[i]}" == "Decision Designer" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "DecisionDesigner" )
                    elif [[ "${optional_components_list[i]}" == "Decision Runtime" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "DecisionRuntime" )
                    elif [[ "${optional_components_list[i]}" == "Content Management Interoperability Services" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "ContentManagementInteroperabilityServices" )
                    elif [[ "${optional_components_list[i]}" == "User Management Service" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "UserManagementService" )
                    elif [[ "${optional_components_list[i]}" == "Business Automation Insights" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "BusinessAutomationInsights" )
                    elif [[ "${optional_components_list[i]}" == "Process Federation Server" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "ProcessFederationServer" )
                    elif [[ "${optional_components_list[i]}" == "Data Collector and Data Indexer" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "DataCollectorandDataIndexer" )
                    elif [[ "${optional_components_list[i]}" == "Exposed Kafka Services" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "ExposedKafkaServices" )
                    elif [[ "${optional_components_list[i]}" == "Exposed OpenSearch" ]]
                    then
                        [[ "${choices_component[i]}" ]] && { optional_component_arr=( "${optional_component_arr[@]}" "ExposedOpenSearch" ); msg=""; }
                    elif [[ "${optional_components_list[i]}" == "Business Automation Machine Learning" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "BusinessAutomationMachineLearning" )
                    elif [[ "${optional_components_list[i]}" == "Application Designer" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "ApplicationDesigner" )
                    elif [[ "${optional_components_list[i]}" == "Business Automation Application Data Persistence" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "BusinessAutomationApplicationDataPersistence" )
                    elif [[ "${optional_components_list[i]}" == "IBM Enterprise Records" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "IBMEnterpriseRecords" )
                    elif [[ "${optional_components_list[i]}" == "IBM Content Collector for SAP" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "IBMContentCollectorforSAP" )
                    elif [[ "${optional_components_list[i]}" == "Content Integration" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "ContentIntegration" )
                    elif [[ "${optional_components_list[i]}" == "IBM Content Navigator" ]]
                    then
                        optional_component_arr=( "${optional_component_arr[@]}" "IBMContentNavigator" )
                    else
                        optional_component_arr=( "${optional_component_arr[@]}" "${optional_components_list[i]}" )
                    fi
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "${optional_components_cr_list[i]}" )
                fi
            fi
        done
        # echo -e "$msg"

        if [ "${#optional_component_arr[@]}" -eq "0" ]; then
            COMPONENTS_SELECTED="None"
        else
            OPT_COMPONENTS_CR_SELECTED=$( IFS=$','; echo "${optional_component_arr[*]}" )

        fi
    }
    for item_pattern in "${pattern_arr[@]}"; do
        while true; do
            case $item_pattern in
                "FileNet Content Manager")
                    # echo "select $item_pattern pattern optional components"
                    if [[ $DEPLOYMENT_TYPE == "starter" ]];then
                        optional_components_list=("Content Search Services" "Content Management Interoperability Services" "IBM Enterprise Records" "IBM Content Collector for SAP" "Business Automation Insights" "Task Manager")
                        optional_components_cr_list=("css" "cmis" "ier" "iccsap" "bai" "tm")
                    elif [[ $DEPLOYMENT_TYPE == "production" ]]
                    then
                        if [[ $PLATFORM_SELECTED == "other" ]]; then
                            optional_components_list=("Content Search Services" "Content Management Interoperability Services" "IBM Enterprise Records" "IBM Content Collector for SAP" "User Management Service" "Business Automation Insights" "Task Manager")
                            optional_components_cr_list=("css" "cmis" "ier" "iccsap" "ums" "bai" "tm")
                        else
                            optional_components_list=("Content Search Services" "Content Management Interoperability Services" "IBM Enterprise Records" "IBM Content Collector for SAP" "Business Automation Insights" "Task Manager")
                            optional_components_cr_list=("css" "cmis" "ier" "iccsap" "bai" "tm")
                        fi
                    fi
                    show_optional_components
                    if [[ $PLATFORM_SELECTED == "other" ]]; then
                        containsElement "bai" "${optional_component_cr_arr[@]}"
                        retVal=$?
                        if [[ $retVal -eq 0 ]]; then
                            optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ums" )
                            optional_component_arr=( "${optional_component_arr[@]}" "UserManagementService" )
                        fi
                    fi
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "Document Processing Engine")
                    # echo "Without optional components for $item_pattern pattern."
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "Operational Decision Manager")
                    # echo "select $item_pattern pattern optional components"
                    if [[ "${DEPLOYMENT_TYPE}" == "starter" ]]; then
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "decisionCenter" )
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "decisionServerRuntime" )
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "decisionRunner" )
                        optional_components_list=("Business Automation Insights")
                        optional_components_cr_list=("bai")
                    else
                        if [[ $PLATFORM_SELECTED == "other" ]]; then
                            optional_components_list=("Decision Center" "Rule Execution Server" "Decision Runner" "User Management Service" "Business Automation Insights")
                            optional_components_cr_list=("decisionCenter" "decisionServerRuntime" "decisionRunner" "ums" "bai")
                        else
                            optional_components_list=("Decision Center" "Rule Execution Server" "Decision Runner" "Business Automation Insights")
                            optional_components_cr_list=("decisionCenter" "decisionServerRuntime" "decisionRunner" "bai")
                        fi
                    fi
                        show_optional_components
                        if [[ $PLATFORM_SELECTED == "other" ]]; then
                            containsElement "bai" "${optional_component_cr_arr[@]}"
                            retVal=$?
                            if [[ $retVal -eq 0 ]]; then
                                optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ums" )
                                optional_component_arr=( "${optional_component_arr[@]}" "UserManagementService" )
                            fi
                        fi
                        optional_components_list=()
                        optional_components_cr_list=()
                    break
                    ;;
                "Automation Decision Services")
                    # echo "select $item_pattern pattern optional components"
                    if [[ "${DEPLOYMENT_TYPE}" == "starter" ]]; then
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ads_designer" )
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ads_runtime" )
                        optional_components_list=("Business Automation Insights")
                        optional_components_cr_list=("bai")
                        show_optional_components
                        optional_components_list=()
                        optional_components_cr_list=()
                    else
                        optional_components_list=("Business Automation Insights" "Decision Designer" "Decision Runtime")
                        optional_components_cr_list=("bai" "ads_designer" "ads_runtime")
                        show_optional_components
                        optional_components_list=()
                        optional_components_cr_list=()
                    fi
                    break
                    ;;
                "Business Automation Workflow")
                    # The logic for BAW only in 4Q
                    if [[ $DEPLOYMENT_TYPE == "starter" && $retVal_baw -eq 0 ]]; then
                        optional_components_list=("Business Automation Insights")
                        optional_components_cr_list=("bai")
                        show_optional_components
                    fi
                    if [[ $DEPLOYMENT_TYPE == "production" && $retVal_baw -eq 0 ]]; then
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "bai" )
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ae_data_persistence" )
                    fi
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "cmis" )
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "(a) Workflow Authoring")
                    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                        optional_components_list=("Business Automation Insights" "Data Collector and Data Indexer" "Exposed Kafka Services")
                        optional_components_cr_list=("bai" "pfs" "kafka")
                        show_optional_components
                    fi
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "cmis" )
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "baw_authoring" )
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "(b) Workflow Runtime")
                    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                        optional_components_list=("Business Automation Insights" "Exposed Kafka Services" "Exposed OpenSearch")
                        optional_components_cr_list=("bai" "kafka" "opensearch")
                        show_optional_components
                    fi
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "cmis" )
                    # optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "bai" )
                    # if [[ ! ((" ${pattern_cr_arr[@]} " =~ "workflow-runtime" && "${#pattern_cr_arr[@]}" -eq "2") || (" ${pattern_cr_arr[@]} " =~ "workflow-runtime" && " ${pattern_cr_arr[@]} " =~ "workstreams" && "${#pattern_cr_arr[@]}" -eq "4")) ]]; then
                    #     optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ae_data_persistence" )
                    # fi
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "Business Automation Workflow Authoring and Automation Workstream Services")
                    if [[ $DEPLOYMENT_TYPE == "starter" ]]; then
                        optional_components_list=("Case" "Content Integration" "Workstreams" "Data Collector and Data Indexer" "Business Automation Insights" "Business Automation Machine Learning")
                        optional_components_cr_list=("case" "content_integration" "workstreams" "pfs" "bai" "baml")
                        show_optional_components
                    # elif [[ $DEPLOYMENT_TYPE == "production" ]]; then
                    #     optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "bai" )
                    #     optional_component_arr=( "${optional_component_arr[@]}" "BusinessAutomationInsights" )
                    fi
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "cmis" )
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "baw_authoring" )
                    if [[ $DEPLOYMENT_TYPE == "starter" ]]; then
                        containsElement "baml" "${optional_component_cr_arr[@]}"
                        retVal=$?
                        if [[ $retVal -eq 0 ]]; then
                            optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "bai" "pfs")
                            optional_component_arr=( "${optional_component_arr[@]}" "BusinessAutomationInsights" "ProcessFederationServer")
                        fi
                    fi
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "Automation Workstream Services")
                    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                        optional_components_list=("Exposed Kafka Services" "Exposed OpenSearch")
                        optional_components_cr_list=("kafka" "opensearch")
                        show_optional_components
                    fi
                    # echo "Without optional components for $item_pattern pattern."
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "cmis" )
                    # if [[ ! ((" ${pattern_cr_arr[@]} " =~ "workstreams" && "${#pattern_cr_arr[@]}" -eq "1") || (" ${pattern_cr_arr[@]} " =~ "workflow-runtime" && " ${pattern_cr_arr[@]} " =~ "workstreams" && "${#pattern_cr_arr[@]}" -eq "4")) ]]; then
                    #     optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ae_data_persistence" )
                    # fi
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "Business Automation Application")
                    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                        # echo "select $item_pattern pattern optional components"
                        optional_components_list=("Application Designer")
                        optional_components_cr_list=("app_designer")
                        show_optional_components
                        optional_components_list=()
                        optional_components_cr_list=()
                    else
                        if [[ ! (" ${pattern_cr_arr[@]} " =~ "content" || " ${pattern_cr_arr[@]} " =~ "document_processing" || " ${pattern_cr_arr[@]} " =~ "workflow") ]]; then
                            optional_components_list=("IBM Content Navigator")
                            optional_components_cr_list=("ban")
                            show_optional_components
                        fi
                        optional_components_list=()
                        optional_components_cr_list=()
                    fi
                    break
                    ;;
                "Automation Digital Worker")
                    optional_components_list=("Business Automation Insights")
                    optional_components_cr_list=("bai")
                    show_optional_components
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "IBM Automation Document Processing")
                    if [[ $DEPLOYMENT_TYPE == "starter" ]]; then
                        optional_components_list=("Content Search Services" "Content Management Interoperability Services" "Task Manager")
                        optional_components_cr_list=("css" "cmis" "tm")
                        show_optional_components
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "document_processing_designer" )
                    fi
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "(a) Development Environment")
                    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                        if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow" || " ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams" || " ${pattern_cr_arr[@]} " =~ "workflow" || " ${pattern_cr_arr[@]} " =~ "workstreams" ]]; then
                            optional_components_list=("Content Search Services" "Task Manager")
                            optional_components_cr_list=("css" "tm")
                        else
                            optional_components_list=("Content Search Services" "Content Management Interoperability Services" "Task Manager")
                            optional_components_cr_list=("css" "cmis" "tm")
                        fi
                        show_optional_components
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ae_data_persistence" )
                    fi
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "document_processing_designer" )
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "(b) Runtime Environment")
                    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                        if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow" || " ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams" || " ${pattern_cr_arr[@]} " =~ "workflow" || " ${pattern_cr_arr[@]} " =~ "workstreams" ]]; then
                            optional_components_list=("Content Search Services")
                            optional_components_cr_list=("css")
                        else
                            optional_components_list=("Content Search Services" "Content Management Interoperability Services" "Task Manager")
                            optional_components_cr_list=("css" "cmis" "tm")
                        fi
                        show_optional_components
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ae_data_persistence" )
                    fi
                    optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "document_processing_runtime" )
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
                "Workflow Process Service Authoring")
                    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                        optional_components_list=("Business Automation Insights" "Data Collector and Data Indexer" "Exposed Kafka Services")
                        optional_components_cr_list=("bai" "pfs" "kafka")
                        show_optional_components
                        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "wfps_authoring" )
                    fi
                    optional_components_list=()
                    optional_components_cr_list=()
                    break
                    ;;
            esac
        done
    done

    if [[ "$AE_DATA_PERSISTENCE_ENABLE" == "Yes" ]]; then
        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ae_data_persistence" )
    fi

    if [[ "$AUTOMATION_SERVICE_ENABLE" == "Yes" ]]; then
        optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "auto_service" )
        foundation_component_arr=( "${foundation_component_arr[@]}" "UMS" )
        # optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ums" ) # remove it when UMS pattern aware auto_service
    fi

    OPT_COMPONENTS_CR_SELECTED=($(echo "${optional_component_cr_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
    OPTIONAL_COMPONENT_DELETE_LIST=($(echo "${OPT_COMPONENTS_CR_SELECTED[@]}" "${OPTIONAL_COMPONENT_FULL_ARR[@]}" | tr ' ' '\n' | sort | uniq -u))
    KEEP_COMPOMENTS=($(echo ${FOUNDATION_CR_SELECTED_LOWCASE[@]} ${OPTIONAL_COMPONENT_DELETE_LIST[@]} | tr ' ' '\n' | sort | uniq -d | uniq))
    OPT_COMPONENTS_SELECTED=($(echo "${optional_component_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))

    # Will an external LDAP be used as part of the configuration?
    containsElement "es" "${OPT_COMPONENTS_CR_SELECTED[@]}"
    retVal_ext_ldap=$?
    if [[ $retVal_ext_ldap -eq 0 && "${DEPLOYMENT_TYPE}" == "production" ]];then
        set_external_ldap
    fi
}

function get_local_registry_password(){
    printf "\n"
    printf "\x1B[1mEnter the password for your docker registry: \x1B[0m"
    local_registry_password=""
    while [[ $local_registry_password == "" ]];
    do
       read -rsp "" local_registry_password
       if [ -z "$local_registry_password" ]; then
       echo -e "\x1B[1;31mEnter a valid password\x1B[0m"
       fi
    done
    export LOCAL_REGISTRY_PWD=${local_registry_password}
    printf "\n"
}

function get_local_registry_password_double(){
    pwdconfirmed=1
    pwd=""
    pwd2=""
        while [ $pwdconfirmed -ne 0 ] # While pwd is not yet received and confirmed (i.e. entered teh same time twice)
        do
                printf "\n"
                while [[ $pwd == '' ]] # While pwd is empty...
                do
                        printf "\x1B[1mEnter the password for your docker registry: \x1B[0m"
                        read -rsp " " pwd
                done

                printf "\n"
                while [[ $pwd2 == '' ]]  # While pwd is empty...
                do
                        printf "\x1B[1mEnter the password again: \x1B[0m"
                        read -rsp " " pwd2
                done

            if [ "$pwd" == "$pwd2" ]; then
                   pwdconfirmed=0
                else
                   printf "\n"
                   echo -e "\x1B[1;31mThe passwords do not match. Try again.\x1B[0m"
                   unset pwd
                   unset pwd2
                fi
        done

        printf "\n"

        export LOCAL_REGISTRY_PWD="${pwd}"
}



function get_entitlement_registry(){

    docker_image_exists() {
    local image_full_name="$1"; shift
    local wait_time="${1:-5}"
    local search_term='Pulling|Copying|is up to date|already exists|not found|unable to pull image|no pull access'
    if [[ $OCP_VERSION == "3.11" ]];then
        local result=$((timeout --preserve-status "$wait_time" docker 2>&1 pull "$image_full_name" &) | grep -v 'Pulling repository' | egrep -o "$search_term")

    elif [[ $OCP_VERSION == "4.4OrLater" ]]
    then
        local result=$((timeout --preserve-status "$wait_time" podman 2>&1 pull "$image_full_name" &) | grep -v 'Pulling repository' | egrep -o "$search_term")

    fi
    test "$result" || { echo "Timed out too soon. Try using a wait_time greater than $wait_time..."; return 1 ;}
    echo $result | grep -vq 'not found'
    }

    # For Entitlement Registry key
    entitlement_key=""
    printf "\n"
    printf "\n"
    printf "\x1B[1;31mFollow the instructions on how to get your Entitlement Key: \n\x1B[0m"
    printf "\x1B[1;31mhttps://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/$CP4BA_RELEASE_BASE?topic=deployment-getting-access-images-from-public-entitled-registry\n\x1B[0m"
    printf "\n"
    while true; do
        printf "\x1B[1mDo you have a Cloud Pak for Business Automation Entitlement Registry key (Yes/No, default: Yes): \x1B[0m"
        read -rp "" ans

        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES"|"")
            use_entitlement="yes"
            if [[ "$SCRIPT_MODE" == "dev" || "$SCRIPT_MODE" == "review" || "$SCRIPT_MODE" == "OLM" ]]
            then
                DOCKER_REG_SERVER="cp.stg.icr.io"
            else
                DOCKER_REG_SERVER="cp.icr.io"
            fi
            break
            ;;
        "n"|"N"|"no"|"No"|"NO")
            use_entitlement="no"
            DOCKER_REG_KEY="None"
            if [[ "$PLATFORM_SELECTED" == "ROKS" || "$PLATFORM_SELECTED" == "OCP" ]]; then
                printf "\n"
                printf "\x1B[1;31m\"${PLATFORM_SELECTED}\" only supports the Entitlement Registry, exiting...\n\x1B[0m"
                exit 1
            else
                break
            fi
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}

function create_secret_entitlement_registry(){
    printf "\x1B[1mCreating docker-registry secret for Entitlement Registry key...\n\x1B[0m"
# Create docker-registry secret for Entitlement Registry Key
    ${CLI_CMD} delete secret "$DOCKER_RES_SECRET_NAME" >/dev/null 2>&1
    CREATE_SECRET_CMD="${CLI_CMD} create secret docker-registry $DOCKER_RES_SECRET_NAME --docker-server=$DOCKER_REG_SERVER --docker-username=$DOCKER_REG_USER --docker-password=$DOCKER_REG_KEY --docker-email=ecmtest@ibm.com"
    if $CREATE_SECRET_CMD ; then
        echo -e "\x1B[1mDone\x1B[0m"
    else
        echo -e "\x1B[1mFailed\x1B[0m"
    fi
}

function get_local_registry_server(){
    # For internal/external Registry Server
    printf "\n"
    if [[ "${REGISTRY_TYPE}" == "internal" && ("${OCP_VERSION}" == "4.4OrLater") ]];then
        #This is required for docker/podman login validation.
        printf "\x1B[1mEnter the public image registry or route (e.g., default-route-openshift-image-registry.apps.<hostname>). \n\x1B[0m"
        printf "\x1B[1mThis is required for docker/podman login validation: \x1B[0m"
        local_public_registry_server=""
        while [[ $local_public_registry_server == "" ]]
        do
            read -rp "" local_public_registry_server
            if [ -z "$local_public_registry_server" ]; then
            echo -e "\x1B[1;31mEnter a valid service name or the URL for the docker registry.\x1B[0m"
            fi
        done
    fi

    if [[ "${OCP_VERSION}" == "3.11" && "${REGISTRY_TYPE}" == "internal" ]];then
        printf "\x1B[1mEnter the OCP docker registry service name, for example: docker-registry.default.svc:5000/<project-name>: \x1B[0m"
    elif [[ "${REGISTRY_TYPE}" == "internal" && "${OCP_VERSION}" == "4.4OrLater" ]]
    then
        printf "\n"
        printf "\x1B[1mEnter the local image registry (e.g., image-registry.openshift-image-registry.svc:5000/<project>)\n\x1B[0m"
        printf "\x1B[1mThis is required to pull container images and Kubernetes secret creation: \x1B[0m"
        builtin_dockercfg_secrect_name=$(${CLI_CMD} get secret | grep default-dockercfg | awk '{print $1}')
        if [ -z "$builtin_dockercfg_secrect_name" ]; then
            DOCKER_RES_SECRET_NAME="ibm-entitlement-key"
        else
            DOCKER_RES_SECRET_NAME=$builtin_dockercfg_secrect_name
        fi
    elif [[ "${REGISTRY_TYPE}" == "external" || $PLATFORM_SELECTED == "other" ]]
    then
        printf "\x1B[1mEnter the URL to the docker registry, for example: abc.xyz.com: \x1B[0m"
    fi
    local_registry_server=""
    while [[ $local_registry_server == "" ]]
    do
        read -rp "" local_registry_server
        if [ -z "$local_registry_server" ]; then
        echo -e "\x1B[1;31mEnter a valid service name or the URL for the docker registry.\x1B[0m"
        fi
    done
    LOCAL_REGISTRY_SERVER=${local_registry_server}
    # convert docker-registry.default.svc:5000/project-name
    # to docker-registry.default.svc:5000\/project-name
    OIFS=$IFS
    IFS='/' read -r -a docker_reg_url_array <<< "$local_registry_server"
    delim=""
    joined=""
    for item in "${docker_reg_url_array[@]}"; do
            joined="$joined$delim$item"
            delim="\/"
    done
    IFS=$OIFS
    CONVERT_LOCAL_REGISTRY_SERVER=${joined}
}

function get_local_registry_user(){
    # For Local Registry User
    printf "\n"
    printf "\x1B[1mEnter the user name for your docker registry: \x1B[0m"
    local_registry_user=""
    while [[ $local_registry_user == "" ]]
    do
       read -rp "" local_registry_user
       if [ -z "$local_registry_user" ]; then
       echo -e "\x1B[1;31mEnter a valid user name.\x1B[0m"
       fi
    done
    export LOCAL_REGISTRY_USER=${local_registry_user}
}

function get_storage_class_name(){

    # For dynamic storage classname
    storage_class_name=""
    block_storage_class_name=""
    sc_slow_file_storage_classname=""
    sc_medium_file_storage_classname=""
    sc_fast_file_storage_classname=""

    printf "\n"
    if [[ $DEPLOYMENT_TYPE == "starter" && ($PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "other")]] ;
    then
        printf "\x1B[1mTo provision the persistent volumes and volume claims, enter the file storage classname(RWX): \x1B[0m"

        while [[ $storage_class_name == "" ]]
        do
            read -rp "" storage_class_name
            if [ -z "$storage_class_name" ]; then
               echo -e "\x1B[1;31mEnter a valid file storage classname(RWX)\x1B[0m"
            fi
        done
        printf "\x1B[1mTo provision the persistent volumes and volume claims, enter the block storage classname(RWO): \x1B[0m"
        if [[ $PLATFORM_SELECTED == "OCP" ]]; then
        while [[ $block_storage_class_name == "" ]]
        do
            read -rp "" block_storage_class_name
            if [ -z "$block_storage_class_name" ]; then
               echo -e "\x1B[1;31mEnter a valid block storage classname(RWO)\x1B[0m"
            fi
        done
        fi
    elif [[ ($DEPLOYMENT_TYPE == "production" && ($PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "other")) || $PLATFORM_SELECTED == "ROKS" ]]
    then
        printf "\x1B[1mTo provision the persistent volumes and volume claims\n\x1B[0m"
        while [[ $sc_slow_file_storage_classname == "" ]] # While get slow storage clase name
        do
            printf "\x1B[1mEnter the file storage classname for slow storage(RWX): \x1B[0m"
            read -rp "" sc_slow_file_storage_classname
            if [ -z "$sc_slow_file_storage_classname" ]; then
               echo -e "\x1B[1;31mEnter a valid file storage classname(RWX)\x1B[0m"
            fi
        done

        while [[ $sc_medium_file_storage_classname == "" ]] # While get medium storage clase name
        do
            printf "\x1B[1mEnter the file storage classname for medium storage(RWX): \x1B[0m"
            read -rp "" sc_medium_file_storage_classname
            if [ -z "$sc_medium_file_storage_classname" ]; then
               echo -e "\x1B[1;31mEnter a valid file storage classname(RWX)\x1B[0m"
            fi
        done

        while [[ $sc_fast_file_storage_classname == "" ]] # While get fast storage clase name
        do
            printf "\x1B[1mEnter the file storage classname for fast storage(RWX): \x1B[0m"
            read -rp "" sc_fast_file_storage_classname
            if [ -z "$sc_fast_file_storage_classname" ]; then
               echo -e "\x1B[1;31mEnter a valid file storage classname(RWX)\x1B[0m"
            fi
        done
        if [[ $PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS" ]]; then
        while [[ $block_storage_class_name == "" ]] # While get block storage clase name
        do
            printf "\x1B[1mEnter the block storage classname for Zen(RWO): \x1B[0m"
            read -rp "" block_storage_class_name
            if [ -z "$block_storage_class_name" ]; then
               echo -e "\x1B[1;31mEnter a valid block storage classname(RWO)\x1B[0m"
            fi
        done
        fi
    fi
    STORAGE_CLASS_NAME=${storage_class_name}
    SLOW_STORAGE_CLASS_NAME=${sc_slow_file_storage_classname}
    MEDIUM_STORAGE_CLASS_NAME=${sc_medium_file_storage_classname}
    FAST_STORAGE_CLASS_NAME=${sc_fast_file_storage_classname}
    BLOCK_STORAGE_CLASS_NAME=${block_storage_class_name}
}

function create_secret_local_registry(){
    echo -e "\x1B[1mCreating the secret based on the local docker registry information...\x1B[0m"
    # Create docker-registry secret for local Registry Key
    # echo -e "Create docker-registry secret for Local Registry...\n"
    if [[ $LOCAL_REGISTRY_SERVER == docker-registry* || $LOCAL_REGISTRY_SERVER == image-registry.openshift-image-registry* ]] ;
    then
        builtin_dockercfg_secrect_name=$(${CLI_CMD} get secret | grep default-dockercfg | awk '{print $1}')
        DOCKER_RES_SECRET_NAME=$builtin_dockercfg_secrect_name
        # CREATE_SECRET_CMD="${CLI_CMD} create secret docker-registry $DOCKER_RES_SECRET_NAME --docker-server=$LOCAL_REGISTRY_SERVER --docker-username=$LOCAL_REGISTRY_USER --docker-password=$(${CLI_CMD} whoami -t) --docker-email=ecmtest@ibm.com"
    else
        ${CLI_CMD} delete secret "$DOCKER_RES_SECRET_NAME" >/dev/null 2>&1
        CREATE_SECRET_CMD="${CLI_CMD} create secret docker-registry $DOCKER_RES_SECRET_NAME --docker-server=$LOCAL_REGISTRY_SERVER --docker-username=$LOCAL_REGISTRY_USER --docker-password=$LOCAL_REGISTRY_PWD --docker-email=ecmtest@ibm.com"
        if $CREATE_SECRET_CMD ; then
            echo -e "\x1B[1mDone\x1B[0m"
        else
            echo -e "\x1B[1;31mFailed\x1B[0m"
        fi
    fi
}

function verify_local_registry_password(){
    # require to preload image for CP4A image and ldap/db2 image for demo
    printf "\n"
    while true; do
        printf "\x1B[1mHave you pushed the images to the local registry using 'loadimages.sh' (CP4A images) (Yes/No)? \x1B[0m"
        # printf "\x1B[1mand 'loadPrereqImages.sh' (Db2 and OpenLDAP for demo) scripts (Yes/No)? \x1B[0m"
        read -rp "" ans
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES")
            PRE_LOADED_IMAGE="Yes"
            break
            ;;
        "n"|"N"|"no"|"No"|"NO")
            echo -e "\x1B[1;31mPull the images to the local images to proceed.\n\x1B[0m"
            exit 1
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done

    # Select which type of image registry to use.
    if [[ "${PLATFORM_SELECTED}" == "OCP" ]]; then
        printf "\n"
        echo -e "\x1B[1mSelect the type of image registry to use: \x1B[0m"
        COLUMNS=12
        options=("Other ( External image registry: abc.xyz.com )")

        PS3='Enter a valid option [1 to 1]: '
        select opt in "${options[@]}"
        do
            case $opt in
                "Openshift Container Platform (OCP) - Internal image registry")
                    REGISTRY_TYPE="internal"
                    break
                    ;;
                "Other ( External image registry: abc.xyz.com )")
                    REGISTRY_TYPE="external"
                    break
                    ;;
                *) echo "invalid option $REPLY";;
            esac
        done
    else
        REGISTRY_TYPE="external"
    fi
    get_local_registry_server
}
function select_installation_type(){
    COLUMNS=12
    echo -e "\x1B[1mIs this a new installation or an existing installation?\x1B[0m"
    options=("New" "Existing")
    PS3='Enter a valid option [1 to 2]: '
    select opt in "${options[@]}"
    do
        case $opt in
            "New")
                INSTALLATION_TYPE="new"
                break
                ;;
            "Existing")
                INSTALLATION_TYPE="existing"
                mkdir -p $TEMP_FOLDER >/dev/null 2>&1
                mkdir -p $BAK_FOLDER >/dev/null 2>&1
                mkdir -p $FINAL_CR_FOLDER >/dev/null 2>&1
                get_existing_pattern_name
                break
                ;;
            *) echo "invalid option $REPLY";;
        esac
    done
    if [[ "${INSTALLATION_TYPE}" == "new" ]]; then
        clean_up_temp_file
        rm -rf $BAK_FOLDER >/dev/null 2>&1
        rm -rf $FINAL_CR_FOLDER >/dev/null 2>&1

        mkdir -p $TEMP_FOLDER >/dev/null 2>&1
        mkdir -p $BAK_FOLDER >/dev/null 2>&1
        mkdir -p $FINAL_CR_FOLDER >/dev/null 2>&1
    fi
}

function select_iam_default_admin(){
    printf "\n"
    while true; do
        echo -e "\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1;31mIf you are unable to use [cpadmin] as the default IAM admin user due to it having the same username in your LDAP Directory, you need to change the Cloud Pak administrator username. See: \"https://www.ibm.com/docs/en/cpfs?topic=configurations-changing-cloud-pak-administrator-access-credentials#user-name\"\x1B[0m"
        printf "\x1B[1mDo you want to use the default IAM admin user: [cpadmin] (Yes/No, default: Yes): \x1B[0m"
        read -rp "" ans
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES"|"")
            USE_DEFAULT_IAM_ADMIN="Yes"
            break
            ;;
        "n"|"N"|"no"|"No"|"NO")
            USE_DEFAULT_IAM_ADMIN="No"
            while [[ $NON_DEFAULT_IAM_ADMIN == "" ]];
            do
                printf "\n"
                echo -e "\x1B[1mWhat is the non default IAM admin user you renamed?\x1B[0m"
                read -p "Enter the admin username: " NON_DEFAULT_IAM_ADMIN

                if [ -z "$NON_DEFAULT_IAM_ADMIN" ]; then
                    echo -e "\x1B[1;31mEnter a valid admin username, username can not be blank\x1B[0m"
                    NON_DEFAULT_IAM_ADMIN=""
                elif [[ "$NON_DEFAULT_IAM_ADMIN" == "cpadmin" ]]; then
                    echo -e "\x1B[1;31mEnter a valid admin username, username should not be 'cpadmin'\x1B[0m"
                    NON_DEFAULT_IAM_ADMIN=""
                fi
            done
            break
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}

function select_profile_type(){
    printf "\n"
    COLUMNS=12
    if [ -z $OPENSEARCH_CATALOG_NS ]; then
        echo -e "\x1B[1mSelect the deployment profile. Refer to the documentation in CP4BA Docs for details on profile.\x1B[0m"
    else
        echo -e "\x1B[1mSelect the deployment profile for Opensearch. Refer to the documentation in Opensearch Docs for details on the profile.\x1B[0m"
    fi
    options=("small" "medium" "large")
    if [ -z "$existing_profile_type" ]; then
      if [[ $PROFILE_TYPE == "" ]]; then
        PS3='Enter a valid option [1 to 3]: '
        select opt in "${options[@]}"
        do
            case $opt in
                "small")
                    PROFILE_TYPE="small"
                    break
                    ;;
                "medium")
                    PROFILE_TYPE="medium"
                    break
                    ;;
                "large")
                    PROFILE_TYPE="large"
                    break
                    ;;
                *) echo "invalid option $REPLY";;
            esac
        done
      fi
    else
        options_var=("small" "medium" "large")
        for i in ${!options_var[@]}; do
            if [[ "${options_var[i]}" == "$existing_profile_type" ]]; then
                printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "(Selected)"
            else
                printf "%1d) %s\n" $((i+1)) "${options[i]}"
            fi
        done
        echo -e "\x1B[1;31mExisting profile size type found in CR: \"$existing_profile_type\"\x1B[0m"
        # echo -e "\x1B[1;31mDo not need to select again.\n\x1B[0m"
        prompt_press_any_key_to_continue
    fi
}

function select_ocp_olm(){
    printf "\n"
    while true; do
        printf "\x1B[1mAre you using the OCP Catalog (OLM) to perform this install? (Yes/No, default: No) \x1B[0m"

        read -rp "" ans
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES")
            SCRIPT_MODE="OLM"
            break
            ;;
        "n"|"N"|"no"|"No"|"NO"|"")
            break
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}


function select_deployment_type(){
    printf "\n"

    if [[ "$SCRIPT_MODE" == "OLM" ]]
    then
        DEPLOYMENT_TYPE="production"
        echo -e "An enterprise deployment will be prepared for the OCP Catalog."
    else
        echo -e "\x1B[1mWhat type of deployment is being performed?\x1B[0m"

        COLUMNS=12
        options=("Starter" "Production")
        if [ -z "$existing_deployment_type" ]; then
          if [[ $CP4BA_DEPLOYMENT_TYPE != ""  ]]; then
            DEPLOYMENT_TYPE=$CP4BA_DEPLOYMENT_TYPE
          else
            PS3='Enter a valid option [1 to 2]: '
            select opt in "${options[@]}"
            do
                case $opt in
                    "Starter")
                        DEPLOYMENT_TYPE="starter"
                        break
                        ;;
                    "Production")
                        DEPLOYMENT_TYPE="production"
                        break
                        ;;
                    *) echo "invalid option $REPLY";;
                esac
            done
          fi
        else
            options_var=("Starter" "Production")
            for i in ${!options_var[@]}; do
                if [[ "${options_var[i]}" == "$existing_deployment_type" ]]; then
                    printf "%1d) %s \x1B[1m%s\x1B[0m\n" $((i+1)) "${options[i]}"  "(Selected)"
                else
                    printf "%1d) %s\n" $((i+1)) "${options[i]}"
                fi
            done
            echo -e "\x1B[1;31mExisting deployment type found in CR: \"$existing_deployment_type\"\x1B[0m"
            # echo -e "\x1B[1;31mDo not need to select again.\n\x1B[0m"
            prompt_press_any_key_to_continue
        fi
    fi
}

function enable_ae_data_persistence_workflow_authoring(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${WORKFLOW_AUTHOR_PATTERN_FILE_BAK} ${WORKFLOW_AUTHOR_PATTERN_FILE_TMP}
        content_start="$(grep -n "## object store for AEOS" ${WORKFLOW_AUTHOR_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        content_stop="$(tail -n +$content_start < ${WORKFLOW_AUTHOR_PATTERN_FILE_TMP} | grep -n "dc_hadr_max_retries_for_client_reroute: 3" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start - 1))
        vi ${WORKFLOW_AUTHOR_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/    # /    ' -c ':wq' >/dev/null 2>&1
        ###########
        content_start="$(grep -n "## Configuration for the application engine object store" ${WORKFLOW_AUTHOR_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        content_stop="$(tail -n +$content_start < ${WORKFLOW_AUTHOR_PATTERN_FILE_TMP} | grep -n "\"<Required>\" # user name and group name for object store admin" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start + 1))
        vi ${WORKFLOW_AUTHOR_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/      # /      ' -c ':wq' >/dev/null 2>&1

        ${COPY_CMD} -rf ${WORKFLOW_AUTHOR_PATTERN_FILE_TMP} ${WORKFLOW_AUTHOR_PATTERN_FILE_BAK}
    fi
}

function enable_ae_data_persistence_baa(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${APPLICATION_PATTERN_FILE_BAK} ${APPLICATION_PATTERN_FILE_TMP}
        content_start="$(grep -n "The beginning section of database configuration for CP4A" ${APPLICATION_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        content_stop="$(tail -n +$content_start < ${APPLICATION_PATTERN_FILE_TMP} | grep -n "dc_os_xa_datasource_name: \"AEOSXA\"" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start + 1))
        vi ${APPLICATION_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/  # /  ' -c ':wq' >/dev/null 2>&1
        ${COPY_CMD} -rf ${APPLICATION_PATTERN_FILE_TMP} ${APPLICATION_PATTERN_FILE_BAK}
    fi
}

function select_ldap_type_for_wfps_authoring(){
    info "LDAP configuration is not required for the IBM Workflow Process Service Authoring, but if you want to login with LDAP user, select Yes. If you select No, you can manually add the LDAP connection after installation by referring to https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/$CP4BA_RELEASE_BASE?topic=cpbaf-business-automation-studio"
    while true; do
        printf "\x1B[1mDo you want use the LDAP for the IBM Workflow Process Service Authoring? (Yes/No): \x1B[0m"
        read -rp "" ans
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES")
            LDAP_WFPS_AUTHORING="Yes"
            break
            ;;
        "n"|"N"|"no"|"No"|"NO")
            LDAP_WFPS_AUTHORING="No"
            break
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}

function select_upgrade_mode(){
    UPGRADE_MODE=""
    while [[ $UPGRADE_MODE == "" ]]; do
        printf "\n"
        options=("Cluster-scoped to Cluster-scoped" "Cluster-scoped to Namespace-scoped")
        tips=("This migration mode will first isolate the shared 3.x version of foundational services on the cluster, then install a new instance of foundational services version 4.6.2. Any data from the original foundational services will be copied to the new instance." "This migration mode will migrate a single shared instance of foundational services that is shared by all IBM Cloud Paks in the entire cluster, all of its core foundational services operators to the cluster-scope namespace will be upgraded to v4.6.2 without creating new instances.")
        pros_tips=("The namespace-scoped foundational services for each Cloud Pak or CP4BA component." "The fewer resource required for a single shared instance of foundational services.")
        cons_tips=("The more resource required for each namespace-scoped instance of IBM Cloud Pak foundational services." "It is not flexible for all Cloud Pak or CP4BA component to share same one instance of IBM Cloud Pak foundational services.")

        if [[ $RUNTIME_MODE == "upgradeOperator" ]]; then
            echo -e "\x1B[1mWhich migration mode for the IBM Cloud Pak foundational services are you migrating to? \x1B[0m"
        elif [[ $RUNTIME_MODE == "upgradeDeployment" ]]; then
            echo -e "\x1B[1mWhich migration mode for the IBM Cloud Pak foundational services are you migrating to when run [upgradeOperator]? \x1B[0m"
        fi
        echo -e "${YELLOW_TEXT}1) Cluster-scoped to Cluster-scoped${RESET_TEXT}"
        if [[ $RUNTIME_MODE == "upgradeOperator" ]]; then
            echo -e "   ${YELLOW_TEXT}[Tips]${RESET_TEXT}: ${tips[1]}"
            echo -e "   ${GREEN_TEXT}[Pros]${RESET_TEXT}: ${pros_tips[1]}"
            echo -e "   ${RED_TEXT}[Cons]${RESET_TEXT}: ${cons_tips[1]}"
        fi
        echo -e "${YELLOW_TEXT}2) Cluster-scoped to Namespace-scoped (Recommended)${RESET_TEXT}"
        if [[ $RUNTIME_MODE == "upgradeOperator" ]]; then
            echo -e "   ${YELLOW_TEXT}[Tips]${RESET_TEXT}: ${tips[0]}"
            echo -e "   ${GREEN_TEXT}[Pros]${RESET_TEXT}: ${pros_tips[0]}"
            echo -e "   ${RED_TEXT}[Cons]${RESET_TEXT}: ${cons_tips[0]}"
        fi

        read -p "Enter your choice [1 or 2]: " choice
        case $choice in
            1)
                UPGRADE_MODE="shared2shared"
                ;;
            2)
                UPGRADE_MODE="shared2dedicated"
                ;;
            *)
                echo "Invalid choice. Select either 1 or 2."
                sleep 2
                ;;
        esac
        if [[ ! -z $UPGRADE_MODE ]]; then
            echo "You selected: ${options[$choice-1]}"
            printf "\n"
            sleep 2
        fi
    done
}

function generate_sample_network_policies(){
    printf "\n"
    echo ""
    while true; do
        printf "\x1B[1mDo you want to generate the network policy templates for this CP4BA deployment?\x1B[0m ${YELLOW_TEXT}(Notes: Starting from $CP4BA_RELEASE_BASE, the CP4BA operators no longer install network policies automatically. If you want the operators to generate network policies from a set of templates, select Yes. You can install the network policies by running a script after the CP4BA Deployment is installed. If you select No, then no network policies will be generated.)${RESET_TEXT} (Yes/No, default: No):" 
        read -rp "" ans
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES")
            GENERATE_SAMPLE_NETWORK_POLICIES="true"
            break
            ;;
        "n"|"N"|"no"|"No"|"NO"|"")
            GENERATE_SAMPLE_NETWORK_POLICIES="false"
            break
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done
}

function select_fips_enable(){
    select_project
    all_fips_enabled_flag=$(${CLI_CMD} get configmap cp4ba-fips-status --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath={.data.all-fips-enabled})
    if [ -z $all_fips_enabled_flag ]; then
        if [[ ("$DEPLOYMENT_TYPE" == "production" && $DEPLOYMENT_WITH_PROPERTY == "No") || "$DEPLOYMENT_TYPE" == "starter" ]]; then
            info "Not found configmap \"cp4ba-fips-status\" in the project \"$CP4BA_SERVICES_NS\". setting \"shared_configuration.enable_fips\" as \"false\" by default in the final custom resource."
            FIPS_ENABLED="false"
        fi
    elif [[ "$all_fips_enabled_flag" == "Yes" ]]; then
        printf "\n"
        while true; do
            printf "\x1B[1mYour OCP cluster has FIPS enabled, do you want to enable FIPS with this CP4BA deployment？\x1B[0m${YELLOW_TEXT} (Notes: If you select \"Yes\", in order to complete enablement of FIPS for CP4BA, refer to \"FIPS wall\" configuration in IBM documentation.)${RESET_TEXT} (Yes/No, default: No): "
            read -rp "" ans
            case "$ans" in
            "y"|"Y"|"yes"|"Yes"|"YES")
            FIPS_ENABLED="true"
            break
            ;;
            "n"|"N"|"no"|"No"|"NO"|"")
                FIPS_ENABLED="false"
                break
                ;;
            *)
                echo -e "Answer must be \"Yes\" or \"No\"\n"
                ;;
            esac
        done
    elif [[ "$all_fips_enabled_flag" == "No" ]]; then
        FIPS_ENABLED="false"
    fi
}

function select_ldap_type(){
    printf "\n"
    COLUMNS=12
    echo -e "\x1B[1mWhat is the LDAP type that is used for this deployment? \x1B[0m"
    options=("Microsoft Active Directory" "IBM Tivoli Directory Server / Security Directory Server")
    PS3='Enter a valid option [1 to 2]: '
    select opt in "${options[@]}"
    do
        case $opt in
            "Microsoft Active Directory")
                LDAP_TYPE="AD"
                break
                ;;
            "IBM Tivoli"*)
                LDAP_TYPE="TDS"
                break
                ;;
            *) echo "invalid option $REPLY";;
        esac
    done

}
function set_ldap_type_foundation(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_BAK} ${CP4A_PATTERN_FILE_TMP}

        if [[ "$LDAP_TYPE" == "AD" ]]; then
            content_start="$(grep -n "ad:" ${CP4A_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        else
            content_start="$(grep -n "tds:" ${CP4A_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        fi
        content_stop="$(tail -n +$content_start < ${CP4A_PATTERN_FILE_TMP} | grep -n "lc_group_filter:" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start - 1))
        vi ${CP4A_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/    # /    ' -c ':wq' >/dev/null 2>&1

        # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_TMP} ${CP4A_PATTERN_FILE_BAK}
    fi
}

function set_ldap_type_content_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${CONTENT_PATTERN_FILE_BAK} ${CONTENT_PATTERN_FILE_TMP}

        if [[ "$LDAP_TYPE" == "AD" ]]; then
            content_start="$(grep -n "## The User script will uncomment" ${CONTENT_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        else
            content_start="$(grep -n "## The User script will uncomment" ${CONTENT_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        fi
        content_stop="$(tail -n +$content_start < ${CONTENT_PATTERN_FILE_TMP} | grep -n "lc_group_filter:" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start + 2))
        vi ${CONTENT_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'d' -c ':wq' >/dev/null 2>&1

        ${COPY_CMD} -rf ${CONTENT_PATTERN_FILE_TMP} ${CONTENT_PATTERN_FILE_BAK}
    fi
}

function set_ldap_type_adp_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${ARIA_PATTERN_FILE_BAK} ${ARIA_PATTERN_FILE_TMP}

        if [[ "$LDAP_TYPE" == "AD" ]]; then
            content_start="$(grep -n "## The User script will uncomment" ${ARIA_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        else
            content_start="$(grep -n "## The User script will uncomment" ${ARIA_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        fi
        content_stop="$(tail -n +$content_start < ${ARIA_PATTERN_FILE_TMP} | grep -n "lc_group_filter:" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start + 2))
        vi ${ARIA_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'d' -c ':wq' >/dev/null 2>&1

        ${COPY_CMD} -rf ${ARIA_PATTERN_FILE_TMP} ${ARIA_PATTERN_FILE_BAK}
    fi
}

function set_ldap_type_workstreams_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${WORKSTREAMS_PATTERN_FILE_BAK} ${WORKSTREAMS_PATTERN_FILE_TMP}

        if [[ "$LDAP_TYPE" == "AD" ]]; then
            content_start="$(grep -n "ad:" ${WORKSTREAMS_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        else
            content_start="$(grep -n "tds:" ${WORKSTREAMS_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        fi
        content_stop="$(tail -n +$content_start < ${WORKSTREAMS_PATTERN_FILE_TMP} | grep -n "lc_group_filter:" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start - 1))
        vi ${WORKSTREAMS_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/    # /    ' -c ':wq' >/dev/null 2>&1

        ${COPY_CMD} -rf ${WORKSTREAMS_PATTERN_FILE_TMP} ${WORKSTREAMS_PATTERN_FILE_BAK}
    fi
}

function set_ldap_type_workflow_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${WORKFLOW_PATTERN_FILE_BAK} ${WORKFLOW_PATTERN_FILE_TMP}

        if [[ "$LDAP_TYPE" == "AD" ]]; then
            content_start="$(grep -n "ad:" ${WORKFLOW_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        else
            content_start="$(grep -n "tds:" ${WORKFLOW_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        fi
        content_stop="$(tail -n +$content_start < ${WORKFLOW_PATTERN_FILE_TMP} | grep -n "lc_group_filter:" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start - 1))
        vi ${WORKFLOW_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/    # /    ' -c ':wq' >/dev/null 2>&1

        ${COPY_CMD} -rf ${WORKFLOW_PATTERN_FILE_TMP} ${WORKFLOW_PATTERN_FILE_BAK}
    fi
}

function set_ldap_type_ww_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${WW_PATTERN_FILE_BAK} ${WW_PATTERN_FILE_TMP}

        if [[ "$LDAP_TYPE" == "AD" ]]; then
            content_start="$(grep -n "ad:" ${WW_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        else
            content_start="$(grep -n "tds:" ${WW_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
        fi
        content_stop="$(tail -n +$content_start < ${WW_PATTERN_FILE_TMP} | grep -n "lc_group_filter:" | head -n1 | cut -d: -f1)"
        content_stop=$(( $content_stop + $content_start - 1))
        vi ${WW_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/    # /    ' -c ':wq' >/dev/null 2>&1

        ${COPY_CMD} -rf ${WW_PATTERN_FILE_TMP} ${WW_PATTERN_FILE_BAK}
    fi
}
function set_external_ldap(){
    printf "\n"

    while true; do
        printf "\x1B[1mWill an external LDAP be used as part of the configuration? \x1B[0m"

        read -rp "" ans
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES")
            SET_EXT_LDAP="Yes"
            break
            ;;
        "n"|"N"|"no"|"No"|"NO")
            SET_EXT_LDAP="No"
            break
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            ;;
        esac
    done

}
function set_external_share_content_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" && $SET_EXT_LDAP == "Yes" ]] ;
    then
        containsElement "es" "${OPT_COMPONENTS_CR_SELECTED[@]}"
        retVal=$?
        if [[ $retVal -eq 0 ]]; then
            ${COPY_CMD} -rf ${CONTENT_PATTERN_FILE_BAK} ${CONTENT_PATTERN_FILE_TMP}
            # un-comment ext_ldap_configuration
            content_start="$(grep -n "ext_ldap_configuration:" ${CONTENT_PATTERN_FILE_TMP} | head -n 1 | cut -d: -f1)"
            content_stop="$(tail -n +$content_start < ${CONTENT_PATTERN_FILE_TMP} | grep -n "lc_ldap_group_member_id_map:" | head -n1 | cut -d: -f1)"
            content_stop=$(( $content_stop + $content_start - 1))
            vi ${CONTENT_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/  # /  ' -c ':wq' >/dev/null 2>&1

            # un-comment LDAP
            if [[ $DEPLOYMENT_TYPE == "starter" || ($DEPLOYMENT_TYPE == "production" && $DEPLOYMENT_WITH_PROPERTY == "No") ]]; then
                if [[ "$LDAP_TYPE" == "AD" ]]; then
                    # content_start="$(grep -n "ad:" ${CONTENT_PATTERN_FILE_TMP} | awk 'NR==2{print $1}' | cut -d: -f1)"
                    content_start="$(grep -n "ad:" ${CONTENT_PATTERN_FILE_TMP} | cut -d: -f1)"
                else
                    # content_start="$(grep -n "tds:" ${CONTENT_PATTERN_FILE_TMP} | awk 'NR==2{print $1}' | cut -d: -f1)"
                    content_start="$(grep -n "tds:" ${CONTENT_PATTERN_FILE_TMP} | cut -d: -f1)"
                fi
            elif [[ $DEPLOYMENT_TYPE == "production" && $DEPLOYMENT_WITH_PROPERTY == "Yes" ]]; then
                tmp_ldap_type="$(prop_ext_ldap_property_file LDAP_TYPE)"
                tmp_ldap_type=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ldap_type")
                if [[ $tmp_ldap_type == "Microsoft Active Directory" ]]; then
                    # content_start="$(grep -n "ad:" ${CONTENT_PATTERN_FILE_TMP} | awk 'NR==2{print $1}' | cut -d: -f1)"
                    content_start="$(grep -n "ad:" ${CONTENT_PATTERN_FILE_TMP} | cut -d: -f1)"
                elif [[ $tmp_ldap_type == "IBM Security Directory Server" ]]; then
                    # content_start="$(grep -n "tds:" ${CONTENT_PATTERN_FILE_TMP} | awk 'NR==2{print $1}' | cut -d: -f1)"
                    content_start="$(grep -n "tds:" ${CONTENT_PATTERN_FILE_TMP} | cut -d: -f1)"
                else
                    fail "The value for \"LDAP_TYPE\" in the property file \"${EXTERNAL_LDAP_PROPERTY_FILE}\" is not valid. The possible values are: \"IBM Security Directory Server\" or \"Microsoft Active Directory\""
                    exit 1
                fi
            fi
            content_stop="$(tail -n +$content_start < ${CONTENT_PATTERN_FILE_TMP} | grep -n "lc_group_filter:" | head -n1 | cut -d: -f1)"
            content_stop=$(( $content_stop + $content_start - 1))
            vi ${CONTENT_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"'s/    # /    ' -c ':wq'

            ${COPY_CMD} -rf ${CONTENT_PATTERN_FILE_TMP} ${CONTENT_PATTERN_FILE_BAK}
        fi
    fi
}

function set_object_store_content_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${CONTENT_PATTERN_FILE_BAK} ${CONTENT_PATTERN_FILE_TMP}
        if [[ $content_os_number -gt 0 ]]; then
            content_start="$(grep -n "datasource_configuration:" ${CONTENT_PATTERN_FILE_TMP} |  head -n 1 | cut -d: -f1)"
            content_tmp="$(tail -n +$content_start < ${CONTENT_PATTERN_FILE_TMP} | grep -n "dc_os_datasources:" | head -n1 | cut -d: -f1)"
            content_tmp=$(( content_tmp + $content_start - 1))
            content_stop="$(tail -n +$content_tmp < ${CONTENT_PATTERN_FILE_TMP} | grep -n "dc_database_type:" | head -n1 | cut -d: -f1)"
            content_start=$(( $content_stop + $content_tmp - 1))
            content_tmp="$(tail -n +$content_start < ${CONTENT_PATTERN_FILE_TMP} | grep -n "dc_hadr_max_retries_for_client_reroute:" | head -n1 | cut -d: -f1)"
            content_stop=$(( $content_start + $content_tmp - 1))

            for ((j=1;j<${content_os_number};j++))
            do
                vi ${CONTENT_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"' copy '"${content_stop}"'' -c ':wq' >/dev/null 2>&1
            done

            for ((j=1;j<${content_os_number};j++))
            do
                ((obj_num=j+1))
                ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[${j}].dc_common_os_datasource_name "\"FNOS${obj_num}DS\""
                ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[${j}].dc_common_os_xa_datasource_name "\"FNOS${obj_num}DSXA\""
            done

            # Add additional OS into initialize_configuration
            content_start="$(grep -n "ic_obj_store_creation:" ${CONTENT_PATTERN_FILE_TMP} |  head -n 1 | cut -d: -f1)"
            content_tmp="$(tail -n +$content_start < ${CONTENT_PATTERN_FILE_TMP} | grep -n "object_stores:" | head -n1 | cut -d: -f1)"
            content_tmp=$(( content_tmp + $content_start - 1))
            content_stop="$(tail -n +$content_tmp < ${CONTENT_PATTERN_FILE_TMP} | grep -n "oc_cpe_obj_store_display_name:" | head -n1 | cut -d: -f1)"
            content_start=$(( $content_stop + $content_tmp - 1))
            content_tmp="$(tail -n +$content_start < ${CONTENT_PATTERN_FILE_TMP} | grep -n "\"<Required>\" # user name and group name for object store admin" | head -n1 | cut -d: -f1)"
            content_stop=$(( $content_start + $content_tmp - 1))

            for ((j=1;j<${content_os_number};j++))
            do
                vi ${CONTENT_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"' copy '"${content_stop}"'' -c ':wq' >/dev/null 2>&1
            done

            ## Adding the Object Stores to the Custom Resource
            ## spec.initialize_configuration
            for ((j=1;j<${content_os_number};j++))
            do
                ((obj_num=j+1))
                if [[ $obj_num -lt "10" ]]; then
                    ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_display_name "\"OS0${obj_num}\""
                    ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_symb_name "\"OS0${obj_num}\""
                else
                    ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_display_name "\"OS${obj_num}\""
                    ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_symb_name "\"OS${obj_num}\""
                fi
                ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_conn.name "\"objectstore${obj_num}_connection\""
                ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_conn.dc_os_datasource_name "\"FNOS${obj_num}DS\""
                ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_conn.dc_os_xa_datasource_name "\"FNOS${obj_num}DSXA\""
            done

            if [[ " ${pattern_cr_arr[@]}" =~ "document_processing" && $DEPLOYMENT_WITH_PROPERTY == "Yes" ]]; then
                for ((j=0;j<${content_os_number};j++))
                do
                    ((obj_num=j+1))
                    tmp_os_db_enable_adp="$(prop_db_name_user_property_file OS${obj_num}_ENABLE_DOCUMENT_PROCESSING)"
                    tmp_os_db_enable_adp=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_enable_adp")
                    if [[ $tmp_os_db_enable_adp == "Yes" || $tmp_os_db_enable_adp == "YES" || $tmp_os_db_enable_adp == "Y" || $tmp_os_db_enable_adp == "True" || $tmp_os_db_enable_adp == "true" ]]; then
                        ${YQ_CMD} w -i ${CONTENT_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_enable_document_processing "true"
                    fi
                done
            fi

        fi
        ${COPY_CMD} -rf ${CONTENT_PATTERN_FILE_TMP} ${CONTENT_PATTERN_FILE_BAK}
    fi
}

function set_object_store_adp_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${ARIA_PATTERN_FILE_BAK} ${ARIA_PATTERN_FILE_TMP}
        if [[ $content_os_number -gt 0 ]]; then
            content_start="$(grep -n "datasource_configuration:" ${ARIA_PATTERN_FILE_TMP} |  head -n 1 | cut -d: -f1)"
            content_tmp="$(tail -n +$content_start < ${ARIA_PATTERN_FILE_TMP} | grep -n "dc_os_datasources:" | head -n1 | cut -d: -f1)"
            content_tmp=$(( content_tmp + $content_start - 1))
            content_stop="$(tail -n +$content_tmp < ${ARIA_PATTERN_FILE_TMP} | grep -n "dc_database_type:" | head -n1 | cut -d: -f1)"
            content_start=$(( $content_stop + $content_tmp - 1))
            content_tmp="$(tail -n +$content_start < ${ARIA_PATTERN_FILE_TMP} | grep -n "dc_hadr_max_retries_for_client_reroute:" | head -n1 | cut -d: -f1)"
            content_stop=$(( $content_start + $content_tmp - 1))

            for ((j=1;j<${content_os_number};j++))
            do
                vi ${ARIA_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"' copy '"${content_stop}"'' -c ':wq' >/dev/null 2>&1
            done

            for ((j=1;j<${content_os_number};j++))
            do
                ((obj_num=j+1))
                ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[${j}].dc_common_os_datasource_name "\"FNOS${obj_num}DS\""
                ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[${j}].dc_common_os_xa_datasource_name "\"FNOS${obj_num}DSXA\""
            done

            # Add additional OS into initialize_configuration
            content_start="$(grep -n "ic_obj_store_creation:" ${ARIA_PATTERN_FILE_TMP} |  head -n 1 | cut -d: -f1)"
            content_tmp="$(tail -n +$content_start < ${ARIA_PATTERN_FILE_TMP} | grep -n "object_stores:" | head -n1 | cut -d: -f1)"
            content_tmp=$(( content_tmp + $content_start - 1))
            content_stop="$(tail -n +$content_tmp < ${ARIA_PATTERN_FILE_TMP} | grep -n "oc_cpe_obj_store_display_name:" | head -n1 | cut -d: -f1)"
            content_start=$(( $content_stop + $content_tmp - 1))
            content_tmp="$(tail -n +$content_start < ${ARIA_PATTERN_FILE_TMP} | grep -n "\"<Required>\" # user name and group name for object store admin" | head -n1 | cut -d: -f1)"
            content_stop=$(( $content_start + $content_tmp - 1))

            for ((j=1;j<${content_os_number};j++))
            do
                vi ${ARIA_PATTERN_FILE_TMP} -c ':'"${content_start}"','"${content_stop}"' copy '"${content_stop}"'' -c ':wq' >/dev/null 2>&1
            done

            for ((j=1;j<${content_os_number};j++))
            do
                ((obj_num=j+1))
                if [[ $obj_num -lt "10" ]]; then
                    ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_display_name "\"OS0${obj_num}\""
                    ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_symb_name "\"OS0${obj_num}\""
                else
                    ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_display_name "\"OS${obj_num}\""
                    ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_symb_name "\"OS${obj_num}\""
                fi
                ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_conn.name "\"objectstore${obj_num}_connection\""
                ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_conn.dc_os_datasource_name "\"FNOS${obj_num}DS\""
                ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_conn.dc_os_xa_datasource_name "\"FNOS${obj_num}DSXA\""
            done

            if [[ $DEPLOYMENT_WITH_PROPERTY == "Yes" ]]; then
                for ((j=0;j<${content_os_number};j++))
                do
                    ((obj_num=j+1))
                    tmp_os_db_enable_adp="$(prop_db_name_user_property_file OS${obj_num}_ENABLE_DOCUMENT_PROCESSING)"
                    tmp_os_db_enable_adp=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_enable_adp")
                    if [[ $tmp_os_db_enable_adp == "Yes" || $tmp_os_db_enable_adp == "YES" || $tmp_os_db_enable_adp == "Y" || $tmp_os_db_enable_adp == "True" || $tmp_os_db_enable_adp == "true" ]]; then
                        ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[${j}].oc_cpe_obj_store_enable_document_processing "true"
                    fi
                done
            fi
        fi
        ${COPY_CMD} -rf ${ARIA_PATTERN_FILE_TMP} ${ARIA_PATTERN_FILE_BAK}
    fi
}

function set_aca_tenant_pattern(){
    if [[ $DEPLOYMENT_TYPE == "production" ]] ;
    then
        ${COPY_CMD} -rf ${ACA_PATTERN_FILE_BAK} ${ACA_PATTERN_FILE_TMP}
        # ${YQ_CMD} d -i ${ACA_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.tenant_databases
        if [ ${#aca_tenant_arr[@]} -eq 0 ]; then
            echo -e "\x1B[1;31mNot any element in ACA tenant list found\x1B[0m:\x1B[1m"
        else
            for i in ${!aca_tenant_arr[@]}; do
               ${YQ_CMD} w -i ${ACA_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.tenant_databases.[${i}] "${aca_tenant_arr[i]}"
             done
        fi
        ${COPY_CMD} -rf ${ACA_PATTERN_FILE_TMP} ${ACA_PATTERN_FILE_BAK}
    fi
}

function select_automation_service(){
    if [[ !(" ${PATTERNS_CR_SELECTED[@]} " =~ "application" || " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-authoring" || " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-runtime" || " ${PATTERNS_CR_SELECTED[@]} " =~ "workstreams" || " ${PATTERNS_CR_SELECTED[@]} " =~ "document_processing")]]; then
        printf "\n"
        while true; do
            printf "\x1B[1mDo you want to enable the Business Automation Service? (Yes/No): \x1B[0m"
            read -rp "" ans
            case "$ans" in
            "y"|"Y"|"yes"|"Yes"|"YES")
                foundation_component_arr=( "${foundation_component_arr[@]}" "AE" )
                AUTOMATION_SERVICE_ENABLE="Yes"
                break
                ;;
            "n"|"N"|"no"|"No"|"NO")
                break
                ;;
            *)
                echo -e "Answer must be \"Yes\" or \"No\"\n"
                ;;
            esac
        done
    fi
}

function select_cpe_full_storage(){
    if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "document_processing" ]]; then
        printf "\n"
        while true; do
            printf "\x1B[1mDo you want limited CPE storage support? (Yes/No): \x1B[0m"
            read -rp "" ans
            case "$ans" in
            "y"|"Y"|"yes"|"Yes"|"YES")
                CPE_FULL_STORAGE="No"
                break
                ;;
            "n"|"N"|"no"|"No"|"NO")
                CPE_FULL_STORAGE="Yes"
                break
                ;;
            *)
                echo -e "Answer must be \"Yes\" or \"No\"\n"
                ;;
            esac
        done
    fi
}

function select_enable_deep_learning(){
    if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "document_processing" ]]; then
        printf "\n"
        while true; do
            printf "\x1B[1mDo you want to enable Deep Learning Capability (Yes/No): \x1B[0m"
            read -rp "" ans
            case "$ans" in
            "y"|"Y"|"yes"|"Yes"|"YES")
                ADP_DL_ENABLED="Yes"
                break
                ;;
            "n"|"N"|"no"|"No"|"NO")
                ADP_DL_ENABLED="No"
                break
                ;;
            *)
                echo -e "Answer must be \"Yes\" or \"No\"\n"
                ;;
            esac
        done
    fi
}

function select_ae_data_persistence(){
    if [[ " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "ae_data_persistence" ]]; then
        foundation_component_arr=( "${foundation_component_arr[@]}" "AE" )
        AE_DATA_PERSISTENCE_ENABLE="Yes"
    else
        if [[ (" ${PATTERNS_CR_SELECTED[@]} " =~ "application") ]]; then
            printf "\n"
            while true; do
                printf "\x1B[1mDo you want to enable Business Automation Application Data Persistence? (Yes/No): \x1B[0m"
                read -rp "" ans
                case "$ans" in
                "y"|"Y"|"yes"|"Yes"|"YES")
                    foundation_component_arr=( "${foundation_component_arr[@]}" "AE" )
                    AE_DATA_PERSISTENCE_ENABLE="Yes"
                    # optional_component_cr_arr=( "${optional_component_cr_arr[@]}" "ae_data_persistence" )
                    break
                    ;;
                "n"|"N"|"no"|"No"|"NO"|"")
                    break
                    ;;
                *)
                    echo -e "Answer must be \"Yes\" or \"No\"\n"
                    ;;
                esac
            done
        fi
    fi
}

function select_aca_tenant(){
    printf "\n"
    printf "\x1B[1mHow many projects do you want to create initially with Document Processing Engine (DPE)? \x1B[0m"
    aca_tenant_number=""
    aca_tenant_arr=()
    while [[ $aca_tenant_number == "" ]];
    do
        read -rp "" aca_tenant_number
        if ! [[ "$aca_tenant_number" =~ ^[0-9]+$ ]]; then
            echo -e "\x1B[1;31mEnter a valid tenant number\x1B[0m"
            aca_tenant_number=""
        fi
    done

    order_number=1
    while (( ${#aca_tenant_arr[@]} < $aca_tenant_number ));
    do
        printf "\x1B[1mWhat is the name of tenant ${order_number}? \x1B[0m"
        read -rp "" aca_tenant_name
        if [ -z "$aca_tenant_number" ]; then
            echo -e "\x1B[1;31mEnter a valid tenant name\x1B[0m"
        else
            aca_tenant_arr=( "${aca_tenant_arr[@]}" "${aca_tenant_name}" )
        fi
        ((order_number++))
        printf "\n"
    done
    printf "\n"
}

function select_baw_only(){
    pattern_arr=()
    pattern_cr_arr=()
    printf "\n"
    echo -e "\x1B[1mSelect the Cloud Pak for Business Automation capability to install: \x1B[0m"
    COLUMNS=12

    options=("Business Automation Workflow")
    PS3='Enter a valid option [1 to 1]: '

    select opt in "${options[@]}"
    do
        case $opt in
            "Business Automation Workflow")
                pattern_arr=("Business Automation Workflow")
                pattern_cr_arr=("workflow")
                break
                ;;
            *) echo "invalid option $REPLY";;
        esac
    done

    if [[ $PLATFORM_SELECTED == "other" ]]; then
        foundation_baw=("BAN" "RR" "UMS" "AE")
    else
        foundation_baw=("BAN" "RR" "AE")
    fi

    foundation_component_arr=( "${foundation_component_arr[@]}" "${foundation_baw[@]}" )
    PATTERNS_CR_SELECTED=$( IFS=$','; echo "${pattern_cr_arr[*]}" )

    FOUNDATION_CR_SELECTED=($(echo "${foundation_component_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
    # FOUNDATION_CR_SELECTED_LOWCASE=( "${FOUNDATION_CR_SELECTED[@],,}" )

    x=0;while [ ${x} -lt ${#FOUNDATION_CR_SELECTED[*]} ] ; do FOUNDATION_CR_SELECTED_LOWCASE[$x]=$(tr [A-Z] [a-z] <<< ${FOUNDATION_CR_SELECTED[$x]}); let x++; done
    FOUNDATION_DELETE_LIST=($(echo "${FOUNDATION_CR_SELECTED[@]}" "${FOUNDATION_FULL_ARR[@]}" | tr ' ' '\n' | sort | uniq -u))

    PATTERNS_CR_SELECTED=($(echo "${pattern_cr_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
}

function clean_up_temp_file(){
    local files=()
    if [[ -d $TEMP_FOLDER ]]; then
        files=($(find $TEMP_FOLDER -name '*.yaml'))
        for item in ${files[*]}
        do
            rm -rf $item >/dev/null 2>&1
        done

        files=($(find $TEMP_FOLDER -name '*.swp'))
        for item in ${files[*]}
        do
            rm -rf $item >/dev/null 2>&1
        done
    fi
}

function input_information(){
    if [[ $DEPLOYMENT_WITH_PROPERTY == "No" || $DEPLOYMENT_TYPE == "starter" ]]; then
        select_installation_type
    elif [[ $DEPLOYMENT_WITH_PROPERTY == "Yes" ]]; then
        INSTALLATION_TYPE="new"
    fi
    # clean_up_temp_file
    # rm -rf $BAK_FOLDER >/dev/null 2>&1
    # rm -rf $FINAL_CR_FOLDER >/dev/null 2>&1

    mkdir -p $TEMP_FOLDER >/dev/null 2>&1
    mkdir -p $BAK_FOLDER >/dev/null 2>&1
    mkdir -p $FINAL_CR_FOLDER >/dev/null 2>&1

    if [[ ${INSTALLATION_TYPE} == "existing" ]]; then
        # INSTALL_BAW_IAWS="No"
        prepare_pattern_file
        select_deployment_type
        if [[ $DEPLOYMENT_TYPE == "production" && (-z $PROFILE_TYPE) ]]; then
            select_profile_type
        fi
        select_platform
        if [[ ("$PLATFORM_SELECTED" == "OCP" || "$PLATFORM_SELECTED" == "ROKS") && "$DEPLOYMENT_TYPE" == "production" && "$USE_DEFAULT_IAM_ADMIN" == "" && "$NON_DEFAULT_IAM_ADMIN" == "" ]]; then
            select_iam_default_admin
        fi
        if [[ ("$PLATFORM_SELECTED" == "OCP" || "$PLATFORM_SELECTED" == "ROKS") && "$DEPLOYMENT_TYPE" == "starter" ]]; then
            select_project
        fi
        check_ocp_version
        validate_docker_podman_cli
    elif [[ ${INSTALLATION_TYPE} == "new" ]]
    then
        # select_ocp_olm
        select_deployment_type
        if [[ $DEPLOYMENT_WITH_PROPERTY == "Yes" && $DEPLOYMENT_TYPE == "production" ]]; then
            if [[ ! -z $CP4BA_AUTO_NAMESPACE ]]; then
                TARGET_PROJECT_NAME=$CP4BA_AUTO_NAMESPACE
            fi
            load_property_before_generate
            show_summary_pattern_selected
        fi
        if [[ $DEPLOYMENT_TYPE == "production" && (-z $PROFILE_TYPE) ]]; then
            select_profile_type
        fi
        select_platform
        if [[ ("$PLATFORM_SELECTED" == "OCP" || "$PLATFORM_SELECTED" == "ROKS") && "$DEPLOYMENT_TYPE" == "production" && "$USE_DEFAULT_IAM_ADMIN" == "" && "$NON_DEFAULT_IAM_ADMIN" == "" ]]; then
            select_iam_default_admin
        fi
        if [[ ("$PLATFORM_SELECTED" == "OCP" || "$PLATFORM_SELECTED" == "ROKS") && "$DEPLOYMENT_TYPE" == "starter" ]]; then
            select_project
        fi
        check_ocp_version
        validate_docker_podman_cli
        prepare_pattern_file
        # select_baw_iaws_installation
    fi

    if [[ "${INSTALLATION_TYPE}" == "existing" ]] && (( ${#EXISTING_PATTERN_ARR[@]} == 0 )); then
        echo -e "\x1B[1;31mNO EXISTING PATTERN FOUND!\x1B[0m"
        prompt_press_any_key_to_continue "Install a new pattern"
    fi

    if [[ "${INSTALL_BAW_ONLY}" == "No" ]];
    then
        if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
            select_pattern
        elif [[ $DEPLOYMENT_WITH_PROPERTY == "Yes" && $DEPLOYMENT_TYPE == "production" ]]; then
            FOUNDATION_CR_SELECTED=($(echo "${foundation_component_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))

            x=0;while [ ${x} -lt ${#FOUNDATION_CR_SELECTED[*]} ] ; do FOUNDATION_CR_SELECTED_LOWCASE[$x]=$(tr [A-Z] [a-z] <<< ${FOUNDATION_CR_SELECTED[$x]}); let x++; done
            FOUNDATION_DELETE_LIST=($(echo "${FOUNDATION_CR_SELECTED[@]}" "${FOUNDATION_FULL_ARR[@]}" | tr ' ' '\n' | sort | uniq -u))

            PATTERNS_CR_SELECTED=($(echo "${pattern_cr_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
        fi
    else
        select_baw_only
    fi

    if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
        select_optional_component
    elif [[ $DEPLOYMENT_WITH_PROPERTY == "Yes" && $DEPLOYMENT_TYPE == "production" ]]; then
        OPT_COMPONENTS_CR_SELECTED=($(echo "${optional_component_cr_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
        OPTIONAL_COMPONENT_DELETE_LIST=($(echo "${OPT_COMPONENTS_CR_SELECTED[@]}" "${OPTIONAL_COMPONENT_FULL_ARR[@]}" | tr ' ' '\n' | sort | uniq -u))
        KEEP_COMPOMENTS=($(echo ${FOUNDATION_CR_SELECTED_LOWCASE[@]} ${OPTIONAL_COMPONENT_DELETE_LIST[@]} | tr ' ' '\n' | sort | uniq -d | uniq))
        OPT_COMPONENTS_SELECTED=($(echo "${optional_component_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
    fi



    # get jdbc url according to whether ICCSAP component selected
    if [[ ( -z $CP4BA_JDBC_URL || $CP4BA_JDBC_URL == "") && (( $DEPLOYMENT_TYPE == "starter" ) && (" ${optional_component_cr_arr[@]} " =~ "iccsap") || $DEPLOYMENT_TYPE == "production") ]]; then
        get_jdbc_url
    fi
    if [[ "$INSTALLATION_TYPE" == "new" ]]; then
        if [[ $PLATFORM_SELECTED == "other" ]]; then
            get_entitlement_registry
        fi
        if [[ "$use_entitlement" == "no" ]]; then
            verify_local_registry_password
        fi

        # if  [[ $PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS" ]];
        # then
        #     get_infra_name
        # fi
        # load storage class name
        if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
            get_storage_class_name
        elif [[ $DEPLOYMENT_WITH_PROPERTY == "Yes" && $DEPLOYMENT_TYPE == "production" ]]; then
            SLOW_STORAGE_CLASS_NAME=$(prop_user_profile_property_file CP4BA.SLOW_FILE_STORAGE_CLASSNAME)
            MEDIUM_STORAGE_CLASS_NAME=$(prop_user_profile_property_file CP4BA.MEDIUM_FILE_STORAGE_CLASSNAME)
            FAST_STORAGE_CLASS_NAME=$(prop_user_profile_property_file CP4BA.FAST_FILE_STORAGE_CLASSNAME)
            BLOCK_STORAGE_CLASS_NAME=$(prop_user_profile_property_file CP4BA.BLOCK_STORAGE_CLASS_NAME)
            if [[ -z $SLOW_STORAGE_CLASS_NAME || -z $MEDIUM_STORAGE_CLASS_NAME || -z $FAST_STORAGE_CLASS_NAME || -z $BLOCK_STORAGE_CLASS_NAME ]]; then
                get_storage_class_name
            fi
        fi

        # Select FIPS enable or not

        if  [[ ("$DEPLOYMENT_TYPE" == "production" && $DEPLOYMENT_WITH_PROPERTY == "No") && ($PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS") ]]; then
            select_fips_enable
        elif [[ "$DEPLOYMENT_TYPE" == "starter" ]]; then
            FIPS_ENABLED="false"
        fi

        if  [[  ("$DEPLOYMENT_TYPE" == "production" && $DEPLOYMENT_WITH_PROPERTY == "No") && ($PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS") ]]; then
            generate_sample_network_policies
        elif [[ "$DEPLOYMENT_TYPE" == "starter" ]]; then
            # For starter deployment, always set generate_sample_network_policies: true
            info "For starter deployment, always setting \"generate_sample_network_policies\" as \"true\" in the final custom resource."
            GENERATE_SAMPLE_NETWORK_POLICIES="true"
        fi

        if [[ "$DEPLOYMENT_TYPE" == "production" && $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then

            # whether wfps authoring require LDAP
            if [[ "${#pattern_cr_arr[@]}" -eq "1" && "${pattern_cr_arr[@]}" =~ "workflow-process-service" ]]; then
                select_ldap_type_for_wfps_authoring
            fi

            if [[ -z $LDAP_WFPS_AUTHORING || $LDAP_WFPS_AUTHORING == "Yes" ]]; then
                select_ldap_type
            fi

        fi
    elif [[ "$INSTALLATION_TYPE" == "existing" ]]
    then
        existing_infra_name=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_hostname_suffix`
        if [ ! -z "$existing_infra_name" ]; then
            chrlen=${#existing_infra_name}
            INFRA_NAME=${existing_infra_name:21:chrlen}
        fi
        existing_ldap_type=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.ldap_configuration.lc_selected_ldap_type`
        if [[ "$existing_ldap_type" == "Microsoft Active Directory" ]];then
            LDAP_TYPE="AD"

        elif [[ "$existing_ldap_type" == "IBM Security Directory Server" ]]
        then
            LDAP_TYPE="TDS"
        fi
        existing_docker_reg_server=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.sc_image_repository`
        if [[ "$existing_docker_reg_server" == *"icr.io"* ]]; then
            use_entitlement="yes"
        fi

        local_registry_server=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.sc_image_repository`
        DOCKER_REG_SERVER="${existing_docker_reg_server}"
        LOCAL_REGISTRY_SERVER=${local_registry_server}
        OIFS=$IFS
        IFS='/' read -r -a docker_reg_url_array <<< "$local_registry_server"
        delim=""
        joined=""
        for item in "${docker_reg_url_array[@]}"; do
                joined="$joined$delim$item"
                delim="\/"
        done
        IFS=$OIFS
        CONVERT_LOCAL_REGISTRY_SERVER=${joined}
        DOCKER_RES_SECRET_NAME=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.image_pull_secrets.[0]`
        STORAGE_CLASS_NAME=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.storage_configuration.sc_dynamic_storage_classname`
        SLOW_STORAGE_CLASS_NAME=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.storage_configuration.sc_slow_file_storage_classname`
        MEDIUM_STORAGE_CLASS_NAME=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.storage_configuration.sc_medium_file_storage_classname`
        FAST_STORAGE_CLASS_NAME=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.storage_configuration.sc_fast_file_storage_classname`
        BLOCK_STORAGE_CLASS_NAME=`cat $CP4A_EXISTING_BAK | ${YQ_CMD} r - spec.shared_configuration.storage_configuration.sc_block_storage_classname`
    fi

    if [[ "$DEPLOYMENT_TYPE" == "production" && $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
        if [[ " ${pattern_cr_arr[@]}" =~ "content" || " ${pattern_cr_arr[@]}" =~ "document_processing" ]]; then
            select_objectstore_number
        fi
    fi
    if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
        select_cpe_full_storage

        containsElement "document_processing_designer" "${PATTERNS_CR_SELECTED[@]}"
        retVal=$?
        if [[ ( $retVal -eq 0 ) && "$DEPLOYMENT_TYPE" == "production" ]]; then
            select_gpu_document_processing
        fi

        containsElement "document_processing" "${PATTERNS_CR_SELECTED[@]}"
        retVal=$?
        if [[ ( $retVal -eq 0 ) && "$DEPLOYMENT_TYPE" == "starter" ]]; then
            select_enable_deep_learning
            if [[ $ADP_DL_ENABLED == "Yes" ]]; then
                select_gpu_document_processing
            elif [[ $ADP_DL_ENABLED == "No" || -z $ADP_DL_ENABLED ]]; then
                ENABLE_GPU_ARIA="No"
            fi
        fi
    fi
    if [[ ! (" ${PATTERNS_CR_SELECTED[@]} " =~ "content" && "${#PATTERNS_CR_SELECTED[@]}" -eq "1") ]]; then
        if [[ $IBM_LICENS == "Accept" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ibm_license "accept"
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ibm_license ""
        fi
    fi
}

function apply_cp4a_operator(){
    ${COPY_CMD} -rf ${OPERATOR_FILE_BAK} ${OPERATOR_FILE_TMP}

    printf "\n"
    if [[ ("$SCRIPT_MODE" != "review") && ("$SCRIPT_MODE" != "OLM") ]]; then
        echo -e "\x1B[1mInstalling the Cloud Pak for Business Automation operator...\x1B[0m"
    fi
    # set db2_license
    ${SED_COMMAND} '/baw_license/{n;s/value:.*/value: accept/;}' ${OPERATOR_FILE_TMP}
    # Set operator image pull secret
    ${SED_COMMAND} "s|ibm-entitlement-key|$DOCKER_RES_SECRET_NAME|g" ${OPERATOR_FILE_TMP}
    ${SED_COMMAND} "s|admin.registrykey|$DOCKER_RES_SECRET_NAME|g" ${OPERATOR_FILE_TMP}
    # Set operator image registry
    new_operator="$REGISTRY_IN_FILE\/cp\/cp4a"

    if [ "$use_entitlement" = "yes" ] ; then
        ${SED_COMMAND} "s/$REGISTRY_IN_FILE/$DOCKER_REG_SERVER/g" ${OPERATOR_FILE_TMP}

    else
        ${SED_COMMAND} "s/$new_operator/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${OPERATOR_FILE_TMP}
    fi

    if [[ "${OCP_VERSION}" == "3.11" ]];then
        ${SED_COMMAND} "s/\# runAsUser\: 1001/runAsUser\: 1001/g" ${OPERATOR_FILE_TMP}
    fi

    if [[ $INSTALLATION_TYPE == "new" ]]; then
        ${CLI_CMD} delete -f ${OPERATOR_FILE_TMP} >/dev/null 2>&1
        sleep 5
    fi

    INSTALL_OPERATOR_CMD="${CLI_CMD} apply -f ${OPERATOR_FILE_TMP}"
    if $INSTALL_OPERATOR_CMD ; then
        echo -e "\x1B[1mDone\x1B[0m"
    else
        echo -e "\x1B[1;31mFailed\x1B[0m"
    fi

    ${COPY_CMD} -rf ${OPERATOR_FILE_TMP} ${OPERATOR_FILE_BAK}
    printf "\n"
    # Check deployment rollout status every 5 seconds (max 10 minutes) until complete.
    echo -e "\x1B[1mWaiting for the Cloud Pak operator to be ready. This might take a few minutes... \x1B[0m"
    ATTEMPTS=0
    ROLLOUT_STATUS_CMD="${CLI_CMD} rollout status deployment/ibm-cp4a-operator"
    until $ROLLOUT_STATUS_CMD || [ $ATTEMPTS -eq 120 ]; do
        $ROLLOUT_STATUS_CMD
        ATTEMPTS=$((ATTEMPTS + 1))
        sleep 5
    done
    if $ROLLOUT_STATUS_CMD ; then
        echo -e "\x1B[1mDone\x1B[0m"
    else
        echo -e "\x1B[1;31mFailed\x1B[0m"
    fi
    printf "\n"
}

function copy_jdbc_driver(){
    # Get pod name
    echo -e "\x1B[1mCopying the JDBC driver for the operator...\x1B[0m"
    operator_podname=$(${CLI_CMD} get pod|grep ibm-cp4a-operator|grep Running|awk '{print $1}')

    # ${CLI_CMD} exec -it ${operator_podname} -- rm -rf /opt/ansible/share/jdbc
    COPY_JDBC_CMD="${CLI_CMD} cp ${JDBC_DRIVER_DIR} ${operator_podname}:/opt/ansible/share/"

    if $COPY_JDBC_CMD ; then
        echo -e "\x1B[1mDone\x1B[0m"
    else
        echo -e "\x1B[1;31mFailed\x1B[0m"
    fi
}


function get_jdbc_url(){
    CP4BA_JDBC_URL=""
    while [[ $CP4BA_JDBC_URL == "" ]];
    do
        if [ -z "$CP4BA_JDBC_URL" ]; then
            printf "\n"
            echo -e "\x1B[1mProvide a URL to zip file that contains JDBC and/or ICCSAP drivers.\x1B[0m"
            read -p "(optional - if not provided, the Operator will configure using the default shipped JDBC driver): " CP4BA_JDBC_URL
            if [[ ( -z "$CP4BA_JDBC_URL" ) && (" ${optional_component_cr_arr[@]} " =~ "iccsap") ]]; then
                printf "\n"
                echo -e "\x1B[1;31mIBM Content Collector for SAP is selected, provide a URL to zip file that contains ICCSAP drivers.\x1B[0m"
                CP4BA_JDBC_URL=""
            elif [[ ( -z "$CP4BA_JDBC_URL" ) && ( ! " ${optional_component_cr_arr[@]} " =~ "iccsap" ) ]]; then
                printf "\n"
                echo -e "\x1B[1mThe Operator will configure using the default shipped JDBC driver.\x1B[0m"
                break
            fi
        fi
    done
}

function copy_sap_libraries(){
    SAP_LIBS_LIST=("libicudata.so.50" "libicudecnumber.so" "libicui18n.so.50" "libicuuc.so.50" "libsapcrypto.so" "libsapjco3.so" "libsapnwrfc.so" "sapjco3.jar" "libsapucum.so")
    # Get pod name

    echo -e "\x1B[1mCopying the SAP libraries for the operator...\x1B[0m"
    #Check if saplibs folder exists
    if [ ! -d ${SAP_LIB_DIR} ]; then
        echo -e "\x1B[1;31m\"${SAP_LIB_DIR}\" directory does not exist! Refer to the documentation to get the SAP libraries for ICCSAP. Exiting...
Check the following KC for details--> https://www.ibm.com/support/knowledgecenter/SSYHZ8_$CP4BA_RELEASE_BASE/com.ibm.dba.install/op_topics/tsk_deploy_demo.html \n\x1B[0m"
        exit 0
    fi

    #Check if all required SAP libs are present and print missing
    missing_libs="no"
    for file in "${SAP_LIBS_LIST[@]}"; do
        if [ ! -f ${SAP_LIB_DIR}/$file ]; then
            echo -e "\x1B[1;31m\"${SAP_LIB_DIR}/$file\" file does not exist!\n\x1B[0m"
            missing_libs="yes"
        fi
    done

    if [ $missing_libs == "yes" ]; then
        echo -e "\x1B[1;31mMissing required SAP Libraries. Refer to the documentation to get the SAP libraries for ICCSAP. Exiting...
Check the following KC for details--> https://www.ibm.com/support/knowledgecenter/SSYHZ8_$CP4BA_RELEASE_BASE/com.ibm.dba.install/op_topics/tsk_deploy_demo.html \n\x1B[0m"
        exit 0
    fi

    operator_podname=$(${CLI_CMD} get pod|grep ibm-cp4a-operator|grep Running|awk '{print $1}')

    #Delete existing saplibs directory from /opt/ansible/share/ before creating new one
    if [[ $INSTALLATION_TYPE == "existing" ]]; then
        ${CLI_CMD} exec -it ${operator_podname} -- rm -rf /opt/ansible/share/saplibs
    fi

    COPY_SAP_CMD="${CLI_CMD} cp ${SAP_LIB_DIR} ${operator_podname}:/opt/ansible/share/"

    if $COPY_SAP_CMD ; then
        echo -e "\x1B[1mDone\x1B[0m"
    else
        echo -e "\x1B[1;31mFailed\x1B[0m"
    fi
}


function set_foundation_components(){
    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_BAK} ${CP4A_PATTERN_FILE_TMP}
    if (( ${#FOUNDATION_DELETE_LIST[@]} > 0 ));then
        if (( ${#OPT_COMPONENTS_CR_SELECTED[@]} > 0 ));then
            # OPT_COMPONENTS_CR_SELECTED
            OPT_COMPONENTS_CR_SELECTED_UPPERCASE=()
            x=0;while [ ${x} -lt ${#OPT_COMPONENTS_CR_SELECTED[*]} ] ; do OPT_COMPONENTS_CR_SELECTED_UPPERCASE[$x]=$(tr [a-z] [A-Z] <<< ${OPT_COMPONENTS_CR_SELECTED[$x]}); let x++; done

            for host in ${OPT_COMPONENTS_CR_SELECTED_UPPERCASE[@]}; do
                FOUNDATION_DELETE_LIST=( "${FOUNDATION_DELETE_LIST[@]/$host}" )
            done
        fi

        for item in "${FOUNDATION_DELETE_LIST[@]}"; do
            if [[ "$item" == "BAS" ]];then
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration
            fi
            if [[ "$item" == "UMS" ]];then
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ums_configuration
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ums_datasource
            fi
            if [[ "$item" == "BAN" ]];then
                if [[ " ${optional_component_cr_arr[@]} " =~ "case" ]]; then
                    break
                else
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.navigator_configuration
                fi
            fi
            if [[ "$item" == "RR" ]];then
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.resource_registry_configuration
            fi
            if [[ "$item" == "AE" ]];then
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration
            fi
        done
    fi
    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_TMP} ${CP4A_PATTERN_FILE_BAK}
}

function merge_pattern(){
    # echo "length of optional_component_cr_arr:${#optional_component_cr_arr[@]}"
    # echo "!!optional_component_cr_arr!!!${optional_component_cr_arr[*]}"
    # echo "EXISTING_PATTERN_ARR: ${EXISTING_PATTERN_ARR[*]}"
    # echo "PATTERNS_CR_SELECTED: ${PATTERNS_CR_SELECTED[*]}"
    # echo "EXISTING_OPT_COMPONENT_ARR: ${EXISTING_OPT_COMPONENT_ARR[*]}"
    # echo "OPT_COMPONENTS_CR_SELECTED: ${OPT_COMPONENTS_CR_SELECTED[*]}"
    # echo "FOUNDATION_CR_SELECTED_LOWCASE: ${FOUNDATION_CR_SELECTED_LOWCASE[*]}"
    # echo "FOUNDATION_DELETE_LIST: ${FOUNDATION_DELETE_LIST[*]}"
    # echo "OPTIONAL_COMPONENT_DELETE_LIST: ${OPTIONAL_COMPONENT_DELETE_LIST[*]}"
    # echo "KEEP_COMPOMENTS: ${KEEP_COMPOMENTS[*]}"
    # echo "REMOVED FOUNDATION_CR_SELECTED FROM OPTIONAL_COMPONENT_DELETE_LIST: ${OPTIONAL_COMPONENT_DELETE_LIST[*]}"
    # echo "pattern list in CR: ${pattern_joined}"
    # echo "optional components list in CR: ${opt_components_joined}"
    # echo "length of optional_component_arr:${#optional_component_arr[@]}"

    # read -rsn1 -p"Press Enter/Return to continue (DEBUG MODEL)";echo

    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_BAK} ${CP4A_PATTERN_FILE_TMP}
    set_ldap_type_foundation
    for item in "${PATTERNS_CR_SELECTED[@]}"; do
        while true; do
            case $item in
                "content")
                    if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "content" && "${#PATTERNS_CR_SELECTED[@]}" -eq "1" ]]; then
                        ${COPY_CMD} -rf "${CONTENT_SEPARATE_PATTERN_FILE}" "${CONTENT_PATTERN_FILE_BAK}"
                    fi
                    set_ldap_type_content_pattern
                    set_external_share_content_pattern
                    set_object_store_content_pattern
                    if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "content" && "${#PATTERNS_CR_SELECTED[@]}" -eq "1" ]]; then
                        ${COPY_CMD} -rf "${CONTENT_PATTERN_FILE_BAK}" "${CP4A_PATTERN_FILE_TMP}"
                    else
                        ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${CONTENT_PATTERN_FILE_BAK}
                    fi
                    break
                    ;;
                "contentanalyzer")
                    set_aca_tenant_pattern
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.tenant_databases
                    ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${ACA_PATTERN_FILE_BAK}
                    break
                    ;;
                "decisions")
                    set_decision_feature
                    ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${DECISIONS_PATTERN_FILE_BAK}
                    break
                    ;;
                "workflow")
                    # set_ldap_type_workflow_pattern
                    if [[ "${INSTALL_BAW_ONLY}" == "Yes" ]]; then
                        # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration
                        if [[ $DEPLOYMENT_TYPE == "production" ]];then
                            # if [[ $INSTALLATION_TYPE == "existing" && (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") ]]; then
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.datasource_configuration.dc_os_datasources
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.initialize_configuration
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.bastudio_configuration
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.baw_configuration
                            # fi
                            ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WORKFLOW_PATTERN_FILE_BAK}
                        elif [[ $DEPLOYMENT_TYPE == "starter" ]]
                        then
                            # if [[ $INSTALLATION_TYPE == "existing" && (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") ]]; then
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.baw_configuration
                            # fi
                            ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WORKFLOW_PATTERN_FILE_BAK}
                            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration
                        fi
                    fi
                    break
                    ;;
                "workflow-authoring")
                    # set_ldap_type_workstreams_pattern
                    ## Remove AE data persistent for workflow authoring from 22.0.2
                    # if [[ "$AE_DATA_PERSISTENCE_ENABLE" == "Yes" ]]; then
                    #     enable_ae_data_persistence_workflow_authoring
                    # fi
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration


                    if [[ $DEPLOYMENT_TYPE == "production" ]];then
                        # if [[ $INSTALLATION_TYPE == "existing" && (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-authoring") ]]; then
                        #     ${YQ_CMD} d -i ${WORKFLOW_AUTHOR_PATTERN_FILE_BAK} spec.datasource_configuration.dc_os_datasources
                        #     ${YQ_CMD} d -i ${WORKFLOW_AUTHOR_PATTERN_FILE_BAK} spec.initialize_configuration
                        #     ${YQ_CMD} d -i ${WORKFLOW_AUTHOR_PATTERN_FILE_BAK} spec.bastudio_configuration
                        # fi
                        ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WORKFLOW_AUTHOR_PATTERN_FILE_BAK}
                    fi
                    break
                    ;;
                "workflow-runtime")
                    # set_ldap_type_workstreams_pattern
                    if [[ $DEPLOYMENT_TYPE == "production" ]];then
                        if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "workstreams" && " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-runtime" ]]; then
                            break
                        else
                            # if [[ $INSTALLATION_TYPE == "existing" ]]; then
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.baw_configuration
                            # fi
                            # if [[ " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime" ]]; then
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.datasource_configuration.dc_os_datasources
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.initialize_configuration
                            # fi
                            ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WORKFLOW_PATTERN_FILE_BAK}
                        fi
                    elif [[ $DEPLOYMENT_TYPE == "starter" ]]
                    then
                        ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WORKFLOW_PATTERN_FILE_BAK}
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration
                    fi
                    break
                    ;;
                "workstreams")
                    # set_ldap_type_workstreams_pattern
                    ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WORKSTREAMS_PATTERN_FILE_BAK}
                    break
                    ;;
                "workflow-workstreams")
                    # set_ldap_type_ww_pattern
                    # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration
                    if [[ $DEPLOYMENT_TYPE == "production" ]];then
                        if [[ $INSTALLATION_TYPE == "existing" ]]; then
                            # if [[ !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime") ]]; then
                            #     ${YQ_CMD} d -i ${WORKSTREAMS_PATTERN_FILE_BAK} spec.datasource_configuration.dc_os_datasources.[1]
                            #     ${YQ_CMD} d -i ${WORKSTREAMS_PATTERN_FILE_BAK} spec.initialize_configuration.ic_ldap_creation
                            #     ${YQ_CMD} d -i ${WORKSTREAMS_PATTERN_FILE_BAK} spec.initialize_configuration.ic_obj_store_creation.object_stores.[1]
                            #     ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WORKSTREAMS_PATTERN_FILE_BAK}
                            # elif [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && !(" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime") ]]
                            # then
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.datasource_configuration.dc_os_datasources.[3]
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.initialize_configuration.ic_ldap_creation
                            #     ${YQ_CMD} d -i ${WORKFLOW_PATTERN_FILE_BAK} spec.initialize_configuration.ic_obj_store_creation.object_stores.[3]
                            #     ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WORKFLOW_PATTERN_FILE_BAK}
                            # fi
                            ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WW_PATTERN_FILE_BAK}
                        else
                            ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WW_PATTERN_FILE_BAK}

                        fi
                    elif [[ $DEPLOYMENT_TYPE == "starter" ]]
                    then
                        ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WW_PATTERN_FILE_BAK}
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration
                    fi
                    break
                    ;;
                "application")
                    set_baa_app_designer
                    if [[ "$AE_DATA_PERSISTENCE_ENABLE" == "Yes" || " ${OPT_COMPONENTS_CR_SELECTED[@]} " =~ "ae_data_persistence" ]]; then
                        enable_ae_data_persistence_baa
                    fi
                    ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${APPLICATION_PATTERN_FILE_BAK}
                    break
                    ;;
                "digitalworker")
                    ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${ADW_PATTERN_FILE_BAK}
                    break
                    ;;
                "decisions_ads")
                    set_ads_designer_runtime
                    ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${ADS_PATTERN_FILE_BAK}
                    break
                    ;;
                "document_processing")
                    set_ldap_type_adp_pattern
                    set_aria_gpu
                    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
                        if [[ $content_os_number -gt 0 && "${pattern_cr_arr[@]}" =~ "document_processing" && (! "${pattern_cr_arr[@]}" =~ "content") ]]; then
                            set_object_store_adp_pattern
                        else
                            OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${ARIA_PATTERN_FILE_BAK} | grep -Fn 'FNOS1DS'|cut -d':' -f1)
                            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                                ${YQ_CMD} d -i ${ARIA_PATTERN_FILE_BAK} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER]
                            fi
                            OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${ARIA_PATTERN_FILE_BAK} | grep -Fn 'FNOS1DS'|cut -d':' -f1)
                            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                                ${YQ_CMD} d -i ${ARIA_PATTERN_FILE_BAK} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER]
                            fi

                        fi
                        if [[ "${pattern_cr_arr[@]}" =~ "document_processing_runtime" ]]; then
                            ${SED_COMMAND} "s/  #ecm_configuration:/  ecm_configuration:/g" ${ARIA_PATTERN_FILE_BAK}
                            ${SED_COMMAND} "s/  #  document_processing:/    document_processing:/g" ${ARIA_PATTERN_FILE_BAK} 
                            if [[ $DB_TYPE = "postgresql-edb" ]]; then
                                # Add ibm_adp_secret under document_processing
                                ${SED_COMMAND} "s/  #    ibm_adp_secret: ibm-adp-secret/      ibm_adp_secret: \"ibm-adp-secret\"/g" ${ARIA_PATTERN_FILE_BAK}
                            else
                                # Remove ibm_adp_secret if DB_TYPE is not postgresql-edb
                                ${YQ_CMD} d -i ${ARIA_PATTERN_FILE_BAK} 'document_processing.ibm_adp_secret'
                            fi

                            ${SED_COMMAND} "s/  #    cpds:/      cpds:/g" ${ARIA_PATTERN_FILE_BAK}
                            ${SED_COMMAND} "s/  #      production_setting:/        production_setting:/g" ${ARIA_PATTERN_FILE_BAK}
                            ${SED_COMMAND} "s/  #        REPO_SERVICE_URL: \"<Required>\"/          REPO_SERVICE_URL: \"<Required>\"/g" ${ARIA_PATTERN_FILE_BAK}
                        else
                            ${SED_COMMAND} "s/  #ecm_configuration:/  ecm_configuration:/g" ${ARIA_PATTERN_FILE_BAK}
                            ${SED_COMMAND} "s/  #  document_processing:/    document_processing:/g" ${ARIA_PATTERN_FILE_BAK}
                            ${SED_COMMAND} "s/  #    ibm_adp_secret: ibm-adp-secret/      ibm_adp_secret: \"ibm-adp-secret\"/g" ${ARIA_PATTERN_FILE_BAK}
                        fi
                    fi
                    ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${ARIA_PATTERN_FILE_BAK}
                    break
                    ;;
                "document_processing_runtime")
                    break
                    ;;
                "document_processing_designer")
                    break
                    ;;
                "workflow-process-service")
                    ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${WFPS_AUTHOR_PATTERN_FILE_BAK}
                    break
                    ;;
                "foundation")
                    break
                    ;;
            esac
        done
    done
}

function merge_optional_components(){
    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_BAK} ${CP4A_PATTERN_FILE_TMP}
    for item in "${OPTIONAL_COMPONENT_DELETE_LIST[@]}"; do
        while true; do
            case $item in
                "bas")
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration
                    break
                    ;;
                "ums")
                    if [[ $PLATFORM_SELECTED == "other" ]]; then
                        containsElement "bai" "${optional_component_cr_arr[@]}"
                        retVal=$?
                        if [[ $retVal -eq 1 ]]; then
                            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ums_configuration
                            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ums_datasource
                        fi
                    else
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ums_configuration
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ums_datasource
                    fi
                    break
                    ;;
                "cmis")
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.cmis
                    break
                    ;;
                "css")
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.css
                    break
                    ;;
                "es")
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.es
                    break
                    ;;
                "tm")
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.tm
                    break
                    ;;
                "ier")
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ier_configuration
                    break
                    ;;
                "iccsap")
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.iccsap_configuration
                    break
                    ;;
                "ban")
                    break
                    ;;
                "case")
                    if [[ "${DEPLOYMENT_TYPE}" == "starter" && (" ${OPTIONAL_COMPONENT_DELETE_LIST[@]} " =~ "workstreams") && (" ${OPTIONAL_COMPONENT_DELETE_LIST[@]} " =~ "content_integration") && (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams") ]]; then
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration
                    fi
                    break
                    ;;
                "workstreams")
                    if [[ "${DEPLOYMENT_TYPE}" == "starter" && (" ${OPTIONAL_COMPONENT_DELETE_LIST[@]} " =~ "case") && (" ${OPTIONAL_COMPONENT_DELETE_LIST[@]} " =~ "content_integration") && (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams") ]]; then
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration
                    fi
                    break
                    ;;
                "content_integration")
                    if [[ "${DEPLOYMENT_TYPE}" == "starter" && (" ${OPTIONAL_COMPONENT_DELETE_LIST[@]} " =~ "workstreams") && (" ${OPTIONAL_COMPONENT_DELETE_LIST[@]} " =~ "case") && (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams") ]]; then
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration
                    fi
                    break
                    ;;
                "bai")
                    if [[ (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-runtime") && (" ${PATTERNS_CR_SELECTED[@]} " =~ "workstreams") ]]; then
                        break
                    elif [[ "${DEPLOYMENT_TYPE}" == "starter" && (" ${OPT_COMPONENTS_CR_SELECTED[@]} " =~ "baml") && (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams") ]]; then
                        break
                    else
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bai_configuration
                        break
                    fi
                    ;;
                "pfs")
                    if [[ "${DEPLOYMENT_TYPE}" == "starter" && (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams") ]]; then
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.pfs_configuration
                    fi
                    break
                    ;;
                "baml")
                    if [[ "${DEPLOYMENT_TYPE}" == "starter" && (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams") ]]; then
                        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baml_configuration
                    fi
                    break
                    ;;
                "ads_designer")
                    break
                    ;;
                "ads_runtime")
                    break
                    ;;
                "decisionCenter")
                    break
                    ;;
                "decisionRunner")
                    break
                    ;;
                "decisionServerRuntime")
                    break
                    ;;
                "app_designer")
                    break
                    ;;
                "ae_data_persistence")
                    break
                    ;;
                "baw_authoring")
                    break
                    ;;
                "auto_service")
                    break
                    ;;
                "document_processing_designer")
                    break
                    ;;
                "document_processing_runtime")
                    break
                    ;;
                "wfps_authoring")
                    break
                    ;;
                "kafka")
                    break
                    ;;
                "opensearch")
                    break
                    ;;
            esac
        done
    done
    FOUNDATION_CR_SELECTED=($(echo "${foundation_component_arr[@]}" | tr ' ' '\n' | sort -u | tr '\n' ' '))
    # FOUNDATION_CR_SELECTED_LOWCASE=( "${FOUNDATION_CR_SELECTED[@],,}" )

    x=0;while [ ${x} -lt ${#FOUNDATION_CR_SELECTED[*]} ] ; do FOUNDATION_CR_SELECTED_LOWCASE[$x]=$(tr [A-Z] [a-z] <<< ${FOUNDATION_CR_SELECTED[$x]}); let x++; done
    FOUNDATION_DELETE_LIST=($(echo "${FOUNDATION_CR_SELECTED[@]}" "${FOUNDATION_FULL_ARR[@]}" | tr ' ' '\n' | sort | uniq -u))

    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_TMP} ${CP4A_PATTERN_FILE_BAK}
}

function get_existing_pattern_name(){
    existing_pattern_cr_name=""
    existing_pattern_list=""
    existing_opt_component_list=""
    existing_platform_type=""
    existing_deployment_type=""
    existing_profile_type=""
    pattern_file_bak=""
    printf "\x1B[1mProvide the path and file name to the existing custom resource (CR)?\n\x1B[0m"
    printf "\x1B[1mPress [Enter] to accept default.\n\x1B[0m"
    # printf "\x1B[1mDefault is \x1B[0m(${CP4A_PATTERN_FILE_BAK}): "
    # existing_pattern_cr_name=`${CLI_CMD} get icp4acluster|awk '{if(NR>1){if(NR==2){ arr=$1; }else{ arr=arr" "$1; }} } END{ print arr }'`

    # <https://jsw.ibm.com/browse/DBACLD-159385> - Created new variable pattern_file_bak, if user had deployed Content it will switch the file path to ibm_content_cr_final.yaml
    if [[ $CONTENT_DEPLOYED == "Yes" ]]; then
        pattern_file_bak=$FNCM_SEPARATE_PATTERN_FILE_BAK
    else
        pattern_file_bak=$CP4A_PATTERN_FILE_BAK
    fi

    while [[ $existing_pattern_cr_name == "" ]];
    do
        read -p "[Default=$pattern_file_bak]: " existing_pattern_cr_name
        : ${existing_pattern_cr_name:=$pattern_file_bak}
        if [ -f "$existing_pattern_cr_name" ]; then
            existing_cr_type=`cat $existing_pattern_cr_name | ${YQ_CMD} r - kind`
            existing_cr_type=$(echo "${existing_cr_type}" | tr '[:upper:]' '[:lower:]')
            # IF the CR kind is content , then some of the parameters are different compared to what we have for an ICP4ACluster type CR
            # The next if else block addresses this
            # For https://jsw.ibm.com/browse/DBACLD-159390
            if [[ "$existing_cr_type" == "content" ]];then
                # For content kind CR the deployment type is under content_deployment_type
                # For https://jsw.ibm.com/browse/DBACLD-159390
                existing_deployment_type=`cat $existing_pattern_cr_name | ${YQ_CMD} r - spec.content_deployment_type`
                # For content kind CR the pattern list is foundation,content
                # For https://jsw.ibm.com/browse/DBACLD-159390
                existing_pattern_list="foundation,content"
                existing_opt_component_list=""
                # Loop through the keys and values of spec.content_optional_components which will give us the list of optional components selected
                optional_components_section=$(${YQ_CMD} r "$existing_pattern_cr_name" "spec.content_optional_components")
                while IFS=: read -r key value; do
                    key=$(echo "$key" | xargs)       
                    value=$(echo "$value" | xargs)
                    if [[ "$value" == "true" ]]; then
                        if [[ -n "$existing_opt_component_list" ]]; then
                        existing_opt_component_list+=","
                        fi
                        existing_opt_component_list+="$key"
                    fi
                done <<< "$optional_components_section"
            else
                # IF the CR is generated from the from UI then the sc_deployment_type is not a key that will be in the CR , it will be under olm_deployment_type
                # For https://jsw.ibm.com/browse/DBACLD-159390
                key_value=$(${YQ_CMD} r $existing_pattern_cr_name 'spec.shared_configuration.sc_deployment_type' 2>/dev/null)
                if [[ "$key_value" ]]; then
                    existing_deployment_type=`cat $existing_pattern_cr_name | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_type`
                else
                    existing_deployment_type=`cat $existing_pattern_cr_name | ${YQ_CMD} r - spec.olm_deployment_type`
                fi
                key_value=$(${YQ_CMD} r $existing_pattern_cr_name 'spec.shared_configuration.sc_deployment_patterns' 2>/dev/null)
                if [[ "$key_value" ]]; then
                    existing_pattern_list=`cat $existing_pattern_cr_name | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_patterns`
                else
                    # IF the CR is generated from the from UI then the sc_deployment_patterns is not a key that will be in the CR , the patterns selected will be mapped to the list of keys below by being set to true for a specific pattern key
                    # For https://jsw.ibm.com/browse/DBACLD-159390
                    keys=("olm_starter_application" "olm_starter_content" "olm_starter_decisions" "olm_starter_decisions_ads" "olm_starter_document_processing" "olm_starter_workflow")
                    for key in "${keys[@]}"; do
                        # Get the value of the current key from the YAML file
                        value=$(${YQ_CMD} r "$existing_pattern_cr_name" "spec.$key")
                        value=$(echo "$value" | xargs)
                        if [[ "$value" == "true" && "$key" == olm_starter_* ]]; then
                            part_after_prefix="${key#olm_starter_}"
                            if [ -z "$existing_pattern_list" ]; then
                                existing_pattern_list="$part_after_prefix"
                            else
                                existing_pattern_list="$existing_pattern_list,$part_after_prefix"
                            fi
                        fi
                    done
                fi
                
                key_value=$(${YQ_CMD} r $existing_pattern_cr_name 'spec.shared_configuration.sc_optional_components' 2>/dev/null)
                if [[ "$key_value" ]]; then
                    existing_opt_component_list=`cat $existing_pattern_cr_name | ${YQ_CMD} r - spec.shared_configuration.sc_optional_components`
                else
                    # IF the CR is generated from the from UI then the sc_deployment_patterns is not a key that will be in the CR , the patterns selected will be mapped to the olm_starter_option section by being set to true for a specific component key
                    # For https://jsw.ibm.com/browse/DBACLD-159390
                    existing_opt_component_list=""
                    # Loop through the keys and values
                    optional_components_section=$(${YQ_CMD} r "$existing_pattern_cr_name" "spec.olm_starter_option")
                    while IFS=: read -r key value; do
                        # Extract key and value using shell parameter expansion
                        key=$(echo "$key" | xargs)       # Trim whitespace from the key
                        value=$(echo "$value" | xargs)
                        if [[ "$value" == "true" ]]; then
                            if [[ -n "$existing_opt_component_list" ]]; then
                            existing_opt_component_list+=","
                            fi
                            existing_opt_component_list+="$key"
                        fi
                    done <<< "$optional_components_section"
                fi 
            fi
            existing_platform_type=`cat $existing_pattern_cr_name | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_platform`
            existing_profile_type=`cat $existing_pattern_cr_name | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_profile_size`

            if [[ $existing_deployment_type == "demo" ]];then
                existing_deployment_type="Starter"
            elif [[ $existing_deployment_type == "enterprise" ]];then
                existing_deployment_type="Production"
            fi
            case "${existing_deployment_type}" in
                starter*|Starter*)     DEPLOYMENT_TYPE="starter";;
                production*|Production*)    DEPLOYMENT_TYPE="production";;
                *)
                    echo -e "\x1B[1;31mInvalid deployment type found in CR, exiting....\n\x1B[0m"
                    exit 0
                    ;;
            esac

            case "${existing_platform_type}" in
                ROKS*)     PLATFORM_SELECTED="ROKS";;
                OCP*)    PLATFORM_SELECTED="OCP";;
                *)
                    echo -e "\x1B[1;31mInvalid platform type found in CR, exiting....\n\x1B[0m"
                    exit 0
                    ;;
            esac
            OIFS=$IFS
            IFS=',' read -r -a EXISTING_PATTERN_ARR <<< "$existing_pattern_list"
            IFS=$OIFS

            OIFS=$IFS
            IFS=',' read -r -a EXISTING_OPT_COMPONENT_ARR <<< "$existing_opt_component_list"
            IFS=$OIFS

            FOUNDATION_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOYMENT_TYPE}_foundation.yaml
            if [[ "$existing_pattern_cr_name" == "$pattern_file_bak" ]]; then
                ${COPY_CMD} -rf "${pattern_file_bak}" "${CP4A_EXISTING_BAK}"
                ${COPY_CMD} -rf "${pattern_file_bak}" "${CP4A_EXISTING_TMP}"
            else
                ${COPY_CMD} -rf "${existing_pattern_cr_name}" "${pattern_file_bak}"
                ${COPY_CMD} -rf "${existing_pattern_cr_name}" "${CP4A_EXISTING_BAK}"
                ${COPY_CMD} -rf "${existing_pattern_cr_name}" "${CP4A_EXISTING_TMP}"
            fi
            # ${COPY_CMD} -rf "${FOUNDATION_PATTERN_FILE}" "${CP4A_PATTERN_FILE_TMP}"
            # ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_TMP} ${CP4A_PATTERN_FILE_BAK}
            # ${COPY_CMD} -rf "${CP4A_PATTERN_FILE_BAK}" "${CP4A_PATTERN_FILE_TMP}"
        else
            echo -e "\x1B[1;31m\"$existing_pattern_cr_name\" file does not exist! \n\x1B[0m"
            existing_pattern_cr_name=""
        fi
    done
    # existing_pattern_list=`${CLI_CMD} get icp4acluster $existing_pattern_cr_name -o yaml | yq r - spec.shared_configuration.sc_deployment_patterns`
    # existing_pattern_deploy_type=`${CLI_CMD} get icp4acluster $existing_pattern_cr_name -o yaml | yq r - spec.shared_configuration.sc_deployment_type`

    if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") && (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "baw_authoring") ]]; then
        EXISTING_PATTERN_ARR=( "${EXISTING_PATTERN_ARR[@]}" "workflow-authoring" )
    fi

    if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") && !(" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "baw_authoring") && ($retVal_baw -eq 1) ]]; then
        EXISTING_PATTERN_ARR=( "${EXISTING_PATTERN_ARR[@]}" "workflow-runtime" )
    fi

    if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing") && (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "document_processing_designer") ]]; then
        EXISTING_PATTERN_ARR=( "${EXISTING_PATTERN_ARR[@]}" "document_processing_designer" )
    fi

    if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing") && (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "document_processing_runtime") ]]; then
        EXISTING_PATTERN_ARR=( "${EXISTING_PATTERN_ARR[@]}" "document_processing_runtime" )
    fi

    if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-workstreams") && ("${DEPLOYMENT_TYPE}" == "production") ]]; then
        echo -e "\x1B[1;31mYou are updating existing patterns including workflow-workstreams which is not supported.\x1B[0m"
        echo -e "\x1B[1;31mRefer to the documentation to upgrade or manually add another pattern.\x1B[0m"
        echo -e "\x1B[1;31mExiting...\x1B[0m"
        read -rsn1 -p"Press Enter/Return to exit";echo
        exit 1
    fi
}

function select_objectstore_number(){
    content_os_number=""
    while true; do
        printf "\n"
        if [[ " ${pattern_cr_arr[@]}" =~ "document_processing" && $DEPLOYMENT_WITH_PROPERTY == "Yes" ]]; then
            info "One default FNCM object store \"DEVOS1\" is added to the property file. You can add more custom object stores for ADP/Content pattern."
        elif [[ " ${pattern_cr_arr[@]}" =~ "document_processing" && $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
            info "One default FNCM object store \"DEVOS1\" is added to the custom resource file. You can add more custom object stores for ADP/Content pattern."
        fi

        if [[ " ${pattern_cr_arr[@]}" =~ "document_processing" && (! " ${pattern_cr_arr[@]}" =~ "content") ]]; then
            printf "\x1B[1mHow many additional object stores will be deployed for the document processing pattern? \x1B[0m"
        elif [[ " ${pattern_cr_arr[@]}" =~ "content" && (! " ${pattern_cr_arr[@]}" =~ "document_processing") ]]; then
            printf "\x1B[1mHow many object stores will be deployed for the content pattern? \x1B[0m"
        elif [[ " ${pattern_cr_arr[@]}" =~ "document_processing" && " ${pattern_cr_arr[@]}" =~ "content" ]]; then
            printf "\x1B[1mHow many object stores will be deployed for the content pattern and how many additional object stores will be deployed for the document processing pattern? \x1B[0m"
        fi
        if [[ " ${pattern_cr_arr[@]}" =~ "document_processing" && (! " ${pattern_cr_arr[@]}" =~ "content") ]]; then
            read -rp "" content_os_number
            [[ $content_os_number =~ ^[0-9]+$ ]] || { echo -e "\x1B[1;31mEnter a valid number [0 to 10]\x1B[0m"; continue; }
            if [ "$content_os_number" -ge 0 ] && [ "$content_os_number" -le 10 ]; then
                break
            else
                echo -e "\x1B[1;31mEnter a valid number [0 to 10]\x1B[0m"
                content_os_number=""
            fi
        elif [[ " ${pattern_cr_arr[@]}" =~ "content" ]]; then
            read -rp "" content_os_number
            [[ $content_os_number =~ ^[0-9]+$ ]] || { echo -e "\x1B[1;31mEnter a valid number [1 to 10]\x1B[0m"; continue; }
            if [ "$content_os_number" -ge 1 ] && [ "$content_os_number" -le 10 ]; then
                break
            else
                echo -e "\x1B[1;31mEnter a valid number [1 to 10]\x1B[0m"
                content_os_number=""
            fi
        fi
    done
}

function select_gpu_document_processing(){
    printf "\n"
    set_gpu_enabled=""
    ENABLE_GPU_ARIA=""
    while [[ $set_gpu_enabled == "" ]];
    do
        printf "\x1B[1mAre there GPU enabled worker nodes (Yes/No)? \x1B[0m"
        read -rp "" set_gpu_enabled
        case "$set_gpu_enabled" in
        "y"|"Y"|"yes"|"Yes"|"YES")
            ENABLE_GPU_ARIA="Yes"
            break
            ;;
        "n"|"N"|"no"|"No"|"NO")
            ENABLE_GPU_ARIA="No"
            break
            ;;
        *)
            echo -e "Answer must be \"Yes\" or \"No\"\n"
            set_gpu_enabled=""
            ENABLE_GPU_ARIA=""
            ;;
        esac
    done
    if [[ "${ENABLE_GPU_ARIA}" == "Yes" ]]; then
        printf "\n"
        printf "\x1B[1mWhat is the node label key used to identify the GPU worker node(s)? \x1B[0m"
        nodelabel_key=""
        while [[ $nodelabel_key == "" ]];
        do
            read -rp "" nodelabel_key
            if [ -z "$nodelabel_key" ]; then
            echo -e "\x1B[1;31mEnter the node label key.\x1B[0m"
            fi
        done

        printf "\n"
        printf "\x1B[1mWhat is the node label value used to identify the GPU worker node(s)? \x1B[0m"
        nodelabel_value=""
        while [[ $nodelabel_value == "" ]];
        do
            read -rp "" nodelabel_value
            if [ -z "$nodelabel_value" ]; then
            echo -e "\x1B[1;31mEnter the node label value.\x1B[0m"
            fi
        done
    fi
}

function set_baa_app_designer(){
    ${COPY_CMD} -rf ${APPLICATION_PATTERN_FILE_BAK} ${APPLICATION_PATTERN_FILE_TMP}
    if [[ $DEPLOYMENT_TYPE == "starter"  ]] ;
    then
        foundation_baa=("BAS")
        foundation_component_arr=( "${foundation_component_arr[@]}" "${foundation_baa[@]}" )

    elif [[ $DEPLOYMENT_TYPE == "production" ]]
    then
        containsElement "app_designer" "${OPT_COMPONENTS_CR_SELECTED[@]}"
        retVal=$?
        if [[ $retVal -eq 0 ]]; then
            foundation_baa=("BAS")
            foundation_component_arr=( "${foundation_component_arr[@]}" "${foundation_baa[@]}" )
        fi
    fi
    ${COPY_CMD} -rf ${APPLICATION_PATTERN_FILE_TMP} ${APPLICATION_PATTERN_FILE_BAK}
}

function set_ads_designer_runtime(){
    ${COPY_CMD} -rf ${ADS_PATTERN_FILE_BAK} ${ADS_PATTERN_FILE_TMP}
    if [[ $DEPLOYMENT_TYPE == "starter"  ]] ;
    then
        ${YQ_CMD} w -i ${ADS_PATTERN_FILE_TMP} spec.ads_configuration.decision_designer.enabled "true"
        ${YQ_CMD} w -i ${ADS_PATTERN_FILE_TMP} spec.ads_configuration.decision_runtime.enabled "true"
        foundation_ads=("BAS")
        foundation_component_arr=( "${foundation_component_arr[@]}" "${foundation_ads[@]}" )

    elif [[ $DEPLOYMENT_TYPE == "production" ]]
    then
        containsElement "ads_designer" "${OPT_COMPONENTS_CR_SELECTED[@]}"
        retVal=$?
        if [[ $retVal -eq 0 ]]; then
            ${YQ_CMD} w -i ${ADS_PATTERN_FILE_TMP} spec.ads_configuration.decision_designer.enabled "true"
            foundation_ads=("BAS")
            foundation_component_arr=( "${foundation_component_arr[@]}" "${foundation_ads[@]}" )
        else
            ${YQ_CMD} w -i ${ADS_PATTERN_FILE_TMP} spec.ads_configuration.decision_designer.enabled "false"
        fi
        containsElement "ads_runtime" "${OPT_COMPONENTS_CR_SELECTED[@]}"
        retVal=$?
        if [[ $retVal -eq 0 ]]; then
            ${YQ_CMD} w -i ${ADS_PATTERN_FILE_TMP} spec.ads_configuration.decision_runtime.enabled "true"
        else
            ${YQ_CMD} w -i ${ADS_PATTERN_FILE_TMP} spec.ads_configuration.decision_runtime.enabled "false"
        fi

    fi
    ${COPY_CMD} -rf ${ADS_PATTERN_FILE_TMP} ${ADS_PATTERN_FILE_BAK}
}


function set_decision_feature(){
    ${COPY_CMD} -rf ${DECISIONS_PATTERN_FILE_BAK} ${DECISIONS_PATTERN_FILE_TMP}
    if [[ $DEPLOYMENT_TYPE == "starter"  ]] ;
    then
        ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionCenter.enabled "true"
        ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionServerRuntime.enabled "true"
        ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionRunner.enabled "true"
    elif [[ $DEPLOYMENT_TYPE == "production" ]]
    then
        containsElement "decisionCenter" "${OPT_COMPONENTS_CR_SELECTED[@]}"
        retVal=$?
        if [[ $retVal -eq 0 ]]; then
            ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionCenter.enabled "true"
        else
            ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionCenter.enabled "false"
        fi
        containsElement "decisionServerRuntime" "${OPT_COMPONENTS_CR_SELECTED[@]}"
        retVal=$?
        if [[ $retVal -eq 0 ]]; then
            ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionServerRuntime.enabled "true"
        else
            ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionServerRuntime.enabled "false"
        fi
        containsElement "decisionRunner" "${OPT_COMPONENTS_CR_SELECTED[@]}"
        retVal=$?
        if [[ $retVal -eq 0 ]]; then
            ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionRunner.enabled "true"
        else
            ${YQ_CMD} w -i ${DECISIONS_PATTERN_FILE_TMP} spec.odm_configuration.decisionRunner.enabled "false"
        fi
    fi
    ${COPY_CMD} -rf ${DECISIONS_PATTERN_FILE_TMP} ${DECISIONS_PATTERN_FILE_BAK}
}

function set_aria_gpu(){
    ${COPY_CMD} -rf ${ARIA_PATTERN_FILE_BAK} ${ARIA_PATTERN_FILE_TMP}
    if [[ ($DEPLOYMENT_TYPE == "production" && (" ${PATTERNS_CR_SELECTED[@]} " =~ "document_processing_designer")) || $DEPLOYMENT_TYPE == "starter" ]] ;
    then
        if [[ "$ENABLE_GPU_ARIA" == "Yes" ]]; then
            ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.ca_configuration.deeplearning.gpu_enabled "true"
            # ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.ca_configuration.deeplearning.nodelabel_key "$nodelabel_key"
            ${SED_COMMAND} "s|nodelabel_key:.*|nodelabel_key: \"$nodelabel_key\"|g" ${ARIA_PATTERN_FILE_TMP}
            # ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.ca_configuration.deeplearning.nodelabel_value "$nodelabel_value"
            ${SED_COMMAND} "s|nodelabel_value:.*|nodelabel_value: \"$nodelabel_value\"|g" ${ARIA_PATTERN_FILE_TMP}

        elif [[ "$ENABLE_GPU_ARIA" == "No" ]]
        then
            ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.ca_configuration.deeplearning.gpu_enabled "false"
        fi
    else
        ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.ca_configuration.deeplearning.gpu_enabled "false"
    fi

    if [[ $DEPLOYMENT_TYPE == "starter" ]] ;
    then
        if [[ "$ADP_DL_ENABLED" == "Yes" ]]; then
            ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.ca_configuration.ocrextraction.deep_learning_object_detection.enabled "true"
        elif [[ "$ADP_DL_ENABLED" == "No" ]]
        then
            ${YQ_CMD} w -i ${ARIA_PATTERN_FILE_TMP} spec.ca_configuration.ocrextraction.deep_learning_object_detection.enabled "false"
        fi
    fi
    ${COPY_CMD} -rf ${ARIA_PATTERN_FILE_TMP} ${ARIA_PATTERN_FILE_BAK}
}

function get_oracle_service_name(){
    local JDBC_URL=$1

    JDBC_URL=$(sed -e 's/^"//' -e 's/"$//' <<<"$JDBC_URL")
    # Parse format is "jdbc:oracle:thin:@(DESCRIPTION=(ADDRESS=(PROTOCOL=TCPS)(HOST=oracle21c1.fyre.ibm.com)(PORT=<your-database-port>))(CONNECT_DATA=(SERVICE_NAME=ORCLPDB1)))"
    if [[ "$JDBC_URL" == *"SERVICE_NAME="* || "$JDBC_URL" == *"service_name="* ]]; then
        if [[ "$machine" == "Mac" ]]; then
            local tmp_val_array=()
            local tmp_svc_array=()
            OIFS=$IFS
            IFS=')' read -ra tmp_val_array <<< "$JDBC_URL"
            IFS=$OIFS
            for item in "${tmp_val_array[@]}"
            do
                if [[ $item == *"SERVICE_NAME="* || $item == *"service_name="* ]]; then
                    OIFS=$IFS
                    IFS='=' read -ra tmp_svc_array <<< "$item"
                    IFS=$OIFS
                    ORACLE_SERVICE_NAME=${tmp_svc_array[${#tmp_svc_array[@]}-1]}
                fi
            done
        else
            ORACLE_SERVICE_NAME=$(echo "$JDBC_URL" | grep -oP '(?<=SERVICE_NAME=).*(?=)')
            if [[ -z "$ORACLE_SERVICE_NAME" ]]; then
                ORACLE_SERVICE_NAME=$(echo "$JDBC_URL" | grep -oP '(?<=service_name=).*(?=)')
            fi
            ORACLE_SERVICE_NAME=$(sed -e 's/^(//' -e 's/)$//' <<<"$ORACLE_SERVICE_NAME")

            until [[ "$ORACLE_SERVICE_NAME" != *")"* ]]; do
                ORACLE_SERVICE_NAME=$(sed -e 's/^(//' -e 's/)$//' <<<"$ORACLE_SERVICE_NAME")
            done
        fi
    else
        # Parse format is "jdbc:oracle:thin:@//oracle_server:1521/orcl"
        ORACLE_SERVICE_NAME=$(echo "$JDBC_URL" | cut -d"/" -f4)
    fi
    ORACLE_SERVICE_NAME=$(echo $ORACLE_SERVICE_NAME | tr '[:lower:]' '[:upper:]')
}

function sync_property_into_final_cr(){
    printf "\n"

    wait_msg "Applying value in property file into final CR"

    # Applying global value in user profile property into final CR
    tmp_value="$(prop_user_profile_property_file CP4BA.CP4BA_LICENSE)"
    ${SED_COMMAND} "s|sc_deployment_license:.*|sc_deployment_license: \"$tmp_value\"|g" ${CP4A_PATTERN_FILE_TMP}

    if [[ " ${pattern_cr_arr[@]}" =~ "workflow-runtime" || " ${pattern_cr_arr[@]}" =~ "workflow-authoring" || " ${pattern_cr_arr[@]}" =~ "workstreams" || " ${pattern_cr_arr[@]}" =~ "content" || " ${pattern_cr_arr[@]}" =~ "document_processing" || "${optional_component_cr_arr[@]}" =~ "ae_data_persistence" ]]; then
        tmp_value="$(prop_user_profile_property_file CP4BA.FNCM_LICENSE)"
        ${SED_COMMAND} "s|sc_deployment_fncm_license:.*|sc_deployment_fncm_license: \"$tmp_value\"|g" ${CP4A_PATTERN_FILE_TMP}

        # sc_deployment_baw_license required for either workflow runtime or workflow authoring
        # For https://jsw.ibm.com/browse/DBACLD-161792
        if [[ " ${pattern_cr_arr[@]}" =~ "workflow-runtime" || " ${pattern_cr_arr[@]}" =~ "workflow-authoring" ]]; then
            tmp_value="$(prop_user_profile_property_file CP4BA.BAW_LICENSE)"
            ${SED_COMMAND} "s|sc_deployment_baw_license:.*|sc_deployment_baw_license: \"$tmp_value\"|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_baw_license
        fi
        # Applying value in GCDDB property file into final CR
        tmp_gcd_db_servername="$(prop_db_name_user_property_file_for_server_name GCD_DB_USER_NAME)"
        tmp_gcd_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_gcd_db_servername")

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_gcd_db_name="$(prop_db_name_user_property_file GCD_DB_USER_NAME)"
        else
            tmp_gcd_db_name="$(prop_db_name_user_property_file GCD_DB_NAME)"
        fi
        tmp_gcd_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_gcd_db_name")
        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_gcd_db_name=$(echo $tmp_gcd_db_name | tr '[:upper:]' '[:lower:]')
        fi
        
        #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
        # Initialize the isfalse variable to validate spec.datasource_configuration.dc_ssl_enabled is true or false
        isfalse=false
        for i in "${!GCDDB_CR_MAPPING[@]}"; do
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${GCDDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_gcd_db_servername.${GCDDB_COMMON_PROPERTY[i]})\""
                 # Check if we are updating spec.datasource_configuration.dc_ssl_enabled value is False
                if [ "${GCDDB_CR_MAPPING[i]}" == "spec.datasource_configuration.dc_ssl_enabled" ] && [[ "$(prop_db_server_property_file $tmp_gcd_db_servername.${GCDDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                    # Set isfalse to true if the value is False
                    isfalse=true
                fi
                # Check if we are updating spec.datasource_configuration.dc_gcd_datasource.database_ssl_secret_name
                if [ "${GCDDB_CR_MAPPING[i]}" == "spec.datasource_configuration.dc_gcd_datasource.database_ssl_secret_name" ]; then
                    if [ "$isfalse" == "true" ]; then
                        # If isfalse is true, set the value to ""
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_gcd_datasource.database_ssl_secret_name" "\"\""
                    fi
                fi
        done

        # remove database_name if oracle
        if [[ $DB_TYPE == "oracle" ]]; then
            get_oracle_service_name $(prop_db_server_property_file $tmp_gcd_db_servername.ORACLE_JDBC_URL)
            if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_gcd_datasource.database_name "\"${ORACLE_SERVICE_NAME}GCD\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_gcd_datasource.database_name "\"<Required>\""
            fi
            # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_gcd_datasource.database_name "\"$tmp_gcd_db_name\""
            if [[ $DB_TYPE == "postgresql-edb" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_gcd_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_gcd_datasource.dc_use_postgres "true"
            fi
        fi

        # Apply customized schema for GCDDB
        tmp_schema_name=$(prop_db_name_user_property_file GCD_DB_CURRENT_SCHEMA)
        tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
        if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
            if [[ $DB_TYPE == "postgresql" ]]; then
                tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
            fi
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.cpe_production_setting.gcd_schema  "\"${tmp_schema_name}\""
        fi

        # Applying user profile for CONTENT INITIONLIZATION
        tmp_init_flag="$(prop_user_profile_property_file CONTENT_INITIALIZATION.ENABLE)"
        tmp_init_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_init_flag")
        if [[ $tmp_init_flag == "Yes" || $tmp_init_flag == "YES" || $tmp_init_flag == "Y" || $tmp_init_flag == "True" || $tmp_init_flag == "true" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_content_initialization "true"

            # Set initialize_configuration.ic_ldap_creation
            tmp_admin_user_name=$(prop_user_profile_property_file CONTENT_INITIALIZATION.LDAP_ADMIN_USER_NAME)
            tmp_admin_user_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_admin_user_name")

            tmp_admin_group_name=$(prop_user_profile_property_file CONTENT_INITIALIZATION.LDAP_ADMINS_GROUPS_NAME)
            tmp_admin_group_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_admin_group_name")

            OIFS=$IFS
            IFS=',' read -ra admin_user_name_array <<< "$tmp_admin_user_name"
            IFS=',' read -ra admin_group_name_array <<< "$tmp_admin_group_name"
            IFS=$OIFS

            for num in "${!admin_user_name_array[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_ldap_creation.ic_ldap_admin_user_name.[$((num))] "\"${admin_user_name_array[num]}\""
            done

            for num in "${!admin_group_name_array[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_ldap_creation.ic_ldap_admins_groups_name.[$((num))] "\"${admin_group_name_array[num]}\""
            done
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_content_initialization "false"
        fi
    else
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_fncm_license
    fi

    # Applying value in FNCM OSDB property file into final CR
    if (( content_os_number > 0 )); then
        for ((j=0;j<${content_os_number};j++)); do
            tmp_os_db_servername="$(prop_db_name_user_property_file_for_server_name OS$((j+1))_DB_USER_NAME)"
            tmp_os_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_servername")

            if [[ $DB_TYPE == "oracle" ]]; then
                tmp_os_db_name="$(prop_db_name_user_property_file OS$((j+1))_DB_USER_NAME)"
            else
                tmp_os_db_name="$(prop_db_name_user_property_file OS$((j+1))_DB_NAME)"
            fi
            # tmp_os_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_name")
            if [[ $DB_TYPE == "postgresql" ]]; then
                tmp_os_db_name=$(echo $tmp_os_db_name | tr '[:upper:]' '[:lower:]')
            fi

            OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn "FNOS$((j+1))DS"|cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
                # Initialize the isfalse variable to validate dc_ssl_enabled is true or false
                isfalse=false
                for i in "${!OSDB_CR_MAPPING[@]}"; do
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].${OSDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})\""
                    # Check if we are updating database_ssl_enable value is False
                    if [ "${OSDB_CR_MAPPING[i]}" == "database_ssl_enable" ] && [[ "$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                        isfalse=true
                    fi
                    # Check if we are updating database_ssl_secret_name
                    if [ "${OSDB_CR_MAPPING[i]}" == "database_ssl_secret_name" ]; then
                        if [ "$isfalse" == "true" ]; then
                            # If isfalse is true, set the value to ""
                            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_ssl_secret_name" "\"\""
                        fi
                    fi
                done
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_os_label "\"os$((j+1))\""

                # remove database_name if oracle
                if [[ $DB_TYPE == "oracle" ]]; then
                    get_oracle_service_name $(prop_db_server_property_file $tmp_os_db_servername.ORACLE_JDBC_URL)
                    if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"${ORACLE_SERVICE_NAME}OS$((j+1))\""
                    else
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"<Required>\""
                    fi
                    # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name $tmp_os_db_name
                    if [[ $DB_TYPE == "postgresql-edb" ]]; then
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_database_type "postgresql"
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_use_postgres "true"
                    fi
                fi
            fi
        done

        # apply oc_cpe_obj_store_admin_user_groups for FNCM OS
        for ((j=0;j<${content_os_number};j++))
        do
            if [[ $DB_TYPE == "oracle" ]]; then
                tmp_os_db_name="$(prop_db_name_user_property_file OS$((j+1))_DB_USER_NAME)"
            else
                tmp_os_db_name="$(prop_db_name_user_property_file OS$((j+1))_DB_NAME)"
            fi
            OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn FNOS$((j+1))DS|cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                tmp_user_group=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_ADMIN_USER_GROUPS)
                tmp_user_group=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_user_group")
                OIFS=$IFS
                IFS=',' read -ra admin_user_group_array <<< "$tmp_user_group"
                IFS=$OIFS

                for num in "${!admin_user_group_array[@]}"; do
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_admin_user_groups.[$((num))]  "\"${admin_user_group_array[num]}\""
                done
                # Apply customized schema
                # if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
                tmp_schema_name=$(prop_db_name_user_property_file OS$((j+1))_DB_CURRENT_SCHEMA)
                tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
                if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
                    if [[ $DB_TYPE == "postgresql" ]]; then
                        tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
                    fi
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_schema_name  "\"${tmp_schema_name}\""
                fi
                # fi
                tmp_table_storage_location_prop="$(prop_db_name_user_property_file OS$((j+1))_DB_TABLE_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
                tmp_index_storage_location_prop="$(prop_db_name_user_property_file OS$((j+1))_DB_INDEX_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
                tmp_lob_storage_location_prop="$(prop_db_name_user_property_file OS$((j+1))_DB_LOB_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//')"
                # Function to populate the os tablespace section for table index and lob storage
                # Function definition in common.sh
                # https://jsw.ibm.com/browse/DBACLD-175710
                populate_os_tablespaces "$tmp_os_db_name" "$DB_TYPE" "$OS_DATASOURCE_NUMBER" "$tmp_table_storage_location_prop" "$tmp_index_storage_location_prop" "$tmp_lob_storage_location_prop"
                
                ## End of custom tables,index,lob tablespaces


            fi
        done

    fi
    # Apply value in FNCM OS required by BAW authoring property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "workflow-authoring" ]]; then
        for i in "${!BAW_AUTH_OS_ARR[@]}"; do
            OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn ${BAW_AUTH_OS_ARR[i]}|cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                tmp_os_db_servername="$(prop_db_name_user_property_file_for_server_name ${BAW_AUTH_OS_ARR[i]}_DB_USER_NAME)"
                tmp_os_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_servername")
                if [[ $DB_TYPE == "oracle" ]]; then
                    tmp_os_db_name="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_USER_NAME)"
                else
                    tmp_os_db_name="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_NAME)"
                fi
                # tmp_os_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_name")
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_os_db_name=$(echo $tmp_os_db_name | tr '[:upper:]' '[:lower:]')
                fi
                #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
                # Initialize the isfalse variable to validate dc_ssl_enabled is true or false
                isfalse=false
                for j in "${!OSDB_CR_MAPPING[@]}"; do
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].${OSDB_CR_MAPPING[j]}" "\"$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[j]})\""
                    # Check if we are updating database_ssl_enable value is False
                    if [ "${OSDB_CR_MAPPING[j]}" == "database_ssl_enable" ] && [[ "$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[j]})" =~ ^[fF]alse$ ]]; then
                        isfalse=true
                    fi
                    # Check if we are updating database_ssl_secret_name
                    if [ "${OSDB_CR_MAPPING[j]}" == "database_ssl_secret_name" ]; then
                        if [ "$isfalse" == "true" ]; then
                            # If isfalse is true, set the value to ""
                            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_ssl_secret_name" "\"\""
                        fi
                    fi
                done

                tmp_label=$(echo ${BAW_AUTH_OS_ARR[i]} | tr '[:upper:]' '[:lower:]')
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_os_label "\"$tmp_label\""
                # remove database_name if oracle
                if [[ $DB_TYPE == "oracle" ]]; then
                    get_oracle_service_name $(prop_db_server_property_file $tmp_os_db_servername.ORACLE_JDBC_URL)
                    if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"${ORACLE_SERVICE_NAME}${BAW_AUTH_OS_ARR[i]}\""
                    else
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"<Required>\""
                    fi
                    # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name $tmp_os_db_name
                    if [[ $DB_TYPE == "postgresql-edb" ]]; then
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_database_type "postgresql"
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_use_postgres "true"
                    fi
                fi
            fi
            # Apply customized schema
            OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn ${BAW_AUTH_OS_ARR[i]} |cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                # if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
                tmp_schema_name=$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_CURRENT_SCHEMA)
                tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
                if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
                    if [[ $DB_TYPE == "postgresql" ]]; then
                        tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
                    fi
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_schema_name  "\"${tmp_schema_name}\""
                fi

                ## Applying custom table, index, log tablespaces for objectstore creation.
                ## Retrieving the tables,index, and lob storage location from the properties files
                ## to be passed to the helper functions to create the sql files.
                tmp_table_storage_location_prop="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_TABLE_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
                tmp_index_storage_location_prop="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_INDEX_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
                tmp_lob_storage_location_prop="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_LOB_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//')"
                # Function to populate the os tablespace section for table index and lob storage
                # Function definition in common.sh
                # https://jsw.ibm.com/browse/DBACLD-175710
                populate_os_tablespaces "$tmp_os_db_name" "$DB_TYPE" "$OS_DATASOURCE_NUMBER" "$tmp_table_storage_location_prop" "$tmp_index_storage_location_prop" "$tmp_lob_storage_location_prop"
                
            fi
        done

        # apply oc_cpe_obj_store_admin_user_groups for FNCM OS used by BAW authoring
        for i in "${!BAW_AUTH_OS_ARR[@]}"; do
            OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn ${BAW_AUTH_OS_ARR[i]}|cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                tmp_user_group=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_ADMIN_USER_GROUPS)
                tmp_user_group=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_user_group")
                OIFS=$IFS
                IFS=',' read -ra admin_user_group_array <<< "$tmp_user_group"
                IFS=$OIFS

                for num in "${!admin_user_group_array[@]}"; do
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_admin_user_groups.[$((num))]  "\"${admin_user_group_array[num]}\""
                done
            fi
        done

        # apply property for workflow initionlization into final cr
        if [[ "${BAW_AUTH_OS_ARR[i]}" == "BAWTOS" ]]; then
            tmp_workflow_flag=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_ENABLE_WORKFLOW)
            tmp_workflow_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_workflow_flag")
            if [[ $tmp_workflow_flag == "Yes" || $tmp_workflow_flag == "YES" || $tmp_workflow_flag == "Y" || $tmp_workflow_flag == "True" || $tmp_workflow_flag == "true" ]]; then
                tmp_val=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_WORKFLOW_DATA_TBL_SPACE)
                tmp_val=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_val")
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_val=$(echo $tmp_val | tr '[:upper:]' '[:lower:]')
                elif [[ $DB_TYPE == "oracle" ]]; then
                    tmp_val=$(echo $tmp_val | tr '[:lower:]' '[:upper:]')
                fi
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_workflow_data_tbl_space  "\"$tmp_val\""

                tmp_val=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_WORKFLOW_ADMIN_GROUP)
                tmp_val=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_val")
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_workflow_admin_group  "\"$tmp_val\""

                tmp_val=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_WORKFLOW_CONFIG_GROUP)
                tmp_val=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_val")
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_workflow_config_group  "\"$tmp_val\""

                tmp_val=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_WORKFLOW_PE_CONN_POINT_NAME)
                tmp_val=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_val")
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_workflow_pe_conn_point_name  "\"$tmp_val\""
            fi
        fi

    fi

    # Apply value in FNCM OS required by BAW runtime property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "workflow-runtime" ]]; then
        BAW_RUNTIME_OS_ARR=("BAWINS1DOCS" "BAWINS1DOS" "BAWINS1TOS")
        for i in "${!BAW_AUTH_OS_ARR[@]}"; do
            OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn ${BAW_RUNTIME_OS_ARR[i]}|cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                tmp_os_db_servername="$(prop_db_name_user_property_file_for_server_name ${BAW_AUTH_OS_ARR[i]}_DB_USER_NAME)"
                tmp_os_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_servername")
                if [[ $DB_TYPE == "oracle" ]]; then
                    tmp_os_db_name="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_USER_NAME)"
                else
                    tmp_os_db_name="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_NAME)"
                fi
                # tmp_os_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_name")
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_os_db_name=$(echo $tmp_os_db_name | tr '[:upper:]' '[:lower:]')
                fi
                #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
                # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for BAWDOCS
                isfalse=false
                for j in "${!OSDB_CR_MAPPING[@]}"; do
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].${OSDB_CR_MAPPING[j]}" "\"$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[j]})\""
                # Check if we are updating database_ssl_enable value to False
                if [ "${OSDB_CR_MAPPING[j]}" == "database_ssl_enable" ] && [[ "$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[j]})" =~ ^[fF]alse$ ]]; then
                    isfalse=true
                fi
                # Check if we are updating database_ssl_secret_name
                if [ "${OSDB_CR_MAPPING[j]}" == "database_ssl_secret_name" ]; then
                    if [ "$isfalse" == "true" ]; then
                        # If isfalse is true, set the value to ""
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_ssl_secret_name" "\"\""
                    fi
                fi
                done

                tmp_label=$(echo ${BAW_AUTH_OS_ARR[i]} | tr '[:upper:]' '[:lower:]')

                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_os_label "\"$tmp_label\""

                # remove database_name if oracle
                if [[ $DB_TYPE == "oracle" ]]; then
                    get_oracle_service_name $(prop_db_server_property_file $tmp_os_db_servername.ORACLE_JDBC_URL)
                    if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"${ORACLE_SERVICE_NAME}${BAW_AUTH_OS_ARR[i]}\""
                    else
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"<Required>\""
                    fi
                    # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name $tmp_os_db_name
                    if [[ $DB_TYPE == "postgresql-edb" ]]; then
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_database_type "postgresql"
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_use_postgres "true"
                    fi
                fi
            fi

            # Apply customized schema
            OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn ${BAW_RUNTIME_OS_ARR[i]} |cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                # if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
                tmp_schema_name=$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_CURRENT_SCHEMA)
                tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
                if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
                    if [[ $DB_TYPE == "postgresql" ]]; then
                        tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
                    fi
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_schema_name  "\"${tmp_schema_name}\""
                fi
                # fi
            fi
            ## Applying custom table, index, log tablespaces for objectstore creation.
            ## Retrieving the tables,index, and lob storage location from the properties files
            ## to be passed to the helper functions to create the sql files.
            OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn ${BAW_RUNTIME_OS_ARR[i]} |cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                tmp_table_storage_location_prop="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_TABLE_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
                tmp_index_storage_location_prop="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_INDEX_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
                tmp_lob_storage_location_prop="$(prop_db_name_user_property_file ${BAW_AUTH_OS_ARR[i]}_DB_LOB_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//')"
                # Function to populate the os tablespace section for table index and lob storage
                # Function definition in common.sh
                # https://jsw.ibm.com/browse/DBACLD-175710
                populate_os_tablespaces "$tmp_os_db_name" "$DB_TYPE" "$OS_DATASOURCE_NUMBER" "$tmp_table_storage_location_prop" "$tmp_index_storage_location_prop" "$tmp_lob_storage_location_prop"
            fi

        done

        # apply oc_cpe_obj_store_admin_user_groups for FNCM OS used by BAW runtime
        for i in "${!BAW_RUNTIME_OS_ARR[@]}"; do
            OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn ${BAW_RUNTIME_OS_ARR[i]}|cut -d':' -f1)
            if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
                OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
                tmp_user_group=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_ADMIN_USER_GROUPS)
                tmp_user_group=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_user_group")
                OIFS=$IFS
                IFS=',' read -ra admin_user_group_array <<< "$tmp_user_group"
                IFS=$OIFS

                for num in "${!admin_user_group_array[@]}"; do
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_admin_user_groups.[$((num))]  "\"${admin_user_group_array[num]}\""
                done
            fi
            # apply property for workflow initionlization into final cr
            if [[ "${BAW_RUNTIME_OS_ARR[i]}" == "BAWINS1TOS" ]]; then
                tmp_workflow_flag=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_ENABLE_WORKFLOW)
                tmp_workflow_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_workflow_flag")
                if [[ $tmp_workflow_flag == "Yes" || $tmp_workflow_flag == "YES" || $tmp_workflow_flag == "Y" || $tmp_workflow_flag == "True" || $tmp_workflow_flag == "true" ]]; then
                    tmp_val=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_WORKFLOW_DATA_TBL_SPACE)
                    tmp_val=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_val")
                    if [[ $DB_TYPE == "postgresql" ]]; then
                        tmp_val=$(echo $tmp_val | tr '[:upper:]' '[:lower:]')
                    elif [[ $DB_TYPE == "oracle" ]]; then
                        tmp_val=$(echo $tmp_val | tr '[:lower:]' '[:upper:]')
                    fi
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_workflow_data_tbl_space  "\"$tmp_val\""

                    tmp_val=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_WORKFLOW_ADMIN_GROUP)
                    tmp_val=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_val")
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_workflow_admin_group  "\"$tmp_val\""

                    tmp_val=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_WORKFLOW_CONFIG_GROUP)
                    tmp_val=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_val")
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_workflow_config_group  "\"$tmp_val\""

                    tmp_val=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_WORKFLOW_PE_CONN_POINT_NAME)
                    tmp_val=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_val")
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_workflow_pe_conn_point_name  "\"$tmp_val\""
                fi
            fi
        done

    fi

    # Apply value in FNCM OS required by AWS property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "workstreams" ]]; then
        OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AWSINS1DOCS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_os_db_servername="$(prop_db_name_user_property_file_for_server_name AWSDOCS_DB_USER_NAME)"
            tmp_os_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_servername")
            if [[ $DB_TYPE == "oracle" ]]; then
                tmp_os_db_name="$(prop_db_name_user_property_file AWSDOCS_DB_USER_NAME)"
            else
                tmp_os_db_name="$(prop_db_name_user_property_file AWSDOCS_DB_NAME)"
            fi
            # tmp_os_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_name")
            if [[ $DB_TYPE == "postgresql" ]]; then
                tmp_os_db_name=$(echo $tmp_os_db_name | tr '[:upper:]' '[:lower:]')
            fi
            #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
            # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for AWSDOCS
            isfalse=false
            for i in "${!OSDB_CR_MAPPING[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].${OSDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})\""
            # Check if we are updating database_ssl_enable value is False
            if [ "${OSDB_CR_MAPPING[i]}" == "database_ssl_enable" ] && [[ "$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                isfalse=true
            fi
            # Check if we are updating database_ssl_secret_name
            if [ "${OSDB_CR_MAPPING[i]}" == "database_ssl_secret_name" ]; then
                if [ "$isfalse" == "true" ]; then
                    # If isfalse is true, set the value to ""
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_ssl_secret_name" "\"\""
                fi
            fi
            done

            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_os_label "\"awsdocs\""

            # remove database_name if oracle
            if [[ $DB_TYPE == "oracle" ]]; then
                get_oracle_service_name $(prop_db_server_property_file $tmp_os_db_servername.ORACLE_JDBC_URL)
                if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"${ORACLE_SERVICE_NAME}AWSDOCS\""
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"<Required>\""
                fi
                # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name $tmp_os_db_name
                if [[ $DB_TYPE == "postgresql-edb" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_database_type "postgresql"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_use_postgres "true"
                fi
            fi
        fi

        # Apply customized schema
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AWSINS1DOCS' |cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            # if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
            tmp_schema_name=$(prop_db_name_user_property_file AWSDOCS_DB_CURRENT_SCHEMA)
            tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
            if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
                fi
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_schema_name  "\"${tmp_schema_name}\""
            fi
            # fi
        fi

        # apply oc_cpe_obj_store_admin_user_groups for FNCM OS used by workstreams
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AWSINS1DOCS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_user_group=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_ADMIN_USER_GROUPS)
            tmp_user_group=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_user_group")
            OIFS=$IFS
            IFS=',' read -ra admin_user_group_array <<< "$tmp_user_group"
            IFS=$OIFS

            for num in "${!admin_user_group_array[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_admin_user_groups.[$((num))]  "\"${admin_user_group_array[num]}\""
            done
        fi


        ## Applying custom table, index, log tablespaces for objectstore creation.
        ## Retrieving the tables,index, and lob storage location from the properties files
        ## to be passed to the helper functions to create the sql files.
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AWSINS1DOCS' |cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_table_storage_location_prop="$(prop_db_name_user_property_file AWSDOCS_DB_TABLE_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
            tmp_index_storage_location_prop="$(prop_db_name_user_property_file AWSDOCS_DB_INDEX_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
            tmp_lob_storage_location_prop="$(prop_db_name_user_property_file AWSDOCS_DB_LOB_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//')"
            # Function to populate the os tablespace section for table index and lob storage
            # Function definition in common.sh
            # https://jsw.ibm.com/browse/DBACLD-175710
            populate_os_tablespaces "$tmp_os_db_name" "$DB_TYPE" "$OS_DATASOURCE_NUMBER" "$tmp_table_storage_location_prop" "$tmp_index_storage_location_prop" "$tmp_lob_storage_location_prop"
        fi


    fi

    # Apply value in DEVOS1 property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "document_processing" ]]; then
        OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'DEVOS1DS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_os_db_servername="$(prop_db_name_user_property_file_for_server_name DEVOS_DB_USER_NAME)"
            tmp_os_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_servername")

            if [[ $DB_TYPE == "oracle" ]]; then
                tmp_os_db_name="$(prop_db_name_user_property_file DEVOS_DB_USER_NAME)"
            else
                tmp_os_db_name="$(prop_db_name_user_property_file DEVOS_DB_NAME)"
            fi
            # tmp_os_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_name")
            if [[ $DB_TYPE == "postgresql" ]]; then
                tmp_os_db_name=$(echo $tmp_os_db_name | tr '[:upper:]' '[:lower:]')
            fi
            #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
            # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for DEVOS1 
            isfalse=false
            for i in "${!OSDB_CR_MAPPING[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].${OSDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})\""
                # Check if we are updating database_ssl_enable value is False
            if [ "${OSDB_CR_MAPPING[i]}" == "database_ssl_enable" ] && [[ "$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                isfalse=true
            fi
            # Check if we are updating database_ssl_secret_name
            if [ "${OSDB_CR_MAPPING[i]}" == "database_ssl_secret_name" ]; then
                if [ "$isfalse" == "true" ]; then
                    # If isfalse is true, set the value to ""
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_ssl_secret_name" "\"\""
                fi
            fi
            done
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_os_label "\"devos1\""
            # remove database_name if oracle
            if [[ $DB_TYPE == "oracle" ]]; then
                get_oracle_service_name $(prop_db_server_property_file $tmp_os_db_servername.ORACLE_JDBC_URL)
                if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"${ORACLE_SERVICE_NAME}DEVOS\""
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"<Required>\""
                fi
                # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name $tmp_os_db_name
                if [[ $DB_TYPE == "postgresql-edb" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_database_type "postgresql"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_use_postgres "true"
                fi
            fi
        fi
        # apply oc_cpe_obj_store_admin_user_groups for FNCM OS used by DEVOS1
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'DEVOS1DS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_user_group=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_ADMIN_USER_GROUPS)
            tmp_user_group=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_user_group")
            OIFS=$IFS
            IFS=',' read -ra admin_user_group_array <<< "$tmp_user_group"
            IFS=$OIFS
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_admin_user_groups
            for num in "${!admin_user_group_array[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_admin_user_groups.[$((num))]  "\"${admin_user_group_array[num]}\""
            done
        fi

        # Apply customized schema
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'DEVOS1DS' |cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            # if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
            tmp_schema_name=$(prop_db_name_user_property_file DEVOS_DB_CURRENT_SCHEMA)
            tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
            if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
                fi
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_schema_name  "\"${tmp_schema_name}\""
            fi
            # fi
        fi

        ## Applying custom table, index, log tablespaces for objectstore creation.
        ## Retrieving the tables,index, and lob storage location from the properties files
        ## to be passed to the helper functions to create the sql files.
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'DEVOS1DS' |cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_table_storage_location_prop="$(prop_db_name_user_property_file DEVOS_DB_TABLE_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
            tmp_index_storage_location_prop="$(prop_db_name_user_property_file DEVOS_DB_INDEX_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
            tmp_lob_storage_location_prop="$(prop_db_name_user_property_file DEVOS_DB_LOB_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//')"
            # Function to populate the os tablespace section for table index and lob storage
            # Function definition in common.sh
            # https://jsw.ibm.com/browse/DBACLD-175710
            populate_os_tablespaces "$tmp_os_db_name" "$DB_TYPE" "$OS_DATASOURCE_NUMBER" "$tmp_table_storage_location_prop" "$tmp_index_storage_location_prop" "$tmp_lob_storage_location_prop"
        fi

    fi

    # Apply value in FNCM OS required by AE data persistent property file into final CR
    if [[ "${optional_component_cr_arr[@]}" =~ "ae_data_persistence" ]]; then
        OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AEOS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_os_db_servername="$(prop_db_name_user_property_file_for_server_name AEOS_DB_USER_NAME)"
            tmp_os_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_servername")

            if [[ $DB_TYPE == "oracle" ]]; then
                tmp_os_db_name="$(prop_db_name_user_property_file AEOS_DB_USER_NAME)"
            else
                tmp_os_db_name="$(prop_db_name_user_property_file AEOS_DB_NAME)"
            fi
            # tmp_os_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_name")
            if [[ $DB_TYPE == "postgresql" ]]; then
                tmp_os_db_name=$(echo $tmp_os_db_name | tr '[:upper:]' '[:lower:]')
            fi
            #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
            # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for AEOS
            isfalse=false
            for i in "${!OSDB_CR_MAPPING[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].${OSDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})\""
            # Check if we are updating database_ssl_enable value is False
            if [ "${OSDB_CR_MAPPING[i]}" == "database_ssl_enable" ] && [[ "$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                isfalse=true
            fi
            # Check if we are updating database_ssl_secret_name
            if [ "${OSDB_CR_MAPPING[i]}" == "database_ssl_secret_name" ]; then
                if [ "$isfalse" == "true" ]; then
                    # If isfalse is true, set the value to ""
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_ssl_secret_name" "\"\""
                fi
            fi
            done
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_os_label "\"aeos\""
            # remove database_name if oracle
            if [[ $DB_TYPE == "oracle" ]]; then
                get_oracle_service_name $(prop_db_server_property_file $tmp_os_db_servername.ORACLE_JDBC_URL)
                if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"${ORACLE_SERVICE_NAME}AEOS\""
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"<Required>\""
                fi
                # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name $tmp_os_db_name
                if [[ $DB_TYPE == "postgresql-edb" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_database_type "postgresql"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_use_postgres "true"
                fi
            fi
        fi

        # Apply customized schema
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AEOS' |cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            # if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
            tmp_schema_name=$(prop_db_name_user_property_file AEOS_DB_CURRENT_SCHEMA)
            tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
            if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
                fi
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_schema_name  "\"${tmp_schema_name}\""
            fi
            # fi
        fi

        # apply oc_cpe_obj_store_admin_user_groups for FNCM OS used by AE data persistent
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AEOS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_user_group=$(prop_user_profile_property_file CONTENT_INITIALIZATION.CPE_OBJ_STORE_ADMIN_USER_GROUPS)
            tmp_user_group=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_user_group")
            OIFS=$IFS
            IFS=',' read -ra admin_user_group_array <<< "$tmp_user_group"
            IFS=$OIFS

            for num in "${!admin_user_group_array[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_admin_user_groups.[$((num))]  "\"${admin_user_group_array[num]}\""
            done
        fi

        ## Applying custom table, index, log tablespaces for objectstore creation.
        ## Retrieving the tables,index, and lob storage location from the properties files
        ## to be passed to the helper functions to create the sql files.
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AEOS' |cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_table_storage_location_prop="$(prop_db_name_user_property_file AEOS_DB_TABLE_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
            tmp_index_storage_location_prop="$(prop_db_name_user_property_file AEOS_DB_INDEX_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//' )"
            tmp_lob_storage_location_prop="$(prop_db_name_user_property_file AEOS_DB_LOB_STORAGE_LOCATION | sed -e 's/^"//' -e 's/"$//')"
            # Function to populate the os tablespace section for table index and lob storage
            # Function definition in common.sh
            # https://jsw.ibm.com/browse/DBACLD-175710
            populate_os_tablespaces "$tmp_os_db_name" "$DB_TYPE" "$OS_DATASOURCE_NUMBER" "$tmp_table_storage_location_prop" "$tmp_index_storage_location_prop" "$tmp_lob_storage_location_prop"
        fi


    fi

    # Apply value in FNCM OS required by Case history property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "workflow-runtime" || " ${pattern_cr_arr[@]}" =~ "workflow-authoring" || " ${pattern_cr_arr[@]}" =~ "workstreams" ]]; then
        OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'CHOS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            tmp_os_db_servername="$(prop_db_name_user_property_file_for_server_name CHOS_DB_USER_NAME)"
            tmp_os_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_servername")

            if [[ $DB_TYPE == "oracle" ]]; then
                tmp_os_db_name="$(prop_db_name_user_property_file CHOS_DB_USER_NAME)"
            else
                tmp_os_db_name="$(prop_db_name_user_property_file CHOS_DB_NAME)"
            fi
            # tmp_os_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_os_db_name")
            if [[ $DB_TYPE == "postgresql" ]]; then
                tmp_os_db_name=$(echo $tmp_os_db_name | tr '[:upper:]' '[:lower:]')
            fi
            for i in "${!OSDB_CR_MAPPING[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].${OSDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_os_db_servername.${OSDB_COMMON_PROPERTY[i]})\""
            done

            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_os_label "\"ch\""

            # remove database_name if oracle
            if [[ $DB_TYPE == "oracle" ]]; then
                get_oracle_service_name $(prop_db_server_property_file $tmp_os_db_servername.ORACLE_JDBC_URL)
                if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"${ORACLE_SERVICE_NAME}CHOS\""
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name "\"<Required>\""
                fi
                # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].database_name $tmp_os_db_name
                if [[ $DB_TYPE == "postgresql-edb" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_database_type "postgresql"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER].dc_use_postgres "true"
                fi
            fi
        fi
        # Apply customized schema
        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'CHOS' |cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            # if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
            tmp_schema_name=$(prop_db_name_user_property_file CHOS_DB_CURRENT_SCHEMA)
            tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
            if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
                fi
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER].oc_cpe_obj_store_schema_name  "\"${tmp_schema_name}\""
            fi
            # fi
        fi
    fi

    # Applying value in ICN property file into final CR
    if [[ " ${foundation_component_arr[@]}" =~ "BAN" ]]; then
        if [[ ! (" ${pattern_cr_arr[@]} " =~ "workstreams" && "${#pattern_cr_arr[@]}" -eq "1") ]]; then
            tmp_icn_db_servername="$(prop_db_name_user_property_file_for_server_name ICN_DB_USER_NAME)"
            tmp_icn_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_icn_db_servername")

            if [[ $DB_TYPE == "oracle" ]]; then
                tmp_icn_db_name="$(prop_db_name_user_property_file ICN_DB_USER_NAME)"
            else
                tmp_icn_db_name="$(prop_db_name_user_property_file ICN_DB_NAME)"
            fi
            # tmp_icn_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_icn_db_name")
            if [[ $DB_TYPE == "postgresql" ]]; then
                tmp_icn_db_name=$(echo $tmp_icn_db_name | tr '[:upper:]' '[:lower:]')
            fi
            
            #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
            # Initialize the isfalse variable to validate dc_ssl_enabled is true or false
            isfalse=false
            for i in "${!ICNDB_CR_MAPPING[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${ICNDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_icn_db_servername.${ICNDB_COMMON_PROPERTY[i]})\""
                # Check if we are updating spec.datasource_configuration.dc_ssl_enabled value is false    
                if [ "${ICNDB_CR_MAPPING[i]}" == "spec.datasource_configuration.dc_ssl_enabled" ] && [[ "$(prop_db_server_property_file $tmp_icn_db_servername.${ICNDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                    isfalse=true
                fi
                if [ "${ICNDB_CR_MAPPING[i]}" == "spec.datasource_configuration.dc_icn_datasource.database_ssl_secret_name" ] && [ "$isfalse" == "true" ]; then
                    # Set the value to "" if isfalse is true
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.datasource_configuration.dc_icn_datasource.database_ssl_secret_name" "\"\""
                fi
                # ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${ICNDB_CR_MAPPING[i]}" "\"$(sed -e 's/^"//' -e 's/"$//' <<<"$(prop_db_server_property_file $tmp_icn_db_servername.${ICNDB_COMMON_PROPERTY[i]})")\""
            done

            # remove database_name if oracle
            if [[ $DB_TYPE == "oracle" ]]; then
                get_oracle_service_name $(prop_db_server_property_file $tmp_icn_db_servername.ORACLE_JDBC_URL)
                if [[ ! -z "$ORACLE_SERVICE_NAME" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_icn_datasource.database_name "\"${ORACLE_SERVICE_NAME}ICN\""
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_icn_datasource.database_name "\"<Required>\""
                fi
                # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_icn_datasource.database_name $tmp_icn_db_name
                if [[ $DB_TYPE == "postgresql-edb" ]]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_icn_datasource.dc_database_type "postgresql"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_icn_datasource.dc_use_postgres "true"
                fi
            fi
            # Apply customized schema for GCDDB
            tmp_schema_name=$(prop_db_name_user_property_file ICN_DB_CURRENT_SCHEMA)
            tmp_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_schema_name")
            if [[ $tmp_schema_name != "<Optional>" && $tmp_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_schema_name=$(echo $tmp_schema_name | tr '[:upper:]' '[:lower:]')
                fi
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.navigator_configuration.icn_production_setting.icn_schema  "\"${tmp_schema_name}\""
            fi
        fi
    fi

    # Applying value in ODM property file into final CR
    containsElement "decisions" "${pattern_cr_arr[@]}"
    odm_Val=$?
    if [[ $odm_Val -eq 0 ]]; then
        tmp_odm_db_servername="$(prop_db_name_user_property_file_for_server_name ODM_DB_USER_NAME)"
        tmp_odm_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_odm_db_servername")

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_odm_db_name="$(prop_db_name_user_property_file ODM_DB_USER_NAME)"
        else
            tmp_odm_db_name="$(prop_db_name_user_property_file ODM_DB_NAME)"
        fi
        tmp_odm_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_odm_db_name")

        for i in "${!ODMDB_CR_MAPPING[@]}"; do
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${ODMDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_odm_db_servername.${ODMDB_COMMON_PROPERTY[i]})\""
        done
        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_odm_datasource.dc_database_type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_odm_datasource.dc_use_postgres "true"
            # set dc_common_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.datasource_configuration.dc_odm_datasource`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_odm_datasource.dc_common_ssl_enabled "true"
            fi
        fi
        # For ODM, set dc_ssl_secret_name only when db is db2/oracle/postgresql with clientAuth
        if [[ $DB_TYPE == "postgresql" ]]; then
            postgresql_ssl_enable=$(sed -e 's/^"//' -e 's/"$//' <<<"$(prop_db_server_property_file $tmp_odm_db_servername.DATABASE_SSL_ENABLE)")
            tmp_ssl_flag=$(echo $postgresql_ssl_enable | tr '[:upper:]' '[:lower:]')

            postgresql_client_auth_enable=$(sed -e 's/^"//' -e 's/"$//' <<<"$(prop_db_server_property_file $tmp_odm_db_servername.POSTGRESQL_SSL_CLIENT_SERVER)")
            tmp_client_auth_flag=$(echo $postgresql_client_auth_enable | tr '[:upper:]' '[:lower:]')
        fi
        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            tmp_ssl_flag="true"
            tmp_client_auth_flag="true"
        fi
        if [[ ($tmp_ssl_flag == "yes" || $tmp_ssl_flag == "true" || $tmp_ssl_flag == "y") && ($tmp_client_auth_flag == "yes" || $tmp_client_auth_flag == "true" || $tmp_client_auth_flag == "y") ]]; then
            create_odm_postgresql_secret="Yes"
        else
            create_odm_postgresql_secret="No"
        fi
        if [[ $DB_TYPE == "db2" || $DB_TYPE == "db2HADR" || $DB_TYPE == "oracle" || $create_odm_postgresql_secret == "Yes" ]]; then
            tmp_ssl_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$(prop_db_server_property_file $tmp_odm_db_servername.DATABASE_SSL_ENABLE)")
            tmp_ssl_flag=$(echo $tmp_ssl_flag | tr '[:upper:]' '[:lower:]')
            if [[ $tmp_ssl_flag == "yes" || $tmp_ssl_flag == "true" || $tmp_ssl_flag == "y" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_odm_datasource.dc_ssl_secret_name "$(prop_db_server_property_file $tmp_odm_db_servername.DATABASE_SSL_SECRET_NAME)"
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_odm_datasource.dc_ssl_secret_name "\"\""
            fi
        else
            ${SED_COMMAND} "s|dc_ssl_secret_name: |# dc_ssl_secret_name: |g" ${CP4A_PATTERN_FILE_TMP}
        fi

        # set dc_odm_datasource.dc_common_database_instance_secret
        tmp_secret_name=`kubectl get secret -l db-name=${tmp_odm_db_name} -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_odm_datasource.dc_common_database_instance_secret "\"$tmp_secret_name\""

        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_odm_db_name=$(echo $tmp_odm_db_name | tr '[:upper:]' '[:lower:]')
        fi
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_odm_datasource.dc_common_database_name "\"$tmp_odm_db_name\""
        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_odm_db_jdbc_url=$(sed -e 's/^"//' -e 's/"$//' <<<"$(prop_db_server_property_file $tmp_odm_db_servername.ORACLE_JDBC_URL)")
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_odm_datasource.dc_common_database_url "\"$tmp_odm_db_jdbc_url\""
        fi
    fi

    # Applying value in BAW Authoring property file into final CR (only 21.0.3.x/22.0.1.x)
    # if [[ " ${pattern_cr_arr[@]}" =~ "workflow-authoring" ]]; then

    #     tmp_baw_authoring_db_servername="$(prop_db_name_user_property_file_for_server_name AUTHORING_DB_USER_NAME)"
    #     tmp_baw_authoring_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_authoring_db_servername")

    #     if [[ $DB_TYPE == "oracle" ]]; then
    #         tmp_baw_authoring_db_name="$(prop_db_name_user_property_file AUTHORING_DB_USER_NAME)"
    #     else
    #         tmp_baw_authoring_db_name="$(prop_db_name_user_property_file AUTHORING_DB_NAME)"
    #     fi

    #     tmp_baw_authoring_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_authoring_db_name")

    #     for i in "${!BAW_AUTHORING_CR_MAPPING[@]}"; do
    #         if [[ ("${BAW_AUTHORING_COMMON_PROPERTY[i]}" == "DATABASE_SERVERNAME"  || "${BAW_AUTHORING_COMMON_PROPERTY[i]}" == "DATABASE_PORT") && $DB_TYPE == "oracle" ]]; then
    #             ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${BAW_AUTHORING_CR_MAPPING[i]}" "\"<Remove>\""
    #         else
    #             ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${BAW_AUTHORING_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_baw_authoring_db_servername.${BAW_AUTHORING_COMMON_PROPERTY[i]})\""
    #         fi
    #     done

    #     tmp_secret_name=`kubectl get secret -l db-name=${tmp_baw_authoring_db_name} -o yaml | ${YQ_CMD} r - items.[0].metadata.name`

    #     # set workflow_authoring_configuration
    #     ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.workflow_authoring_configuration.database.secret_name "\"$tmp_secret_name\""
    #     if [[ $DB_TYPE == "postgresql" ]]; then
    #         tmp_baw_authoring_db_name=$(echo $tmp_baw_authoring_db_name | tr '[:upper:]' '[:lower:]')
    #     fi

    #     # remove database_name if oracle
    #     if [[ $DB_TYPE == "oracle" ]]; then
    #         ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.workflow_authoring_configuration.database.database_name "\"<Remove>\""
    #         # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
    #     else
    #         ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.workflow_authoring_configuration.database.database_name "\"$tmp_baw_authoring_db_name\""
    #     fi
    #     # remove jdbc_url/custom_jdbc_pvc
    #     if [[ $DB_TYPE == "oracle" || $DB_TYPE == "postgresql" ]]; then
    #         tmp_baw_authoring_db_jdbc_url="$(prop_db_server_property_file $tmp_baw_runtime_db_servername.ORACLE_JDBC_URL)"
    #         tmp_baw_authoring_db_jdbc_url=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_authoring_db_jdbc_url")
    #         ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.workflow_authoring_configuration.database.jdbc_url "\"$tmp_baw_authoring_db_jdbc_url\""
    #     else
    #         ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.workflow_authoring_configuration.database.jdbc_url "\"<Remove>\""
    #         ${SED_COMMAND} "s|jdbc_url: '\"<Remove>\"'|# jdbc_url: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
    #     fi
    #     ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.workflow_authoring_configuration.database.custom_jdbc_pvc "\"<Remove>\""
    #     ${SED_COMMAND} "s|custom_jdbc_pvc: '\"<Remove>\"'|# custom_jdbc_pvc: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
    # fi

    # Applying value in BAW Runtime property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "workflow-runtime" && (! " ${pattern_cr_arr[@]}" =~ "workflow-workstreams") ]]; then

        tmp_baw_runtime_db_servername="$(prop_db_name_user_property_file_for_server_name BAW_RUNTIME_DB_USER_NAME)"
        tmp_baw_runtime_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_runtime_db_servername")

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_baw_runtime_db_name="$(prop_db_name_user_property_file BAW_RUNTIME_DB_USER_NAME)"
        else
            tmp_baw_runtime_db_name="$(prop_db_name_user_property_file BAW_RUNTIME_DB_NAME)"
        fi

        tmp_baw_runtime_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_runtime_db_name")
        
        #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
        # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for BAWDB
        isfalse=false
        for i in "${!BAW_RUNTIME_CR_MAPPING[@]}"; do
            if [[ ("${BAW_RUNTIME_COMMON_PROPERTY[i]}" == "DATABASE_SERVERNAME"  || "${BAW_RUNTIME_COMMON_PROPERTY[i]}" == "DATABASE_PORT") && $DB_TYPE == "oracle" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${BAW_RUNTIME_CR_MAPPING[i]}" "\"<Remove>\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${BAW_RUNTIME_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_baw_runtime_db_servername.${BAW_RUNTIME_COMMON_PROPERTY[i]})\""
                # Check if we are updating spec.datasource_configuration.dc_ssl_enabled value to false
                if [ "${BAW_RUNTIME_CR_MAPPING[i]}" == "spec.baw_configuration.[0].database.enable_ssl" ] && [[ "$(prop_db_server_property_file $tmp_baw_runtime_db_servername.${BAW_RUNTIME_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                    isfalse=true
                fi
                # If SSL is disabled, set the database_ssl_secret_name to ""
                if [ "${BAW_RUNTIME_CR_MAPPING[i]}" == "spec.baw_configuration.[0].database.db_cert_secret_name" ] && [ "$isfalse" == "true" ]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.baw_configuration.[0].database.db_cert_secret_name" "\"\""
                fi
            fi
        done
        
        # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
        tmp_secret_name=`kubectl get secret -l db-name=${tmp_baw_runtime_db_name} -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`

        # set baw_configuration
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.secret_name "\"$tmp_secret_name\""
        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_baw_runtime_db_name=$(echo $tmp_baw_runtime_db_name | tr '[:upper:]' '[:lower:]')
        fi

        # remove database_name if oracle
        if [[ $DB_TYPE == "oracle" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.database_name "\"<Remove>\""
            # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.database_name "\"$tmp_baw_runtime_db_name\""
        fi

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_baw_runtime_db_jdbc_url="$(prop_db_server_property_file $tmp_baw_runtime_db_servername.ORACLE_JDBC_URL)"
            tmp_baw_runtime_db_jdbc_url=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_runtime_db_jdbc_url")
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.jdbc_url "\"$tmp_baw_runtime_db_jdbc_url\""
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.jdbc_url "\"<Remove>\""
            ${SED_COMMAND} "s|jdbc_url: '\"<Remove>\"'|# jdbc_url: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        fi
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.custom_jdbc_pvc "\"<Remove>\""
        ${SED_COMMAND} "s|custom_jdbc_pvc: '\"<Remove>\"'|# custom_jdbc_pvc: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}

        # set current schema name for db2 and postgresql
        if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
            tmp_baw_runtime_db_current_schema_name="$(prop_db_name_user_property_file BAW_RUNTIME_DB_CURRENT_SCHEMA)"
            tmp_baw_runtime_db_current_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_runtime_db_current_schema_name")
            if [[ $tmp_baw_runtime_db_current_schema_name != "<Optional>" && $tmp_baw_runtime_db_current_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_baw_runtime_db_current_schema_name=$(echo $tmp_baw_runtime_db_current_schema_name | tr '[:upper:]' '[:lower:]')
                fi
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.current_schema "\"$tmp_baw_runtime_db_current_schema_name\""
            fi
        fi
        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.dc_use_postgres "true"
            # always use default schema for EDB Postgres
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.current_schema
            # set dc_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.baw_configuration.[0].database`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.enable_ssl "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.db_cert_secret_name "{{ meta.name }}-pg-client-cert-secret"
            fi
        fi
        # Applying user profile for BAW runtime
        tmp_baw_runtime_admin="$(prop_user_profile_property_file BAW_RUNTIME.ADMIN_USER)"
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].admin_user "\"$tmp_baw_runtime_admin\""
    fi

    # Applying value in BAW Runtime+Workstreams property file into final CR
    if [[ "${pattern_cr_arr[@]}" =~ "workflow-workstreams" ]]; then
        tmp_baw_runtime_db_servername="$(prop_db_name_user_property_file_for_server_name BAW_RUNTIME_DB_USER_NAME)"
        tmp_baw_runtime_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_runtime_db_servername")

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_baw_runtime_db_name="$(prop_db_name_user_property_file BAW_RUNTIME_DB_USER_NAME)"
        else
            tmp_baw_runtime_db_name="$(prop_db_name_user_property_file BAW_RUNTIME_DB_NAME)"
        fi

        tmp_baw_runtime_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_runtime_db_name")
        #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
        # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for BAWDB
        isfalse=false
        for i in "${!BAW_RUNTIME_CR_MAPPING[@]}"; do
            if [[ ("${BAW_RUNTIME_COMMON_PROPERTY[i]}" == "DATABASE_SERVERNAME"  || "${BAW_RUNTIME_COMMON_PROPERTY[i]}" == "DATABASE_PORT") && $DB_TYPE == "oracle" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${BAW_RUNTIME_CR_MAPPING[i]}" "\"<Remove>\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${BAW_RUNTIME_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_baw_runtime_db_servername.${BAW_RUNTIME_COMMON_PROPERTY[i]})\""
                # Check if we are updating spec.datasource_configuration.dc_ssl_enabled value to false
                if [ "${BAW_RUNTIME_CR_MAPPING[i]}" == "spec.baw_configuration.[0].database.enable_ssl" ] && [[ "$(prop_db_server_property_file $tmp_baw_runtime_db_servername.${BAW_RUNTIME_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                    isfalse=true
                fi
                # If SSL is disabled, set the database_ssl_secret_name to ""
                if [ "${BAW_RUNTIME_CR_MAPPING[i]}" == "spec.baw_configuration.[0].database.db_cert_secret_name" ] && [ "$isfalse" == "true" ]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.baw_configuration.[0].database.db_cert_secret_name" "\"\""
                fi
            fi
        done

        # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
        tmp_secret_name=`kubectl get secret -l db-name=${tmp_baw_runtime_db_name} -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`

        # set baw_configuration
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.secret_name "\"$tmp_secret_name\""
        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_baw_runtime_db_name=$(echo $tmp_baw_runtime_db_name | tr '[:upper:]' '[:lower:]')
        fi

        if [[ $DB_TYPE == "oracle" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.database_name "\"<Remove>\""
            # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.database_name "\"$tmp_baw_runtime_db_name\""
        fi

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_baw_runtime_db_jdbc_url="$(prop_db_server_property_file $tmp_baw_runtime_db_servername.ORACLE_JDBC_URL)"
            tmp_baw_runtime_db_jdbc_url=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_runtime_db_jdbc_url")
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.jdbc_url "\"$tmp_baw_runtime_db_jdbc_url\""
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.jdbc_url "\"<Remove>\""
            ${SED_COMMAND} "s|jdbc_url: '\"<Remove>\"'|# jdbc_url: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        fi
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.custom_jdbc_pvc "\"<Remove>\""
        ${SED_COMMAND} "s|custom_jdbc_pvc: '\"<Remove>\"'|# custom_jdbc_pvc: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}

        # set current schema name for db2 and postgresql
        if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
            tmp_baw_runtime_db_current_schema_name="$(prop_db_name_user_property_file BAW_RUNTIME_DB_CURRENT_SCHEMA)"
            tmp_baw_runtime_db_current_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_baw_runtime_db_current_schema_name")
            if [[ $tmp_baw_runtime_db_current_schema_name != "<Optional>" && $tmp_baw_runtime_db_current_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_baw_runtime_db_current_schema_name=$(echo $tmp_baw_runtime_db_current_schema_name | tr '[:upper:]' '[:lower:]')
                fi

                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.current_schema "\"$tmp_baw_runtime_db_current_schema_name\""
            fi
        fi
        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.dc_use_postgres "true"
            # always use default schema for EDB Postgres
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.current_schema
            # set dc_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.baw_configuration.[0].database`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.enable_ssl "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.db_cert_secret_name "{{ meta.name }}-pg-client-cert-secret"
            fi
        fi

        tmp_aws_db_servername="$(prop_db_name_user_property_file_for_server_name AWS_DB_USER_NAME)"
        tmp_aws_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_aws_db_servername")
        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_aws_db_name="$(prop_db_name_user_property_file AWS_DB_USER_NAME)"
        else
            tmp_aws_db_name="$(prop_db_name_user_property_file AWS_DB_NAME)"
        fi
        tmp_aws_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_aws_db_name")
        #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
        # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for AWSDB
        isfalse=false
        for i in "${!AWS_CR_MAPPING[@]}"; do
            if [[ ("${AWS_COMMON_PROPERTY[i]}" == "DATABASE_SERVERNAME"  || "${AWS_COMMON_PROPERTY[i]}" == "DATABASE_PORT") && $DB_TYPE == "oracle" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${AWS_CR_MAPPING[i]}" "\"<Remove>\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${AWS_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_aws_db_servername.${AWS_COMMON_PROPERTY[i]})\""
                # Check if we are updating spec.datasource_configuration.dc_ssl_enabled value to false
                if [ "${AWS_CR_MAPPING[i]}" == "spec.baw_configuration.[1].database.enable_ssl" ] && [[ "$(prop_db_server_property_file $tmp_aws_db_servername.${AWS_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                    isfalse=true
                fi
                # If SSL is disabled, set the database_ssl_secret_name to ""
                if [ "${AWS_CR_MAPPING[i]}" == "spec.baw_configuration.[1].database.db_cert_secret_name" ] && [ "$isfalse" == "true" ]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.baw_configuration.[1].database.db_cert_secret_name" "\"\""
                fi
            fi
        done
        # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
        tmp_secret_name=`kubectl get secret -l db-name=${tmp_aws_db_name} -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`

        # set baw_configuration
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.secret_name "\"$tmp_secret_name\""
        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_aws_db_name=$(echo $tmp_aws_db_name | tr '[:upper:]' '[:lower:]')
        fi

        if [[ $DB_TYPE == "oracle" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.database_name "\"<Remove>\""
            # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.database_name "\"$tmp_aws_db_name\""
        fi

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_aws_db_jdbc_url="$(prop_db_server_property_file $tmp_aws_db_servername.ORACLE_JDBC_URL)"
            tmp_aws_db_jdbc_url=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_aws_db_jdbc_url")
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.jdbc_url "\"$tmp_aws_db_jdbc_url\""
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.jdbc_url "\"<Remove>\""
            ${SED_COMMAND} "s|jdbc_url: '\"<Remove>\"'|# jdbc_url: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        fi
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.custom_jdbc_pvc "\"<Remove>\""
        ${SED_COMMAND} "s|custom_jdbc_pvc: '\"<Remove>\"'|# custom_jdbc_pvc: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}

        # set current schema name for db2 and postgresql
        if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
            tmp_aws_db_current_schema_name="$(prop_db_name_user_property_file AWS_DB_CURRENT_SCHEMA)"
            tmp_aws_db_current_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_aws_db_current_schema_name")
            if [[ $tmp_aws_db_current_schema_name != "<Optional>" && $tmp_aws_db_current_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_aws_db_current_schema_name=$(echo $tmp_aws_db_current_schema_name | tr '[:upper:]' '[:lower:]')
                fi

                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.current_schema "\"$tmp_aws_db_current_schema_name\""
            fi
        fi
        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.dc_use_postgres "true"
            # always use default schema for EDB Postgres
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.current_schema
            # set dc_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.baw_configuration.[1].database`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.enable_ssl "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].database.db_cert_secret_name "{{ meta.name }}-pg-client-cert-secret"
            fi
        fi

        # Applying user profile for BAW runtime
        tmp_baw_runtime_admin="$(prop_user_profile_property_file BAW_RUNTIME.ADMIN_USER)"
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].admin_user "\"$tmp_baw_runtime_admin\""

        # Applying user profile for AWS
        tmp_aws_admin="$(prop_user_profile_property_file AWS.ADMIN_USER)"
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[1].admin_user "\"$tmp_aws_admin\""
    fi

    # Applying value in Workstreams property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "workstreams" && (! " ${pattern_cr_arr[@]}" =~ "workflow-workstreams") ]]; then
        tmp_aws_db_servername="$(prop_db_name_user_property_file_for_server_name AWS_DB_USER_NAME)"
        tmp_aws_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_aws_db_servername")

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_aws_db_name="$(prop_db_name_user_property_file AWS_DB_USER_NAME)"
        else
            tmp_aws_db_name="$(prop_db_name_user_property_file AWS_DB_NAME)"
        fi
        tmp_aws_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_aws_db_name")
        for i in "${!AWS_ONLY_CR_MAPPING[@]}"; do
            if [[ ("${AWS_COMMON_PROPERTY[i]}" == "DATABASE_SERVERNAME"  || "${AWS_COMMON_PROPERTY[i]}" == "DATABASE_PORT") && $DB_TYPE == "oracle" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${AWS_ONLY_CR_MAPPING[i]}" "\"<Remove>\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${AWS_ONLY_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_aws_db_servername.${AWS_COMMON_PROPERTY[i]})\""
            fi
        done
        # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
        tmp_secret_name=`kubectl get secret -l db-name=${tmp_aws_db_name} -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`

        # set baw_configuration
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.secret_name "\"$tmp_secret_name\""

        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_aws_db_name=$(echo $tmp_aws_db_name | tr '[:upper:]' '[:lower:]')
        fi

        if [[ $DB_TYPE == "oracle" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.database_name "\"<Remove>\""
            # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.database_name "\"$tmp_aws_db_name\""
        fi

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_aws_db_jdbc_url="$(prop_db_server_property_file $tmp_aws_db_servername.ORACLE_JDBC_URL)"
            tmp_aws_db_jdbc_url=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_aws_db_jdbc_url")
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.jdbc_url "\"$tmp_aws_db_jdbc_url\""
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.jdbc_url "\"<Remove>\""
            ${SED_COMMAND} "s|jdbc_url: '\"<Remove>\"'|# jdbc_url: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        fi
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.custom_jdbc_pvc "\"<Remove>\""
        ${SED_COMMAND} "s|custom_jdbc_pvc: '\"<Remove>\"'|# custom_jdbc_pvc: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}

        if [[ $DB_TYPE == "postgresql" || $DB_TYPE == "db2" ]]; then
            tmp_aws_db_current_schema_name="$(prop_db_name_user_property_file AWS_DB_CURRENT_SCHEMA)"
            tmp_aws_db_current_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_aws_db_current_schema_name")
            if [[ $tmp_aws_db_current_schema_name != "<Optional>" && $tmp_aws_db_current_schema_name != ""  ]]; then
                if [[ $DB_TYPE == "postgresql" ]]; then
                    tmp_aws_db_current_schema_name=$(echo $tmp_aws_db_current_schema_name | tr '[:upper:]' '[:lower:]')
                fi

                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.current_schema "\"$tmp_aws_db_current_schema_name\""
            fi
        fi
        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.dc_use_postgres "true"
            # always use default schema for EDB Postgres
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.current_schema
            # set dc_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.baw_configuration.[0].database`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.enable_ssl "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].database.db_cert_secret_name "{{ meta.name }}-pg-client-cert-secret"
            fi
        fi

        # Applying user profile for AWS
        tmp_aws_admin="$(prop_user_profile_property_file AWS.ADMIN_USER)"
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[0].admin_user "\"$tmp_aws_admin\""
    fi

    # Applying values in ADS property file into final CR
    if [[ "${pattern_cr_arr[@]}" =~ "decisions_ads" ]]; then
        # Applying values for ADS Designer into final CR
        if [[ "${optional_component_arr[@]}" =~ "DecisionDesigner" ]]; then

            ### -- https://jsw.ibm.com/browse/DBACLD-151937 - <Migration from Mongo to external Postgresql for ADS>
            # Applying values for external postgresql into final CR
            if [[ $DB_TYPE = "postgresql" ]]; then
                # Getting values from propertyfiles for external postgresql
                tmp_ads_designer_db_servername_prefix="$(prop_db_name_user_property_file_for_server_name ADS_DESIGNER_DB_USER_NAME)"
                tmp_ads_designer_db_servername_prefix=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_designer_db_servername_prefix")

                tmp_ads_designer_db_port="$(prop_db_server_property_file $tmp_ads_designer_db_servername_prefix.DATABASE_PORT)"
                tmp_ads_designer_db_port=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_designer_db_port")

                tmp_ads_designer_db_servername="$(prop_db_server_property_file $tmp_ads_designer_db_servername_prefix.DATABASE_SERVERNAME)"
                tmp_ads_designer_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_designer_db_servername")

                tmp_ads_designer_db_name="$(prop_db_name_user_property_file ADS_DESIGNER_DB_NAME)"
                tmp_ads_designer_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_designer_db_name")
                tmp_ads_designer_db_name=$(echo $tmp_ads_designer_db_name | tr '[:upper:]' '[:lower:]')

                # Applying values into final CR
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.dc_use_postgres "false"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_servername "\"$tmp_ads_designer_db_servername\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_name "\"$tmp_ads_designer_db_name\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_port "\"$tmp_ads_designer_db_port\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_instance_secret "\"ibm-ads-designer-database\""
            
                # Checking if SSL is enable
                tmp_ads_designer_ssl_flag="$(prop_db_server_property_file $tmp_ads_designer_db_servername_prefix.DATABASE_SSL_ENABLE)"
                tmp_ads_designer_ssl_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_designer_ssl_flag")
                tmp_ads_designer_ssl_flag=$(echo $tmp_ads_designer_ssl_flag | tr '[:upper:]' '[:lower:]')

                if [[ $tmp_ads_designer_ssl_flag == "yes" || $tmp_ads_designer_ssl_flag == "true" || $tmp_ads_designer_ssl_flag == "y" ]]; then
                    # Getting SSL related values from propertyfile
                    tmp_ads_designer_db_ssl_secret="$(prop_db_server_property_file $tmp_ads_designer_db_servername_prefix.DATABASE_SSL_SECRET_NAME)"
                    tmp_ads_designer_db_ssl_secret=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_designer_db_ssl_secret")
                    tmp_ads_designer_db_ssl_secret=$(echo $tmp_ads_designer_db_ssl_secret | tr '[:upper:]' '[:lower:]')

                    tmp_ads_designer_db_ssl_mode="$(prop_db_server_property_file $tmp_ads_designer_db_servername_prefix.POSTGRESQL_SSL_MODE)"
                    tmp_ads_designer_db_ssl_mode=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_designer_db_ssl_mode")
                    tmp_ads_designer_db_ssl_mode=$(echo $tmp_ads_designer_db_ssl_mode | tr '[:upper:]' '[:lower:]')

                    # Applying SSL related values into final CR
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_secret_name "\"$tmp_ads_designer_db_ssl_secret\""
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_enabled "true"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_mode "\"$tmp_ads_designer_db_ssl_mode\""
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_secret_name
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_enabled "false"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_mode
                fi

                # Applying customized schema for ads designer into final CR
                tmp_ads_designer_schema_name=$(prop_db_name_user_property_file ADS_DESIGNER_DB_CURRENT_SCHEMA)
                tmp_ads_designer_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_designer_schema_name")
                # Check if schema name is not empty <https://jsw.ibm.com/browse/DBACLD-168541>
                if [[ -n $tmp_ads_designer_schema_name ]]; then
                    # Convert schema name to lowercase
                    tmp_ads_designer_schema_name=$(echo $tmp_ads_designer_schema_name | tr '[:upper:]' '[:lower:]')
                    # Update the schema in the configuration
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.current_schema "\"${tmp_ads_designer_schema_name}\""
                else
                    # Set default schema to "ads" if empty
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.current_schema "\"ads\""
                fi
            fi
        fi
        
        # Applying values for ADS Runtime into final CR
        if [[ "${optional_component_arr[@]}" =~ "DecisionRuntime" ]]; then

            ### -- https://jsw.ibm.com/browse/DBACLD-151937 - <Migration from Mongo to external Postgresql for ADS>
            # Applying values for external postgresql into final CR
            if [[ $DB_TYPE = "postgresql" ]]; then
                # Getting properties from propertyfiles for external postgresql
                tmp_ads_runtime_db_servername_prefix="$(prop_db_name_user_property_file_for_server_name ADS_RUNTIME_DB_USER_NAME)"
                tmp_ads_runtime_db_servername_prefix=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_runtime_db_servername_prefix")

                tmp_ads_runtime_db_port="$(prop_db_server_property_file $tmp_ads_runtime_db_servername_prefix.DATABASE_PORT)"
                tmp_ads_runtime_db_port=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_runtime_db_port")

                tmp_ads_runtime_db_servername="$(prop_db_server_property_file $tmp_ads_runtime_db_servername_prefix.DATABASE_SERVERNAME)"
                tmp_ads_runtime_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_runtime_db_servername")

                tmp_ads_runtime_db_name="$(prop_db_name_user_property_file ADS_RUNTIME_DB_NAME)"
                tmp_ads_runtime_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_runtime_db_name")
                tmp_ads_runtime_db_name=$(echo $tmp_ads_runtime_db_name | tr '[:upper:]' '[:lower:]')

                # Applying values into final CR
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.dc_use_postgres "false"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_servername "\"$tmp_ads_runtime_db_servername\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_name "\"$tmp_ads_runtime_db_name\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_port "\"$tmp_ads_runtime_db_port\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_instance_secret "\"ibm-ads-runtime-database\""

                # Checking if SSL is enable
                tmp_ads_runtime_ssl_flag="$(prop_db_server_property_file $tmp_ads_runtime_db_servername_prefix.DATABASE_SSL_ENABLE)"
                tmp_ads_runtime_ssl_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_runtime_ssl_flag")
                tmp_ads_runtime_ssl_flag=$(echo $tmp_ads_runtime_ssl_flag | tr '[:upper:]' '[:lower:]')

                if [[ $tmp_ads_runtime_ssl_flag == "yes" || $tmp_ads_runtime_ssl_flag == "true" || $tmp_ads_runtime_ssl_flag == "y" ]]; then
                    # Getting SSL related values from propertyfile
                    tmp_ads_runtime_db_ssl_secret="$(prop_db_server_property_file $tmp_ads_runtime_db_servername_prefix.DATABASE_SSL_SECRET_NAME)"
                    tmp_ads_runtime_db_ssl_secret=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_runtime_db_ssl_secret")
                    tmp_ads_runtime_db_ssl_secret=$(echo $tmp_ads_runtime_db_ssl_secret | tr '[:upper:]' '[:lower:]')

                    tmp_ads_runtime_db_ssl_mode="$(prop_db_server_property_file $tmp_ads_runtime_db_servername_prefix.POSTGRESQL_SSL_MODE)"
                    tmp_ads_runtime_db_ssl_mode=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_runtime_db_ssl_mode")
                    tmp_ads_runtime_db_ssl_mode=$(echo $tmp_ads_runtime_db_ssl_mode | tr '[:upper:]' '[:lower:]')

                    # Applying SSL related values into final CR
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_secret_name "\"$tmp_ads_runtime_db_ssl_secret\""
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_enabled "true"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_mode "\"$tmp_ads_runtime_db_ssl_mode\""
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_secret_name
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_enabled "false"
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_mode
                fi

                # Applying customized schema for ads runtime into final CR
                tmp_ads_runtime_schema_name=$(prop_db_name_user_property_file ADS_RUNTIME_DB_CURRENT_SCHEMA)
                tmp_ads_runtime_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ads_runtime_schema_name")
                # Check if runtime schema name is not empty <https://jsw.ibm.com/browse/DBACLD-168541>
                if [[ -n $tmp_ads_runtime_schema_name ]]; then
                    # Convert runtime schema name to lowercase
                    tmp_ads_runtime_schema_name=$(echo $tmp_ads_runtime_schema_name | tr '[:upper:]' '[:lower:]')
                    # Update the runtime schema in the configuration
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.current_schema "\"${tmp_ads_runtime_schema_name}\""
                else
                    # Set default runtime schema to "ads" if empty
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.current_schema "\"ads\""
                fi
            fi
        fi
    fi

    # Applying value from ADP Gitgateway property file into final CR
    if [[ "${pattern_cr_arr[@]}" =~ "document_processing" ]]; then
        # DBACLD-178324: Remove dc_adp_datasouce section when document_processing_designer is not selected
        if ! [[ "${pattern_cr_arr[@]}" =~ "document_processing_designer" ]]; then
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource
        else
            ### -- https://jsw.ibm.com/browse/DBACLD-151937 - <Migration from Mongo to external Postgresql for ADP>
            # Applying values for external postgresql into final CR
            if [[ $DB_TYPE = "postgresql" ]]; then
                # Getting properties from propertyfiles for external postgresql
                tmp_adp_gg_db_servername_prefix="$(prop_db_name_user_property_file_for_server_name ADP_GG_DB_USER_NAME)"
                tmp_adp_gg_db_servername_prefix=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_gg_db_servername_prefix")

                tmp_adp_gg_db_port="$(prop_db_server_property_file $tmp_adp_gg_db_servername_prefix.DATABASE_PORT)"
                tmp_adp_gg_db_port=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_gg_db_port")

                tmp_adp_gg_db_servername="$(prop_db_server_property_file $tmp_adp_gg_db_servername_prefix.DATABASE_SERVERNAME)"
                tmp_adp_gg_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_gg_db_servername")

                tmp_adp_gg_db_name="$(prop_db_name_user_property_file ADP_GG_DB_NAME)"
                tmp_adp_gg_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_gg_db_name")
                tmp_adp_gg_db_name=$(echo $tmp_adp_gg_db_name | tr '[:upper:]' '[:lower:]')

                # Applying values into final CR
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.dc_use_postgres "false"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_servername "\"$tmp_adp_gg_db_servername\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_name "\"$tmp_adp_gg_db_name\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_port "\"$tmp_adp_gg_db_port\""

                # https://jsw.ibm.com/browse/DBACLD-167259 ( Adding the SSL check )
                # Checking if SSL is enable
                tmp_adp_gg_ssl_flag="$(prop_db_server_property_file $tmp_adp_gg_db_servername_prefix.DATABASE_SSL_ENABLE)"
                tmp_adp_gg_ssl_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_gg_ssl_flag")
                tmp_adp_gg_ssl_flag=$(echo $tmp_adp_gg_ssl_flag | tr '[:upper:]' '[:lower:]')

                    if [[ $tmp_adp_gg_ssl_flag == "yes" || $tmp_adp_gg_ssl_flag == "true" || $tmp_adp_gg_ssl_flag == "y" ]]; then
                        # Getting SSL related values from propertyfile
                        tmp_adp_gg_db_ssl_secret="$(prop_db_server_property_file $tmp_adp_gg_db_servername_prefix.DATABASE_SSL_SECRET_NAME)"
                        tmp_adp_gg_db_ssl_secret=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_gg_db_ssl_secret")
                        tmp_adp_gg_db_ssl_secret=$(echo $tmp_adp_gg_db_ssl_secret | tr '[:upper:]' '[:lower:]')

                        # Applying SSL related values into final CR
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_ssl_secret_name "\"$tmp_adp_gg_db_ssl_secret\""
                    fi
                            
                # Applying customized schema for ADP into final CR
                tmp_adp_gg_schema_name=$(prop_db_name_user_property_file ADP_GG_DB_SCHEMA)
                tmp_adp_gg_schema_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_gg_schema_name")
                if [[ $tmp_adp_gg_schema_name != "<Optional>" && $tmp_adp_gg_schema_name != ""  ]]; then
                    tmp_adp_gg_schema_name=$(echo $tmp_adp_gg_schema_name | tr '[:upper:]' '[:lower:]')
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_schema "\"${tmp_adp_gg_schema_name}\""
                else
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_schema
                fi

                # Applying ADP database secret into final CR
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.document_processing.ibm_adp_secret "\"ibm-adp-secret\""

            fi
        fi
    fi

    ### -- https://jsw.ibm.com/browse/DBACLD-153348 - <Migration from Mongo to Postgres-edb for ADS>
    # Check if the required pattern is present in pattern_cr_arr for decisions_ads 
    if [[ "${pattern_cr_arr[@]}" =~ "decisions_ads" ]]; then

        # Set database names for ADS Designer and ADS Runtime
        tmp_dbname_designer="$(prop_db_name_user_property_file ADS_DESIGNER_DB_NAME)"
        tmp_dbname_runtime="$(prop_db_name_user_property_file ADS_RUNTIME_DB_NAME)"

        # Applying value in ADS property file into final CR if postgre-edb selected for all cp4ba for ads_designer
        if [[ "${optional_component_arr[@]}" =~ "DecisionDesigner" ]]; then
            if [[ $DB_TYPE == "postgresql-edb" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.dc_use_postgres "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_servername "\"postgres-cp4ba-rw.{{ meta.namespace }}.svc\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_name "\"$tmp_dbname_designer\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_port "\"5432\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.current_schema "\"adsdesigner\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_enabled "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_mode "\"verify-ca\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_secret_name "\"{{ meta.name }}-pg-client-cert-secret\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_instance_secret "\"ibm-ads-designer-database\""
            fi
        fi
        # Applying value in ADS property file into final CR if postgre-edb selected for all cp4ba for ads_runtime
        if [[ "${optional_component_arr[@]}" =~ "DecisionRuntime" ]]; then
            if [[ $DB_TYPE == "postgresql-edb" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.dc_use_postgres "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_servername "\"postgres-cp4ba-rw.{{ meta.namespace }}.svc\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_name "\"$tmp_dbname_runtime\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_port "\"5432\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.current_schema "\"adsruntime\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_enabled "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_mode "\"verify-ca\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_secret_name "\"{{ meta.name }}-pg-client-cert-secret\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_instance_secret "\"ibm-ads-runtime-database\""
            fi
        fi
    fi
    ### -- https://jsw.ibm.com/browse/DBACLD-153348 - <Migration from Mongo to Postgres-edb for ADP>
     # Applying value in ADP property file into final CR if postgre-edb selected for all cp4ba for ADP
     #document_processing
    if [[ "${pattern_cr_arr[@]}" =~ "document_processing" ]]; then
        # DBACLD-178324: Remove dc_adp_datasouce section when document_processing_designer is not selected
        if ! [[ "${pattern_cr_arr[@]}" =~ "document_processing_designer" ]]; then
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource
        else
            tmp_dbname="$(prop_db_name_user_property_file ADP_GG_DB_NAME)"

            if [[ $DB_TYPE = "postgresql-edb" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.dc_use_postgres "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_servername "\"postgres-cp4ba-rw.{{ meta.namespace }}.svc\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_name "\"$tmp_dbname\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_port "\"5432\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_ssl_secret_name "\"{{ meta.name }}-pg-client-cert-secret\"" 
            fi
        fi
    fi

    ### -- https://jsw.ibm.com/browse/DBACLD-154816 - <Migration from Mongo to Postgres-edb for ADS>
    # Check if the required pattern is present in pattern_cr_arr for decisions_ads if select db2 
    if [[ "${pattern_cr_arr[@]}" =~ "decisions_ads" ]]; then

        # Set database names for ADS Designer and ADS Runtime 
        tmp_dbname_designer="$(prop_db_name_user_property_file ADS_DESIGNER_DB_NAME)"
        tmp_dbname_runtime="$(prop_db_name_user_property_file ADS_RUNTIME_DB_NAME)"

        # Applying value in ADS property file into final CR if postgre-edb selected for all cp4ba for ads_designer
        if [[ "${optional_component_arr[@]}" =~ "DecisionDesigner" ]]; then
            if [[ $DB_TYPE == "db2" || $DB_TYPE == "oracle" || $DB_TYPE == "sqlserver" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.dc_use_postgres "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_servername "\"postgres-cp4ba-rw.{{ meta.namespace }}.svc\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_name "\"$tmp_dbname_designer\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_port "\"5432\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.current_schema "\"adsdesigner\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_enabled "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_mode "\"verify-ca\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.ssl_secret_name "\"{{ meta.name }}-pg-client-cert-secret\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_designer_datasource.database_instance_secret "\"ibm-ads-designer-database\""
            fi
        fi
        # Applying value in ADS property file into final CR if postgre-edb selected for all cp4ba for ads_runtime
        if [[ "${optional_component_arr[@]}" =~ "DecisionRuntime" ]]; then
            if [[ $DB_TYPE == "db2" || $DB_TYPE == "oracle" || $DB_TYPE == "sqlserver" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.dc_use_postgres "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_servername "\"postgres-cp4ba-rw.{{ meta.namespace }}.svc\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_name "\"$tmp_dbname_runtime\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_port "\"5432\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.current_schema "\"adsruntime\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_enabled "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_mode "\"verify-ca\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.ssl_secret_name "\"{{ meta.name }}-pg-client-cert-secret\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ads_runtime_datasource.database_instance_secret "\"ibm-ads-runtime-database\""
            fi
        fi  
    fi

    ### -- https://jsw.ibm.com/browse/DBACLD-154816 - <Migration from Mongo to Postgres-edb for ADP>
     # Applying value in ADP property file into final CR if select the db2
     #document_processing
    if [[ "${pattern_cr_arr[@]}" =~ "document_processing" ]]; then
        # DBACLD-178324: Remove dc_adp_datasouce section when document_processing_designer is not selected
        if ! [[ "${pattern_cr_arr[@]}" =~ "document_processing_designer" ]]; then
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource
        else
            tmp_dbname="$(prop_db_name_user_property_file ADP_GG_DB_NAME)"

            if [[ $DB_TYPE == "db2" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.dc_use_postgres "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.dc_database_type "postgresql"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_servername "\"postgres-cp4ba-rw.{{ meta.namespace }}.svc\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_name "\"$tmp_dbname\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_port "\"5432\""
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_adp_datasource.database_ssl_secret_name "\"{{ meta.name }}-pg-client-cert-secret\"" 
            fi
        fi
    fi

    # Applying value in ACA property file into final CR
    if [[ "${pattern_cr_arr[@]}" =~ "document_processing" ]]; then
        tmp_adp_db_servername="$(prop_db_name_user_property_file_for_server_name ADP_BASE_DB_USER_NAME)"
        tmp_adp_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_db_servername")

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_adp_db_name="$(prop_db_name_user_property_file ADP_BASE_DB_USER_NAME)"
        else
            tmp_adp_db_name="$(prop_db_name_user_property_file ADP_BASE_DB_NAME)"
        fi
        tmp_adp_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_adp_db_name")

        for i in "${!ADPDB_CR_MAPPING[@]}"; do
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${ADPDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_adp_db_servername.${ADPDB_COMMON_PROPERTY[i]})\""
        done
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.database_name "\"$tmp_adp_db_name\""

        # set dc_ca_datasource.tenant_databases
        local db_name_array=()
        tmp_dbname=$(prop_db_name_user_property_file ADP_PROJECT_DB_NAME)

        tmp_dbname=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_dbname")

        OIFS=$IFS
        IFS=',' read -ra db_name_array <<< "$tmp_dbname"
        IFS=$OIFS
        # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.tenant_databases
        for i in ${!db_name_array[@]}; do
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.tenant_databases.[${i}] "\"${db_name_array[i]}\""
        done
        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.dc_database_type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.dc_use_postgres "true"
            # set dc_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.datasource_configuration.dc_ca_datasource`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.dc_database_ssl_enabled "true"
                # ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.dc_ssl_secret_name "{{ meta.name }}-pg-client-cert-secret"
            fi
        fi

        # Recommend to provide an external MongoDB for production deployments.
        tmp_mongo_flag="$(prop_user_profile_property_file ADP.USE_EXTERNAL_MONGODB)"
        tmp_mongo_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_mongo_flag")

        if [[ $tmp_mongo_flag == "Yes" || $tmp_mongo_flag == "YES" || $tmp_mongo_flag == "Y" || $tmp_mongo_flag == "True" || $tmp_mongo_flag == "true" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.document_processing.deploy_mongo "false"
        elif [[ $tmp_mongo_flag == "No" || $tmp_mongo_flag == "NO" || $tmp_mongo_flag == "N" || $tmp_mongo_flag == "False" || $tmp_mongo_flag == "false" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.document_processing.deploy_mongo "true"
        fi

        # Apply git connection secret if true when ADP designer
        if [[ " ${pattern_cr_arr[@]}" =~ "document_processing_designer" ]]; then
            tmp_git_flag="$(prop_user_profile_property_file ADP.ENABLE_GIT_SSL_CONNECTION)"
            tmp_git_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_git_flag")
            if [[ $tmp_git_flag == "Yes" || $tmp_git_flag == "YES" || $tmp_git_flag == "Y" || $tmp_git_flag == "True" || $tmp_git_flag == "true" ]]; then
                tmp_git_secret_name="$(prop_user_profile_property_file ADP.GIT_SSL_SECRET_NAME)"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.trusted_certificate_list.[0] "\"$tmp_git_secret_name\""
            fi
        fi

        # Apply REPO_SERVICE_URL and CDRA route certificate and runtime_feedback if ADP runtime
        if [[ " ${pattern_cr_arr[@]}" =~ "document_processing_runtime" ]]; then
            tmp_REPO_SERVICE_URL="$(prop_user_profile_property_file ADP.REPO_SERVICE_URL)"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.document_processing.cpds.production_setting.repo_service_url "\"$tmp_REPO_SERVICE_URL\""

            tmp_cdra_secret_name="$(prop_user_profile_property_file ADP.CDRA_SSL_SECRET_NAME)"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.trusted_certificate_list.[0] "\"$tmp_cdra_secret_name\""

            tmp_runtime_feedback_enabled="$(prop_user_profile_property_file ADP.RUNTIME_FEEDBACK_ENABLED)"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ca_configuration.global.runtime_feedback.enabled "\"$tmp_runtime_feedback_enabled\""

            tmp_runtime_feedback_runtime_type="$(prop_user_profile_property_file ADP.RUNTIME_FEEDBACK_RUNTIME_TYPE)"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ca_configuration.global.runtime_feedback.runtime_type "\"$tmp_runtime_feedback_runtime_type\""

            tmp_runtime_feedback_design_api_secret="$(prop_user_profile_property_file ADP.RUNTIME_FEEDBACK_DESIGN_API_SECRET)"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ca_configuration.global.runtime_feedback.design_api_secret "\"$tmp_runtime_feedback_design_api_secret\""

            tmp_runtime_feedback_design_tls_secret="$(prop_user_profile_property_file ADP.RUNTIME_FEEDBACK_DESIGN_TLS_SECRET)"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ca_configuration.global.runtime_feedback.design_tls_secret "\"$tmp_runtime_feedback_design_tls_secret\""
        fi
    fi

    # Applying value in BAStudio property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "workflow-process-service" || " ${pattern_cr_arr[@]}" =~ "document_processing_designer" || "${pattern_cr_arr[@]}" =~ "workflow-authoring" || " ${optional_component_cr_arr[@]}" =~ "app_designer" || " ${optional_component_cr_arr[@]}" =~ "ads_designer" ]]; then
        tmp_bas_db_servername="$(prop_db_name_user_property_file_for_server_name STUDIO_DB_USER_NAME)"
        tmp_bas_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_bas_db_servername")

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_bas_db_name="$(prop_db_name_user_property_file STUDIO_DB_USER_NAME)"
        else
            tmp_bas_db_name="$(prop_db_name_user_property_file STUDIO_DB_NAME)"
        fi
        tmp_bas_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_bas_db_name")
        #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
        # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for BASDB
        isfalse=false
        for i in "${!BASDB_CR_MAPPING[@]}"; do
            if [[ ("${BASDB_COMMON_PROPERTY[i]}" == "DATABASE_SERVERNAME"  || "${BASDB_COMMON_PROPERTY[i]}" == "DATABASE_PORT") && $DB_TYPE == "oracle" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${BASDB_CR_MAPPING[i]}" "\"<Remove>\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${BASDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_bas_db_servername.${BASDB_COMMON_PROPERTY[i]})\""
                # Check if we are updating spec.datasource_configuration.dc_ssl_enabled value to false
                if [ "${BASDB_CR_MAPPING[i]}" == "spec.bastudio_configuration.database.ssl_enabled" ] && [[ "$(prop_db_server_property_file $tmp_bas_db_servername.${BASDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                    isfalse=true
                fi
                # If SSL is disabled, set the database_ssl_secret_name to ""
                if [ "${BASDB_CR_MAPPING[i]}" == "spec.bastudio_configuration.database.certificate_secret_name" ] && [ "$isfalse" == "true" ]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.bastudio_configuration.database.certificate_secret_name" "\"\""
                fi
            fi
        done
        # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
        tmp_secret_name=`kubectl get secret -l db-name=${tmp_bas_db_name} -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`

        # set bastudio_configuration
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.admin_secret_name "\"$tmp_secret_name\""
        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_bas_db_name=$(echo $tmp_bas_db_name | tr '[:upper:]' '[:lower:]')
        fi

        # remove database_name if oracle
        if [[ $DB_TYPE == "oracle" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.database.name "\"<Remove>\""
            # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.database.name "\"$tmp_bas_db_name\""
        fi


        if [[ $DB_TYPE != "oracle" ]]; then
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.database.oracle_url
        fi

        # Applying user profile for BAS
        tmp_bas_admin="$(prop_user_profile_property_file BASTUDIO.ADMIN_USER)"
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.admin_user "\"$tmp_bas_admin\""

        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.database.type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.database.dc_use_postgres "true"
            # set dc_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.bastudio_configuration.database`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.database.ssl_enabled "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.database.certificate_secret_name "{{ meta.name }}-pg-client-cert-secret"
            fi
        fi
    fi

    # Applying value in playback server property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "document_processing_designer" || " ${optional_component_cr_arr[@]}" =~ "app_designer" || " ${optional_component_cr_arr[@]}" =~ "ads_designer" ]]; then

        tmp_app_db_servername="$(prop_db_name_user_property_file_for_server_name APP_PLAYBACK_DB_USER_NAME)"
        tmp_app_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_app_db_servername")

        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_app_db_name="$(prop_db_name_user_property_file APP_PLAYBACK_DB_USER_NAME)"
        else
            tmp_app_db_name="$(prop_db_name_user_property_file APP_PLAYBACK_DB_NAME)"
        fi
        tmp_app_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_app_db_name")
        
        #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
        # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for Playback
        isfalse=false
        for i in "${!PLAYBACKDB_CR_MAPPING[@]}"; do
            if [[ ("${PLAYBACKDB_COMMON_PROPERTY[i]}" == "DATABASE_SERVERNAME"  || "${PLAYBACKDB_COMMON_PROPERTY[i]}" == "DATABASE_PORT") && $DB_TYPE == "oracle" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${PLAYBACKDB_CR_MAPPING[i]}" "\"<Remove>\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${PLAYBACKDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_app_db_servername.${PLAYBACKDB_COMMON_PROPERTY[i]})\""
                # Check if we are updating spec.bastudio_configuration.playback_server.database.enable_ssl to false
                if [ "${PLAYBACKDB_CR_MAPPING[i]}" == "spec.bastudio_configuration.playback_server.database.enable_ssl" ] && [[ "$(prop_db_server_property_file $tmp_app_db_servername.${PLAYBACKDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                    isfalse=true
                fi
                # If SSL is disabled, set the db_cert_secret_name to ""
                if [ "${PLAYBACKDB_CR_MAPPING[i]}" == "spec.bastudio_configuration.playback_server.database.db_cert_secret_name" ] && [ "$isfalse" == "true" ]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.bastudio_configuration.playback_server.database.db_cert_secret_name" "\"\""
                fi
            fi
        done
        # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
        tmp_secret_name=`kubectl get secret -l db-name=${tmp_app_db_name} -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`

        # set bastudio_configuration.playback_server
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.admin_secret_name "\"$tmp_secret_name\""
        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_app_db_name=$(echo $tmp_app_db_name | tr '[:upper:]' '[:lower:]')
        fi

        # remove database_name if oracle
        if [[ $DB_TYPE == "oracle" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.name "\"<Remove>\""
            # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.name "\"$tmp_app_db_name\""
        fi

        if [[ $DB_TYPE != "oracle" ]]; then
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.oracle_url_without_wallet_directory
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.oracle_url_with_wallet_directory
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.oracle_sso_wallet_secret_name
        fi
        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.dc_use_postgres "true"
            # set dc_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.bastudio_configuration.playback_server.database`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.enable_ssl "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.database.db_cert_secret_name "{{ meta.name }}-pg-client-cert-secret"
            fi
        fi
        # Applying user profile for Playback
        tmp_playback_admin="$(prop_user_profile_property_file APP_PLAYBACK.ADMIN_USER)"
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.admin_user "\"$tmp_playback_admin\""

        # Applying user profile for AE HA Redis session
        tmp_session_flag="$(prop_user_profile_property_file APP_PLAYBACK.SESSION_REDIS_USE_EXTERNAL_STORE)"
        tmp_session_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_session_flag")
        if [[ $tmp_session_flag == "Yes" || $tmp_session_flag == "YES" || $tmp_session_flag == "Y" || $tmp_session_flag == "True" || $tmp_session_flag == "true" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.session.use_external_store "true"

            tmp_redis_host="$(prop_user_profile_property_file APP_PLAYBACK.SESSION_REDIS_HOST)"
            tmp_redis_port="$(prop_user_profile_property_file APP_PLAYBACK.SESSION_REDIS_PORT)"
            tmp_redis_ssl_secret_name="$(prop_user_profile_property_file APP_PLAYBACK.SESSION_REDIS_SSL_SECRET_NAME)"
            tmp_redis_tls_enabled="$(prop_user_profile_property_file APP_PLAYBACK.SESSION_REDIS_TLS_ENABLED)"
            tmp_redis_tls_enabled=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_redis_tls_enabled")
            tmp_redis_username="$(prop_user_profile_property_file APP_PLAYBACK.SESSION_REDIS_USERNAME)"

            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.redis.host "\"$tmp_redis_host\""
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.redis.port "\"$tmp_redis_port\""
            if [[ $tmp_redis_tls_enabled == "Yes" || $tmp_redis_tls_enabled == "YES" || $tmp_redis_tls_enabled == "Y" || $tmp_redis_tls_enabled == "True" || $tmp_redis_tls_enabled == "true" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.redis.tls_enabled "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.tls.tls_trust_list.[0] "\"$tmp_redis_ssl_secret_name\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.redis.tls_enabled "false"
            fi
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.redis.username "\"$tmp_redis_username\""
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.session.use_external_store "false"
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server.redis
        fi
    fi

    # Applying value in Automation Application server property file into final CR
    if [[ " ${pattern_cr_arr[@]}" =~ "document_processing" || " ${pattern_cr_arr[@]}" =~ "application" || " ${pattern_cr_arr[@]}" =~ "workflow-runtime" || " ${pattern_cr_arr[@]}" =~ "workstreams" || " ${pattern_cr_arr[@]}" =~ "workflow-authoring" ]]; then

        tmp_ae_db_servername="$(prop_db_name_user_property_file_for_server_name APP_ENGINE_DB_USER_NAME)"
        tmp_ae_db_servername=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ae_db_servername")
        if [[ $DB_TYPE == "oracle" ]]; then
            tmp_ae_db_name="$(prop_db_name_user_property_file APP_ENGINE_DB_USER_NAME)"
        else
            tmp_ae_db_name="$(prop_db_name_user_property_file APP_ENGINE_DB_NAME)"
        fi
        tmp_ae_db_name=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ae_db_name")
        
        #https://jsw.ibm.com/browse/DBACLD-158651 << Make ssl_secret_name empty when dc_ssl_enabled value is False>>
        # Initialize the isfalse variable to validate dc_ssl_enabled is true or false for AAE
        isfalse=false
        for i in "${!AEDB_CR_MAPPING[@]}"; do
            if [[ ("${AEDB_COMMON_PROPERTY[i]}" == "DATABASE_SERVERNAME"  || "${AEDB_COMMON_PROPERTY[i]}" == "DATABASE_PORT") && $DB_TYPE == "oracle" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${AEDB_CR_MAPPING[i]}" "\"<Remove>\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${AEDB_CR_MAPPING[i]}" "\"$(prop_db_server_property_file $tmp_ae_db_servername.${AEDB_COMMON_PROPERTY[i]})\""
                # Check if we are updating spec.datasource_configuration.dc_ssl_enabled value to false
                if [ "${AEDB_CR_MAPPING[i]}" == "spec.application_engine_configuration.[0].database.enable_ssl" ] && [[ "$(prop_db_server_property_file $tmp_ae_db_servername.${AEDB_COMMON_PROPERTY[i]})" =~ ^[fF]alse$ ]]; then
                    isfalse=true
                fi
                # If SSL is disabled, set the database_ssl_secret_name to ""
                if [ "${AEDB_CR_MAPPING[i]}" == "spec.application_engine_configuration.[0].database.db_cert_secret_name" ] && [ "$isfalse" == "true" ]; then
                    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "spec.application_engine_configuration.[0].database.db_cert_secret_name" "\"\""
                fi
            fi
        done
        # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
        tmp_secret_name=`kubectl get secret -l db-name=${tmp_ae_db_name} -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`

        # set application_engine_configuration
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].admin_secret_name "\"$tmp_secret_name\""
        if [[ $DB_TYPE == "postgresql" ]]; then
            tmp_ae_db_name=$(echo $tmp_ae_db_name | tr '[:upper:]' '[:lower:]')
        fi
        # remove database_name if oracle
        if [[ $DB_TYPE == "oracle" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.name "\"<Remove>\""
            # ${SED_COMMAND} "s|database_name: '\"<Remove>\"'|# database_name: '\"\"'|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.name "\"$tmp_ae_db_name\""
        fi

        if [[ $DB_TYPE != "oracle" ]]; then
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.oracle_url_without_wallet_directory
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.oracle_url_with_wallet_directory
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.oracle_sso_wallet_secret_name
        fi

        if [[ $DB_TYPE == "postgresql-edb" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.type "postgresql"
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.dc_use_postgres "true"
            # set dc_ssl_enabled always true for postgresql-edb
            ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.application_engine_configuration.[0].database`
            if [[ ! -z "$ds_cfg_val" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.enable_ssl "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].database.db_cert_secret_name "{{ meta.name }}-pg-client-cert-secret"
            fi
        fi
        # Applying user profile for AE
        tmp_ae_admin="$(prop_user_profile_property_file APP_ENGINE.ADMIN_USER)"
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].admin_user "\"$tmp_ae_admin\""

        # Applying user profile for AE HA Redis session
        tmp_session_flag="$(prop_user_profile_property_file APP_ENGINE.SESSION_REDIS_USE_EXTERNAL_STORE)"
        tmp_session_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_session_flag")
        if [[ $tmp_session_flag == "Yes" || $tmp_session_flag == "YES" || $tmp_session_flag == "Y" || $tmp_session_flag == "True" || $tmp_session_flag == "true" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].session.use_external_store "true"

            tmp_redis_host="$(prop_user_profile_property_file APP_ENGINE.SESSION_REDIS_HOST)"
            tmp_redis_port="$(prop_user_profile_property_file APP_ENGINE.SESSION_REDIS_PORT)"
            tmp_redis_ssl_secret_name="$(prop_user_profile_property_file APP_ENGINE.SESSION_REDIS_SSL_SECRET_NAME)"
            tmp_redis_tls_enabled="$(prop_user_profile_property_file APP_ENGINE.SESSION_REDIS_TLS_ENABLED)"
            tmp_redis_tls_enabled=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_redis_tls_enabled")
            tmp_redis_username="$(prop_user_profile_property_file APP_ENGINE.SESSION_REDIS_USERNAME)"

            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].redis.host "\"$tmp_redis_host\""
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].redis.port "\"$tmp_redis_port\""
            if [[ $tmp_redis_tls_enabled == "Yes" || $tmp_redis_tls_enabled == "YES" || $tmp_redis_tls_enabled == "Y" || $tmp_redis_tls_enabled == "True" || $tmp_redis_tls_enabled == "true" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].redis.tls_enabled "true"
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].tls.tls_trust_list.[0] "\"$tmp_redis_ssl_secret_name\""
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].redis.tls_enabled "false"
            fi
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].redis.username "\"$tmp_redis_username\""
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].session.use_external_store "false"
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[0].redis
        fi
    fi

    # set dc_ssl_enabled always true for postgresql-edb
    if [[ $DB_TYPE == "postgresql-edb" ]]; then

        ds_cfg_val=`cat $CP4A_PATTERN_FILE_TMP | ${YQ_CMD} r - spec.datasource_configuration`
        if [[ ! -z "$ds_cfg_val" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ssl_enabled "true"
        fi
    fi

    # Applying value in LDAP property file into final CR
    for i in "${!LDAP_COMMON_CR_MAPPING[@]}"; do
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${LDAP_COMMON_CR_MAPPING[i]}" "\"$(prop_ldap_property_file ${LDAP_COMMON_PROPERTY[i]})\""
    done

    if [[ $LDAP_TYPE == "AD" ]]; then
        for i in "${!AD_LDAP_CR_MAPPING[@]}"; do
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${AD_LDAP_CR_MAPPING[i]}" "\"$(prop_ldap_property_file ${AD_LDAP_PROPERTY[i]})\""
        done
    else
        for i in "${!TDS_LDAP_CR_MAPPING[@]}"; do
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${TDS_LDAP_CR_MAPPING[i]}" "\"$(prop_ldap_property_file ${TDS_LDAP_PROPERTY[i]})\""
        done
    fi
    # set lc_bind_secret
    # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
    tmp_secret_name=`kubectl get secret -l name=ldap-bind-secret -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`
    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ldap_configuration.lc_bind_secret "\"$tmp_secret_name\""
    # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ldap_configuration.lc_ldap_bind_dn
    # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ldap_configuration.lc_ldap_bind_dn_pwd
    # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ldap_configuration.lc_ldap_ssl_secret_folder


    # Applying value in External LDAP property file into final CR
    if [[ $SET_EXT_LDAP == "Yes" ]]; then
        for i in "${!LDAP_COMMON_CR_MAPPING[@]}"; do
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${EXT_LDAP_COMMON_CR_MAPPING[i]}" "\"$(prop_ext_ldap_property_file ${LDAP_COMMON_PROPERTY[i]})\""
        done

        tmp_ldap_type="$(prop_ext_ldap_property_file LDAP_TYPE)"
        tmp_ldap_type=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_ldap_type")
        if [[ $tmp_ldap_type == "Microsoft Active Directory" ]]; then
            for i in "${!AD_LDAP_CR_MAPPING[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${EXT_AD_LDAP_CR_MAPPING[i]}" "\"$(prop_ext_ldap_property_file ${AD_LDAP_PROPERTY[i]})\""
            done
        elif [[ $tmp_ldap_type == "IBM Security Directory Server" ]]; then
            for i in "${!TDS_LDAP_CR_MAPPING[@]}"; do
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${EXT_TDS_LDAP_CR_MAPPING[i]}" "\"$(prop_ext_ldap_property_file ${TDS_LDAP_PROPERTY[i]})\""
            done
        else
            fail "The value for \"LDAP_TYPE\" in the property file \"${EXTERNAL_LDAP_PROPERTY_FILE}\" is not valid. The possible values are: \"IBM Security Directory Server\" or \"Microsoft Active Directory\""
            exit 1
        fi

        # set lc_bind_secret
        # For DBACLD-155445 where we need to use the namespace value passed to find the secret name and populate the CR accordingly
        tmp_secret_name=`kubectl get secret -l name=ext-ldap-bind-secret -o yaml -n $CP4BA_SERVICES_NS | ${YQ_CMD} r - items.[0].metadata.name`
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ext_ldap_configuration.lc_bind_secret "\"$tmp_secret_name\""
        # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ext_ldap_configuration.lc_ldap_bind_dn
        # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ext_ldap_configuration.lc_ldap_bind_dn_pwd
        # ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ext_ldap_configuration.lc_ldap_ssl_secret_folder
    fi

    # Applying value in scim property file into final CR
    set_scim_attr="true"
    if [[ "${set_scim_attr}" == "true" ]]; then
      if [[ " ${pattern_cr_arr[@]}" =~ "workflow-runtime" || " ${pattern_cr_arr[@]}" =~ "workflow-authoring" || " ${pattern_cr_arr[@]}" =~ "content" || " ${pattern_cr_arr[@]}" =~ "document_processing" || "${optional_component_cr_arr[@]}" =~ "ae_data_persistence" || (" ${pattern_cr_arr[@]}" =~ "workflow-process-service" && "${optional_component_cr_arr[@]}" =~ "wfps_authoring") ]]; then
          for i in "${!SCIM_PROPERTY[@]}"; do
              ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} "${SCIM_CR_MAPPING[i]}" "\"$(prop_user_profile_property_file ${SCIM_PROPERTY[i]})\""
          done
      fi
    fi
    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} null
    ${SED_COMMAND} "s|'\"|\"|g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s|\"'|\"|g" ${CP4A_PATTERN_FILE_TMP}
    # ${SED_COMMAND} "s|\"\"|\"|g" ${CP4A_PATTERN_FILE_TMP}
    # Remove HADR if dose not input value
    ${SED_COMMAND} "s/: \"<Optional>\"/: \"\"/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: \"\"<Optional>\"\"/: \"\"/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: <Optional>/: \"\"/g" ${CP4A_PATTERN_FILE_TMP}

    ${SED_COMMAND} "s/database_ip: \"<Required>\"/database_ip: \"\"/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/dc_hadr_standby_ip: \"<Required>\"/dc_hadr_standby_ip: \"\"/g" ${CP4A_PATTERN_FILE_TMP}

    # convert ssl enable true or false to meet CSV
    ${SED_COMMAND} "s/: \"True\"/: true/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: \"False\"/: false/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: \"true\"/: true/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: \"false\"/: false/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: \"Yes\"/: true/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: \"yes\"/: true/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: \"No\"/: false/g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s/: \"no\"/: false/g" ${CP4A_PATTERN_FILE_TMP}

    # Remove all null string
    ${SED_COMMAND} "s/: null/: /g" ${CP4A_PATTERN_FILE_TMP}


    # comment out sc_ingress_tls_secret_name if OCP platform
    if [[ $PLATFORM_SELECTED == "OCP" ]]; then
        ${SED_COMMAND} "s/sc_ingress_tls_secret_name: /# sc_ingress_tls_secret_name: /g" ${CP4A_PATTERN_FILE_TMP}
    fi

    # comment out the database_servername/database_port/database_name/HADR if the db is oracle
    if [[ $DB_TYPE == "oracle" ]]; then
        ${SED_COMMAND} "s/database_servername:/# database_servername:/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/database_port:/# database_port:/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/database_name: \"<Remove>\"/# database_name: \"\"/g" ${CP4A_PATTERN_FILE_TMP}

        ${SED_COMMAND} "s/alternative_host:/# alternative_host:/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/alternative_port:/# alternative_port:/g" ${CP4A_PATTERN_FILE_TMP}

        ${SED_COMMAND} "s/server_name: \"<Remove>\"/# server_name: \"\"/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/port: \"<Remove>\"/# port: \"\"/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/host: \"<Remove>\"/# host: \"\"/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/name: \"<Remove>\"/# name: \"\"/g" ${CP4A_PATTERN_FILE_TMP}

        ${SED_COMMAND} "s/dc_hadr_standby_servername:/# dc_hadr_standby_servername:/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/dc_hadr_standby_port:/# dc_hadr_standby_port:/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/dc_hadr_retry_interval_for_client_reroute:/# dc_hadr_retry_interval_for_client_reroute:/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/dc_hadr_max_retries_for_client_reroute:/# dc_hadr_max_retries_for_client_reroute:/g" ${CP4A_PATTERN_FILE_TMP}
    fi

    # ensure nodelabel_value is string
    if [[ "$ENABLE_GPU_ARIA" == "Yes" ]]; then
        ${SED_COMMAND} "s|nodelabel_value:.*|nodelabel_value: \"$nodelabel_value\"|g" ${CP4A_PATTERN_FILE_TMP}
    fi
    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_TMP} ${CP4A_PATTERN_FILE_BAK}
    success "Applied value in property file into final CR under $FINAL_CR_FOLDER"
    msgB "Confirm final custom resource under $FINAL_CR_FOLDER"
}

# Begin - Modify FOUNDATION pattern yaml according patterns/components selected
function apply_pattern_cr(){
    # echo -e "\x1B[1mCreating a custom resource YAML file for IBM CP4A Operator ......\x1B[0m"
    # echo "length of optional_component_cr_arr:${#optional_component_cr_arr[@]}"
    # echo "!!optional_component_cr_arr!!!${optional_component_cr_arr[*]}"
    # echo "EXISTING_PATTERN_ARR: ${EXISTING_PATTERN_ARR[*]}"
    # echo "PATTERNS_CR_SELECTED: ${PATTERNS_CR_SELECTED[*]}"
    # echo "EXISTING_OPT_COMPONENT_ARR: ${EXISTING_OPT_COMPONENT_ARR[*]}"
    # echo "OPT_COMPONENTS_CR_SELECTED: ${OPT_COMPONENTS_CR_SELECTED[*]}"
    # echo "FOUNDATION_CR_SELECTED_LOWCASE: ${FOUNDATION_CR_SELECTED_LOWCASE[*]}"
    # echo "FOUNDATION_DELETE_LIST: ${FOUNDATION_DELETE_LIST[*]}"
    # echo "OPTIONAL_COMPONENT_DELETE_LIST: ${OPTIONAL_COMPONENT_DELETE_LIST[*]}"
    # echo "KEEP_COMPOMENTS: ${KEEP_COMPOMENTS[*]}"
    # echo "REMOVED FOUNDATION_CR_SELECTED FROM OPTIONAL_COMPONENT_DELETE_LIST: ${OPTIONAL_COMPONENT_DELETE_LIST[*]}"
    # echo "pattern list in CR: ${pattern_joined}"
    # echo "optional components list in CR: ${opt_components_joined}"
    # echo "length of optional_component_arr:${#optional_component_arr[@]}"

    # read -rsn1 -p"Press Enter/Return to continue (DEBUG MODEL)";echo

    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_BAK} ${CP4A_PATTERN_FILE_TMP}
    # remove merge issue
    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} metadata.labels.app.*

    # Keep existing value
    if [[ "${INSTALLATION_TYPE}" == "existing" ]]; then
        ${YQ_CMD} d -i ${CP4A_EXISTING_TMP} spec.shared_configuration.sc_deployment_patterns
        ${YQ_CMD} d -i ${CP4A_EXISTING_TMP} spec.shared_configuration.sc_optional_components
        # If a starter deployment CR generated from the form UI is supplied it will not need these fields anymore
        #For DBACLD-159390
        keys=("olm_production_workflow_process_service" "olm_production_workflow" "olm_production_document_processing" "olm_production_application" "olm_production_decisions_ads" "olm_production_decisions" "olm_production_content" "olm_production_option" "olm_starter_option" "olm_starter_application" "olm_starter_content" "olm_starter_decisions" "olm_starter_decisions_ads" "olm_starter_document_processing" "olm_starter_workflow")
        for key in "${keys[@]}"; do
            ${YQ_CMD} d -i ${CP4A_EXISTING_TMP} spec.${key}
        done
        ${SED_COMMAND} '/tag: /d' ${CP4A_EXISTING_TMP}
        # updating the namespace of the existing CR passed by the user to whatever namespace value is passed while executing the script
        #For DBACLD-159390
        ${YQ_CMD} w -i ${CP4A_EXISTING_TMP} metadata.namespace $CP4BA_SERVICES_NS
    fi

    ${SED_COMMAND_FORMAT} ${CP4A_PATTERN_FILE_TMP}
    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_TMP} ${CP4A_PATTERN_FILE_BAK}

    tps=" ${OPTIONAL_COMPONENT_DELETE_LIST[*]} "
    for item in ${KEEP_COMPOMENTS[@]}; do
        tps=${tps/ ${item} / }
    done
    OPTIONAL_COMPONENT_DELETE_LIST=( $tps )
    # Convert pattern array to pattern list by common
    delim=""
    pattern_joined=""
    for item in "${PATTERNS_CR_SELECTED[@]}"; do
        if [[ "${DEPLOYMENT_TYPE}" == "starter" ]]; then
            pattern_joined="$pattern_joined$delim$item"
            delim=","
        elif [[ ${DEPLOYMENT_TYPE} == "production" ]]
        then
            case "$item" in
            "workflow-authoring"|"workflow-runtime"|"workflow-workstreams"|"document_processing_designer"|"document_processing_runtime")
                ;;
            *)
                pattern_joined="$pattern_joined$delim$item"
                delim=","
                ;;
            esac
        fi
    done

    if [[ "${DEPLOYMENT_TYPE}" == "starter" ]]; then
        if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams" && "${#PATTERNS_CR_SELECTED[@]}" -eq "1" ]]; then
            echo
        elif [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "application" && "${#PATTERNS_CR_SELECTED[@]}" -eq "1" ]]; then
            echo
        elif [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams" && " ${PATTERNS_CR_SELECTED[@]} " =~ "application" && "${#PATTERNS_CR_SELECTED[@]}" -eq "2" ]]; then
            echo
        else
            pattern_joined="foundation$delim$pattern_joined"
        fi
    else
        if [[ ! ("${#pattern_cr_arr[@]}" -eq "1" && "${pattern_cr_arr[@]}" =~ "workflow-process-service") ]]; then
            pattern_joined="foundation$delim$pattern_joined"
        else
            pattern_joined="$pattern_joined"
        fi

    fi
    # if [[ $INSTALL_BAW_IAWS == "No" ]];then
    #     pattern_joined="foundation$delim$pattern_joined"
    # fi

    # remove cmis for BAW when starter deployment
    local tmp_val="cmis"
    local tmp_idx
    for idx in "${!OPT_COMPONENTS_CR_SELECTED[@]}"; do
        if [[ "${OPT_COMPONENTS_CR_SELECTED[$idx]}" = "${tmp_val}" ]]; then
            tmp_idx=$idx;
        fi
    done

    if [[ "${DEPLOYMENT_TYPE}" == "starter" ]]; then
        if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams" && "${#PATTERNS_CR_SELECTED[@]}" -eq "1" ]]; then
            OPT_COMPONENTS_CR_SELECTED=(${OPT_COMPONENTS_CR_SELECTED[@]:0:$tmp_idx} ${OPT_COMPONENTS_CR_SELECTED[@]:$(($tmp_idx + 1))})
        elif [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams" && " ${PATTERNS_CR_SELECTED[@]} " =~ "application" && "${#PATTERNS_CR_SELECTED[@]}" -eq "2" ]]; then
            OPT_COMPONENTS_CR_SELECTED=(${OPT_COMPONENTS_CR_SELECTED[@]:0:$tmp_idx} ${OPT_COMPONENTS_CR_SELECTED[@]:$(($tmp_idx + 1))})
        fi
    fi

   # Convert optional components array to list by common
    delim=""
    opt_components_joined=""
    for item in "${OPT_COMPONENTS_CR_SELECTED[@]}"; do
        opt_components_joined="$opt_components_joined$delim$item"
        delim=","
    done

    
    merge_pattern
    merge_optional_components
    set_foundation_components

    if [[ $INSTALLATION_TYPE == "existing" ]]; then
        if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-authoring") && !(" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-authoring") ]]; then
            # Delete Object Store for BAW Authoring
            object_array=("BAWDOCS" "BAWDOS" "BAWTOS")
        elif [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-runtime") && !(" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-runtime") ]]; then
            # Delete Object Store for BAW Runtime
            object_array=("BAWINS1DOCS" "BAWINS1DOS" "BAWINS1TOS")
        elif [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && !(" ${PATTERNS_CR_SELECTED[@]} " =~ "workstreams") ]]; then
            # Delete Object Store for workstreams
            object_array=("AWSINS1DOCS")
        else
            object_array=()
        fi
        if (( ${#object_array[@]} >= 1 ));then
            for object_name in "${object_array[@]}"
            do
                containsObjectStore "$object_name" "${CP4A_EXISTING_TMP}"
                if (( ${#os_index_array[@]} >= 1 ));then
                    # ((index_array_temp=${#os_index_array[@]}-1))
                    for ((j=0;j<${#os_index_array[@]};j++))
                    do
                        ((index_os=${os_index_array[$j]}-j))
                        ${YQ_CMD} d -i ${CP4A_EXISTING_TMP} spec.datasource_configuration.dc_os_datasources.[$index_os]
                    done
                fi
                containsInitObjectStore "$object_name" "${CP4A_EXISTING_TMP}"
                if (( ${#os_index_array[@]} >= 1 ));then
                    # ((index_array_temp=${#os_index_array[@]}-1))
                    for ((j=0;j<${#os_index_array[@]};j++))
                    do
                        ((index_os=${os_index_array[$j]}-j))
                        ${YQ_CMD} d -i ${CP4A_EXISTING_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$index_os]
                    done
                fi
            done
            object_array=()
        fi
        if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "content") && !(" ${PATTERNS_CR_SELECTED[@]} " =~ "content") ]]; then
            # Delete Object Store for FNCM
            object_array=("FNOS1DS" "FNOS2DS" "FNOS3DS" "FNOS4DS" "FNOS5DS" "FNOS6DS" "FNOS7DS" "FNOS8DS" "FNOS9DS" "FNOS10DS")
        else
            object_array=()
        fi
        if (( ${#object_array[@]} >= 1 ));then
            for object_name in "${object_array[@]}"
            do
                containsObjectStore "$object_name" "${CP4A_EXISTING_TMP}"
                if (( ${#os_index_array[@]} >= 1 ));then
                    # ((index_array_temp=${#os_index_array[@]}-1))
                    for ((j=0;j<${#os_index_array[@]};j++))
                    do
                        ((index_os=${os_index_array[$j]}-j))
                        ${YQ_CMD} d -i ${CP4A_EXISTING_TMP} spec.datasource_configuration.dc_os_datasources.[$index_os]
                    done
                fi
            done
            object_array=()
        fi

        if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "content") && (" ${PATTERNS_CR_SELECTED[@]} " =~ "content") ]]; then
            total_os_new=0
            total_os_exist=0
            os_index_array_new=()
            os_index_array_exist=()

            getTotalFNCMObjectStore "${CP4A_PATTERN_FILE_TMP}"
            total_os_new=$total_os
            os_index_array_new=( "${os_index_array[@]}" )
            # echo "total_os_new: ${total_os_new}"
            # echo "os_index_array_new: ${os_index_array_new[*]}"
            # echo "length of os_index_array_new:${#os_index_array_new[@]}"

            getTotalFNCMObjectStore "${CP4A_EXISTING_TMP}"
            total_os_exist=$total_os
            os_index_array_exist=( "${os_index_array[@]}" )
            # echo "total_os_exist: ${total_os_exist}"
            # echo "os_index_array_exist: ${os_index_array_exist[*]}"
            # echo "length of os_index_array_exist:${#os_index_array_exist[@]}"
        fi

        if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") && !(" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow") ]]; then
            # Delete BAW Instance
            baw_name_array=("bawins1")
        elif [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "workstreams") && !(" ${PATTERNS_CR_SELECTED[@]} " =~ "workstreams") ]]; then
            baw_name_array=("awsins1")
        else
            baw_name_array=()
        fi
        if (( ${#baw_name_array[@]} >= 1 ));then
            for object_name in "${baw_name_array[@]}"
            do
                containsBAWInstance "$object_name" "${CP4A_EXISTING_TMP}"
                if (( ${#baw_index_array[@]} >= 1 ));then
                    # ((index_array_temp=${#baw_index_array[@]}-1))
                    for ((j=0;j<${#baw_index_array[@]};j++))
                    do
                        ((index_os=${baw_index_array[$j]}-j))
                        ${YQ_CMD} d -i ${CP4A_EXISTING_TMP} spec.baw_configuration
                    done
                fi
            done
            baw_name_array=()
        fi

        if grep "ums_configuration:" $CP4A_EXISTING_TMP > /dev/null
        then
            ${YQ_CMD} w -i ${CP4A_EXISTING_TMP} spec.ums_configuration.fix "dummy"
        fi
        # read -rsn1 -p"Before:Press Enter/Return to exit";echo
        ${YQ_CMD} m -i -a -M --overwrite --autocreate=false ${CP4A_PATTERN_FILE_TMP} ${CP4A_EXISTING_TMP}
        # read -rsn1 -p"After:Press Enter/Return to exit";echo
        ${YQ_CMD} d -i ${CP4A_EXISTING_TMP} spec.ums_configuration.fix
        ${SED_COMMAND} "s|ums_configuration: {}|ums_configuration:|g" ${CP4A_EXISTING_TMP}
        ${SED_COMMAND} "s|ums_configuration: {}|ums_configuration:|g" ${CP4A_PATTERN_FILE_TMP}
    fi

    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_BAK} ${CP4A_PATTERN_FILE_TMP}
    if [[ " ${OPT_COMPONENTS_CR_SELECTED[@]} " =~ "ae_data_persistence" ]]; then
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_content_initialization "true"
    elif [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow" || " ${PATTERNS_CR_SELECTED[@]} " =~ "workstreams" ]]; then
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_content_initialization "true"
    fi

    if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "document_processing" ]]; then
        if [[ "$CPE_FULL_STORAGE" == "Yes" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_cpe_limited_storage "false"
        elif [[ "$CPE_FULL_STORAGE" == "No" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_cpe_limited_storage "true"
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_cpe_limited_storage "false"
        fi
    fi

    # If only select FNCM pattern, only generate "kind: Content" cr
    if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "content" && "${#PATTERNS_CR_SELECTED[@]}" -eq "1" ]]; then
        for item in "${OPT_COMPONENTS_CR_SELECTED[@]}"; do
            while true; do
                case $item in
                    "bai")
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.content_optional_components.bai "true"
                        break
                        ;;
                    "cmis")
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.content_optional_components.cmis "true"
                        break
                        ;;
                    "css")
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.content_optional_components.css "true"
                        break
                        ;;
                    "es")
                        # ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.content_optional_components.es "true"
                        break
                        ;;
                    "iccsap")
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.content_optional_components.iccsap "true"
                        break
                        ;;
                    "ier")
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.content_optional_components.ier "true"
                        break
                        ;;
                    "tm")
                        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.content_optional_components.tm "true"
                        break
                        ;;
                esac
            done
        done
    else
        # Set sc_deployment_patterns
        ${SED_COMMAND} "s|sc_deployment_patterns:.*|sc_deployment_patterns: \"$pattern_joined\"|g" ${CP4A_PATTERN_FILE_TMP}

        # Set sc_optional_components='' when none optional component selected
        if [ "${#optional_component_cr_arr[@]}" -eq "0" ]; then
            ${SED_COMMAND} "s|sc_optional_components:.*|sc_optional_components: \"\"|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${SED_COMMAND} "s|sc_optional_components:.*|sc_optional_components: \"$opt_components_joined\"|g" ${CP4A_PATTERN_FILE_TMP}
        fi
    fi
    # Set sc_deployment_platform
    ${SED_COMMAND} "s|sc_deployment_platform:.*|sc_deployment_platform: \"$PLATFORM_SELECTED\"|g" ${CP4A_PATTERN_FILE_TMP}

    # Set sc_deployment_type
    case "${DEPLOYMENT_TYPE}" in
    starter*|Starter*)
    ${SED_COMMAND} "s|sc_deployment_type:.*|sc_deployment_type: \"Starter\"|g" ${CP4A_PATTERN_FILE_TMP}
    ;;
    production*|Production*)
    ${SED_COMMAND} "s|sc_deployment_type:.*|sc_deployment_type: \"Production\"|g" ${CP4A_PATTERN_FILE_TMP}
    ;;
    esac

    # Set sc_deployment_hostname_suffix

    if [ -z "$existing_infra_name" ]; then
        echo ""
    else
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_hostname_suffix "$existing_infra_name"
        if  [[ $PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS" ]];
        then
            ${SED_COMMAND} "s|sc_deployment_hostname_suffix:.*|sc_deployment_hostname_suffix: \"{{ meta.namespace }}.${INFRA_NAME}\"|g" ${CP4A_PATTERN_FILE_TMP}
        else
            ${SED_COMMAND} "s|sc_deployment_hostname_suffix:.*|sc_deployment_hostname_suffix: \"{{ meta.namespace }}\"|g" ${CP4A_PATTERN_FILE_TMP}
        fi
    fi


    # Set lc_selected_ldap_type

    if [[ $DEPLOYMENT_TYPE == "production" ]];then
        if [[ $LDAP_TYPE == "AD" ]];then
            # ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ldap_configuration.lc_selected_ldap_type "\"Microsoft Active Directory\""
            ${SED_COMMAND} "s|lc_selected_ldap_type:.*|lc_selected_ldap_type: \"Microsoft Active Directory\"|g" ${CP4A_PATTERN_FILE_TMP}

        elif [[ $LDAP_TYPE == "TDS" ]]
        then
            # ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.ldap_configuration.lc_selected_ldap_type "IBM Security Directory Server"
            ${SED_COMMAND} "s|lc_selected_ldap_type:.*|lc_selected_ldap_type: \"IBM Security Directory Server\"|g" ${CP4A_PATTERN_FILE_TMP}
        fi
    fi

    # Set fips_enable
    if  [[ ("$DEPLOYMENT_TYPE" == "starter" || ("$DEPLOYMENT_TYPE" == "production" && $DEPLOYMENT_WITH_PROPERTY == "No")) && ($PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS") ]]; then
        if [[ $FIPS_ENABLED == "true" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.enable_fips "true"
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.enable_fips "false"
        fi
    elif [[ $DEPLOYMENT_WITH_PROPERTY == "Yes" && ($PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS") ]]; then
         fips_flag="$(prop_user_profile_property_file CP4BA.ENABLE_FIPS)"
        fips_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$fips_flag")
        fips_flag=$(echo $fips_flag | tr '[:upper:]' '[:lower:]')
        if [[ ! -z $fips_flag ]]; then
            if [[ $fips_flag == "true" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.enable_fips "true"
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.enable_fips "false"
            fi
        fi
    fi

    # Set sc_generate_sample_network_policies
    if  [[ ("$DEPLOYMENT_TYPE" == "starter" || ("$DEPLOYMENT_TYPE" == "production" && $DEPLOYMENT_WITH_PROPERTY == "No")) && ($PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS") ]]; then
        if [[ $GENERATE_SAMPLE_NETWORK_POLICIES == "true" ]]; then
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_generate_sample_network_policies "true"
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_generate_sample_network_policies "false"
        fi
    elif [[ $DEPLOYMENT_WITH_PROPERTY == "Yes" && ($PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS") ]]; then
        generate_network_policy_flag="$(prop_user_profile_property_file CP4BA.ENABLE_GENERATE_SAMPLE_NETWORK_POLICIES)"
        generate_network_policy_flag=$(sed -e 's/^"//' -e 's/"$//' <<<"$generate_network_policy_flag")
        generate_network_policy_flag=$(echo $generate_network_policy_flag | tr '[:upper:]' '[:lower:]')
        if [[ ! -z $generate_network_policy_flag ]]; then
            if [[ $generate_network_policy_flag == "true" ]]; then
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_generate_sample_network_policies "true"
            else
                ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_generate_sample_network_policies "false"
            fi
        else
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_generate_sample_network_policies "true"
        fi
    fi

    # Set sc_dynamic_storage_classname
    if [[ "$PLATFORM_SELECTED" == "ROKS" ]]; then
        ${SED_COMMAND} "s|sc_dynamic_storage_classname:.*|sc_dynamic_storage_classname: \"${FAST_STORAGE_CLASS_NAME}\"|g" ${CP4A_PATTERN_FILE_TMP}
    else
        ${SED_COMMAND} "s|sc_dynamic_storage_classname:.*|sc_dynamic_storage_classname: \"${STORAGE_CLASS_NAME}\"|g" ${CP4A_PATTERN_FILE_TMP}
    fi
    ${SED_COMMAND} "s|sc_slow_file_storage_classname:.*|sc_slow_file_storage_classname: \"${SLOW_STORAGE_CLASS_NAME}\"|g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s|sc_medium_file_storage_classname:.*|sc_medium_file_storage_classname: \"${MEDIUM_STORAGE_CLASS_NAME}\"|g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s|sc_fast_file_storage_classname:.*|sc_fast_file_storage_classname: \"${FAST_STORAGE_CLASS_NAME}\"|g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s|sc_block_storage_classname:.*|sc_block_storage_classname: \"${BLOCK_STORAGE_CLASS_NAME}\"|g" ${CP4A_PATTERN_FILE_TMP}
    # Set image_pull_secrets
    # ${SED_COMMAND} "s|image-pull-secret|$DOCKER_RES_SECRET_NAME|g" ${CP4A_PATTERN_FILE_TMP}
    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.image_pull_secrets
    ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.image_pull_secrets.[0] "$DOCKER_RES_SECRET_NAME"

    # set sc_drivers_url
    if [ -z "$CP4BA_JDBC_URL" ]; then
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_drivers_url ""
    else
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_drivers_url "$CP4BA_JDBC_URL"
    fi

    # support profile size for production
    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_profile_size "\"$PROFILE_TYPE\""
    fi

    # set the sc_iam.default_admin_username
    if [[ ("$PLATFORM_SELECTED" == "OCP" || "$PLATFORM_SELECTED" == "ROKS") && "$DEPLOYMENT_TYPE" == "production" && "$USE_DEFAULT_IAM_ADMIN" == "No" ]]; then
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_iam.default_admin_username "\"$NON_DEFAULT_IAM_ADMIN\""
    fi

    # set sc_image_repository
    if [ "$use_entitlement" = "yes" ] ; then
        ${SED_COMMAND} "s|sc_image_repository:.*|sc_image_repository: ${DOCKER_REG_SERVER}|g" ${CP4A_PATTERN_FILE_TMP}
    else
        ${SED_COMMAND} "s|sc_image_repository:.*|sc_image_repository: ${CONVERT_LOCAL_REGISTRY_SERVER}|g" ${CP4A_PATTERN_FILE_TMP}
    fi

    # Replace image URL
    old_fmcn="$REGISTRY_IN_FILE\/cp\/cp4a\/fncm"
    old_ban="$REGISTRY_IN_FILE\/cp\/cp4a\/ban"
    old_ums="$REGISTRY_IN_FILE\/cp\/cp4a\/ums"
    old_bas="$REGISTRY_IN_FILE\/cp\/cp4a\/bas"
    old_aae="$REGISTRY_IN_FILE\/cp\/cp4a\/aae"
    old_baca="$REGISTRY_IN_FILE\/cp\/cp4a\/baca"
    old_odm="$REGISTRY_IN_FILE\/cp\/cp4a\/odm"
    old_baw="$REGISTRY_IN_FILE\/cp\/cp4a\/baw"
    old_iaws="$REGISTRY_IN_FILE\/cp\/cp4a\/iaws"
    old_ads="$REGISTRY_IN_FILE\/cp\/cp4a\/ads"
    old_bai="$REGISTRY_IN_FILE\/cp\/cp4a"
    old_workflow="$REGISTRY_IN_FILE\/cp\/cp4a\/workflow"
    old_demo="$REGISTRY_IN_FILE\/cp\/cp4a\/demo"
    old_adp="$REGISTRY_IN_FILE\/cp\/cp4a\/iadp"
    old_ier="$REGISTRY_IN_FILE\/cp\/cp4a\/ier"
    old_iccsap="$REGISTRY_IN_FILE\/cp\/cp4a\/iccsap"

    if [ "$use_entitlement" = "yes" ] ; then
        ${SED_COMMAND} "s/$REGISTRY_IN_FILE/$DOCKER_REG_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
    else
        ${SED_COMMAND} "s/$old_db2/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_db2_alpine/$CONVERT_LOCAL_REGISTRY_SERVER\/alpine/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_ldap/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_db2_etcd/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_busybox/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_demo/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_fmcn/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_ban/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_ums/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_bas/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_aae/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_baca/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_odm/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_baw/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_iaws/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_ads/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_workflow/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_adp/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_ier/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s/$old_iccsap/$CONVERT_LOCAL_REGISTRY_SERVER/g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/imageCredentials:/{n;s/registry:.*/registry: "${CONVERT_LOCAL_REGISTRY_SERVER}"/;}" ${CP4A_PATTERN_FILE_TMP}

    fi

    object_array=("DEVOS1DS" "DEVOS1" "AEOS" "BAWINS1DOCS" "BAWINS1DOS" "BAWINS1TOS" "BAWDOCS" "BAWDOS" "BAWTOS" "AWSINS1DOCS")
    for object_name in "${object_array[@]}"
    do
        containsObjectStore "$object_name" "${CP4A_PATTERN_FILE_TMP}"
        if (( ${#os_index_array[@]} > 1 ));then
            ((index_array_temp=${#os_index_array[@]}-1))
            # read -rsn1 -p"index_array_temp: $index_array_temp";echo
            for ((j=0;j<${index_array_temp};j++))
            do
                ((index_os=${os_index_array[$j]}-j))
                # read -rsn1 -p"index_os: $index_os";echo
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$index_os]
            done
        fi
        containsInitObjectStore "$object_name" "${CP4A_PATTERN_FILE_TMP}"
        if (( ${#os_index_array[@]} > 1 ));then
            ((index_array_temp=${#os_index_array[@]}-1))
            for ((j=0;j<${index_array_temp};j++))
            do
                ((index_os=${os_index_array[$j]}-j))
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$index_os]
            done
        fi
    done

    if (( total_os_new >= total_os_exist ));then
        object_array=()
        for ((j=0;j<${#os_index_array_exist[@]};j++))
        do
            ((num_os=j+1))
            object_array=( "${object_array[@]}" "FNOS${num_os}DS" )
        done

        for object_name in "${object_array[@]}"
        do
            containsObjectStore "$object_name" "${CP4A_PATTERN_FILE_TMP}"
            if (( ${#os_index_array[@]} > 1 ));then
                ((index_array_temp=${#os_index_array[@]}-1))
                for ((j=0;j<${index_array_temp};j++))
                do
                    ((index_os=${os_index_array[$j]}-j))
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$index_os]
                done
            fi
        done
    elif (( total_os_new < total_os_exist ))
    then
        object_array=()
        for ((j=0;j<${#os_index_array_new[@]};j++))
        do
            ((num_os=j+1))
            object_array=( "${object_array[@]}" "FNOS${num_os}DS" )
        done

        for object_name in "${object_array[@]}"
        do
            containsObjectStore "$object_name" "${CP4A_PATTERN_FILE_TMP}"
            if (( ${#os_index_array[@]} > 1 ));then
                ((index_array_temp=${#os_index_array[@]}-1))
                for ((j=0;j<${index_array_temp};j++))
                do
                    ((index_os=${os_index_array[$j]}-j))
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$index_os]
                done
            fi
        done

        object_array_new=()
        object_array_exist=()

        for ((j=0;j<${#os_index_array_new[@]};j++))
        do
            ((num_os=j+1))
            object_array_new=( "${object_array_new[@]}" "FNOS${num_os}DS" )
        done

        for ((j=0;j<${#os_index_array_exist[@]};j++))
        do
            ((num_os=j+1))
            object_array_exist=( "${object_array_exist[@]}" "FNOS${num_os}DS" )
        done
        object_array=($(echo "${object_array_new[@]}" "${object_array_exist[@]}" | tr ' ' '\n' | sort | uniq -u))

        for object_name in "${object_array[@]}"
        do
            containsObjectStore "$object_name" "${CP4A_PATTERN_FILE_TMP}"
            if (( ${#os_index_array[@]} > 0 ));then
                ((index_array_temp=${#os_index_array[@]}))
                for ((j=0;j<${index_array_temp};j++))
                do
                    ((index_os=${os_index_array[$j]}-j))
                    ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$index_os]
                done
            fi
        done
    fi

    containsInitLDAPGroups "${CP4A_PATTERN_FILE_TMP}"
    if (( ${#ldap_groups_index_array[@]} > 1 ));then
        ((index_array_temp=${#ldap_groups_index_array[@]}-1))
        for ((j=0;j<${index_array_temp};j++))
        do
            ((index_os=${ldap_groups_index_array[$j]}-j))
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_ldap_creation.ic_ldap_admins_groups_name.[$index_os]
        done

    fi

    containsInitLDAPUsers "${CP4A_PATTERN_FILE_TMP}"
     if (( ${#ldap_users_index_array[@]} > 1 ));then
        ((index_array_temp=${#ldap_users_index_array[@]}-1))
        for ((j=0;j<${index_array_temp};j++))
        do
            ((index_os=${ldap_users_index_array[$j]}-j))
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_ldap_creation.ic_ldap_admin_user_name.[$index_os]
        done

    fi

    baw_name_array=("bawins1" "awsins1")
    for object_name in "${baw_name_array[@]}"
    do
        containsBAWInstance "$object_name" "${CP4A_PATTERN_FILE_TMP}"
        if (( ${#baw_index_array[@]} > 1 ));then
            ((index_array_temp=${#baw_index_array[@]}-1))
            for ((j=0;j<${index_array_temp};j++))
            do
                ((index_os=${baw_index_array[$j]}-j))
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.baw_configuration.[$index_os]
            done
        fi
    done

    containsAEInstance "${CP4A_PATTERN_FILE_TMP}"
     if (( ${#ae_index_array[@]} > 1 ));then
        ((index_array_temp=${#ae_index_array[@]}-1))
        for ((j=0;j<${index_array_temp};j++))
        do
            ((index_os=${ae_index_array[$j]}-j))
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration.[$index_os]
        done
    fi

    containsICNRepos "${CP4A_PATTERN_FILE_TMP}"
     if (( ${#icn_repo_index_array[@]} > 1 ));then
        ((index_array_temp=${#icn_repo_index_array[@]}-1))
        for ((j=0;j<${index_array_temp};j++))
        do
            ((index_os=${icn_repo_index_array[$j]}-j))
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_icn_init_info.icn_repos.[$index_os]
        done
    fi

    containsICNDesktop "${CP4A_PATTERN_FILE_TMP}"
     if (( ${#icn_desktop_index_array[@]} > 1 ));then
        ((index_array_temp=${#icn_desktop_index_array[@]}-1))
        for ((j=0;j<${index_array_temp};j++))
        do
            ((index_os=${icn_desktop_index_array[$j]}-j))
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_icn_init_info.icn_desktop.[$index_os]
        done
    fi

    containsTenantDB "${CP4A_PATTERN_FILE_TMP}"
     if (( ${#tenant_db_index_array[@]} > 1 ));then
        ((index_array_temp=${#tenant_db_index_array[@]}-1))
        for ((j=0;j<${index_array_temp};j++))
        do
            ((index_os=${tenant_db_index_array[$j]}-j))
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_ca_datasource.tenant_databases.[$index_os]
        done
    fi

    # ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_TMP} ${CP4A_PATTERN_FILE_BAK}
    if [[ "$SCRIPT_MODE" == "baw-dev" || "$SCRIPT_MODE" == "dev" || "$SCRIPT_MODE" == "review" ]]; then
        ${SED_COMMAND} "s|tag: \"${IMAGE_TAG_FINAL}\"|tag: \"${IMAGE_TAG_DEV}\"|g" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "s|tag: ${IMAGE_TAG_FINAL}|tag: \"${IMAGE_TAG_DEV}\"|g" ${CP4A_PATTERN_FILE_TMP}
    fi

    if [[ "$IMAGE_TAG_DEV" != "$IMAGE_TAG_FINAL" && ("$SCRIPT_MODE" == "baw-dev" || "$SCRIPT_MODE" == "dev" || "$SCRIPT_MODE" == "review") ]]; then
        ${SED_COMMAND} "/cp\/cp4a\/fncm\/cpe/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/fncm\/css/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/fncm\/graphql/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/fncm\/cmis/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/fncm\/extshare/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/fncm\/taskmgr/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/ier\/ier/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/iccsap\/iccsap/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/ban\/navigator/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
        ${SED_COMMAND} "/cp\/cp4a\/ban\/navigator-sso/{n;s/tag:.*/tag: \"${IMAGE_TAG_DEV}\"/;}" ${CP4A_PATTERN_FILE_TMP}
    fi

    if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "document_processing" ]]; then
        ${SED_COMMAND} "s/.*# ecm_configuration:.*/  # ecm_configuration:/g" ${CP4A_PATTERN_FILE_TMP}
    fi

    if [[ $PLATFORM_SELECTED == "other" ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.storage_configuration.sc_block_storage_classname
    fi

    # if [[ $DEPLOYMENT_TYPE == "starter" ]]; then
    #     if [[ (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams" || " ${PATTERNS_CR_SELECTED[@]} " =~ "decisions_ads") && !(" ${PATTERNS_CR_SELECTED[@]} " =~ "document_processing") && !(" ${PATTERNS_CR_SELECTED[@]} " =~ "application") ]]; then
    #         ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server
    #     fi
    # fi

    # Remove dc_os_datasources for `FNOS1DS` if content_os_number = 0
    if [[ $DEPLOYMENT_TYPE == "production" && $content_os_number -eq 0 ]]; then
        OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'FNOS1DS'|cut -d':' -f1)
        # prompt_press_any_key_to_continue "$OS_DATASOURCE_NUMBER"
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
        OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER]
        fi
    fi

    # Apply value in property file into final cr
    if [[ $DEPLOYMENT_TYPE == "production" && $DEPLOYMENT_WITH_PROPERTY == "Yes" ]]; then
        sync_property_into_final_cr
    fi

    # Format value
    ${SED_COMMAND} "s|'\"|\"|g" ${CP4A_PATTERN_FILE_TMP}
    ${SED_COMMAND} "s|\"'|\"|g" ${CP4A_PATTERN_FILE_TMP}
    # remove ldap_configuration and datasource_configuration when only select WfPS authoring
    if [[ "${#pattern_cr_arr[@]}" -eq "1" && "${pattern_cr_arr[@]}" =~ "workflow-process-service" && $LDAP_WFPS_AUTHORING == "No" ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ldap_configuration
    fi

    if [[ "${#pattern_cr_arr[@]}" -eq "1" && "${pattern_cr_arr[@]}" =~ "workflow-process-service" && $EXTERNAL_DB_WFPS_AUTHORING == "No" ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.admin_secret_name
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.database
    fi

    if [[ "${#pattern_cr_arr[@]}" -eq "1" && "${pattern_cr_arr[@]}" =~ "workflow-process-service" ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_fncm_license
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_baw_license
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_content_initialization
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_cpe_limited_storage
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration
    fi

    # remove application_engine_configuration/playback_server when only select BAW authoring/only WfPS authoring/both BAW authoring and WfPS authoring
    if [[ (! (" ${pattern_cr_arr[@]}" =~ "document_processing" || " ${pattern_cr_arr[@]}" =~ "application" || " ${pattern_cr_arr[@]}" =~ "workflow-runtime" || " ${pattern_cr_arr[@]}" =~ "workstreams")) && ("${pattern_cr_arr[@]}" =~ "workflow-authoring" || "${pattern_cr_arr[@]}" =~ "workflow-process-service") ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration
    fi

    if [[ (! (" ${pattern_cr_arr[@]}" =~ "document_processing_designer" || " ${optional_component_cr_arr[@]}" =~ "app_designer" || " ${optional_component_cr_arr[@]}" =~ "ads_designer")) && ("${pattern_cr_arr[@]}" =~ "workflow-authoring" || "${pattern_cr_arr[@]}" =~ "workflow-process-service") ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.bastudio_configuration.playback_server
    fi
    # remove gcd/aeos/init without ae data persistent when only select BAA pattern

    if [[ "${#pattern_cr_arr[@]}" -eq "1" && "${pattern_cr_arr[@]}" =~ "application" && (! "${optional_component_cr_arr[@]}" =~ "ae_data_persistence") ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_gcd_datasource
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration
    fi

    if [[ "${#pattern_cr_arr[@]}" -gt "1" && (! "${optional_component_cr_arr[@]}" =~ "ae_data_persistence") ]]; then
        OS_DATASOURCE_NUMBER=$(grep "^      dc_common_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AEOS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_os_datasources.[$OS_DATASOURCE_NUMBER]
        fi

        OS_DATASOURCE_NUMBER=$(grep "^          dc_os_datasource_name: " ${CP4A_PATTERN_FILE_TMP} | grep -Fn 'AEOS'|cut -d':' -f1)
        if [[ -n $OS_DATASOURCE_NUMBER && $OS_DATASOURCE_NUMBER -gt 0 ]]; then
            OS_DATASOURCE_NUMBER=$(( OS_DATASOURCE_NUMBER - 1 ))
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.initialize_configuration.ic_obj_store_creation.object_stores.[$OS_DATASOURCE_NUMBER]
        fi
    fi

    if [[ $DEPLOYMENT_TYPE == "production" ]]; then
        # 5b/6: Remove pfs_configuration/application_engine_configuration ae_data_persistence/AEOS
        if [[ " ${pattern_cr_arr[@]} " =~ "application" || " ${pattern_cr_arr[@]} " =~ "document_processing" ]]; then
            echo # keep application_engine_configuration for BAA/ADP
        else
            if [[ (" ${pattern_cr_arr[@]} " =~ "workstreams") || (" ${pattern_cr_arr[@]} " =~ "workflow-runtime") ]]; then
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.pfs_configuration
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.application_engine_configuration
                ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.elasticsearch_configuration
            fi
        fi
    # 6: remove Navigator/GraphQL
        if [[ " ${pattern_cr_arr[@]} " =~ "workstreams" && "${#pattern_cr_arr[@]}" -eq "1" ]]; then
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.graphql
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.ecm_configuration.navigator_configuration
            ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.datasource_configuration.dc_icn_datasource
        fi
    fi

    # For ARO/ROSA platform type
    if [[ $OCP_PLATFORM == "ARO" ]]; then
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_ocp_platform "ARO"
    elif [[ $OCP_PLATFORM == "ROSA" ]]; then
        ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_ocp_platform "ROSA"
    fi

    # sc_deployment_baw_license required for either workflow runtime or workflow authoring
    # For https://jsw.ibm.com/browse/DBACLD-161792
    if [[ ! (" ${pattern_cr_arr[@]}" =~ "workflow-runtime" || " ${pattern_cr_arr[@]}" =~ "workflow-authoring") ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_baw_license
    fi

    # Remove sc_deployment_fncm_license
    if [[ ! (" ${pattern_cr_arr[@]}" =~ "workflow-runtime" || " ${pattern_cr_arr[@]}" =~ "workflow-authoring" || " ${pattern_cr_arr[@]}" =~ "workstreams" || " ${pattern_cr_arr[@]}" =~ "content" || " ${pattern_cr_arr[@]}" =~ "document_processing" || "${optional_component_cr_arr[@]}" =~ "ae_data_persistence") ]]; then
        ${YQ_CMD} d -i ${CP4A_PATTERN_FILE_TMP} spec.shared_configuration.sc_deployment_baw_license
    fi

    # while [[ $TARGET_PROJECT_NAME == "" ]];
    # do
    #     printf "\n"
    #     echo -e "\x1B[1mWhere (namespace) do you want to deploy CP4BA operands (i.e., runtime pods)? \x1B[0m"
    #     read -p "Enter the name for an existing project (namespace): " TARGET_PROJECT_NAME
    #     if [ -z "$TARGET_PROJECT_NAME" ]; then
    #         echo -e "\x1B[1;31mEnter a valid project name, project name can not be blank\x1B[0m"
    #     elif [[ "$TARGET_PROJECT_NAME" == openshift* ]]; then
    #         echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'openshift' or start with 'openshift' \x1B[0m"
    #         TARGET_PROJECT_NAME=""
    #     elif [[ "$TARGET_PROJECT_NAME" == kube* ]]; then
    #         echo -e "\x1B[1;31mEnter a valid project name, project name should not be 'kube' or start with 'kube' \x1B[0m"
    #         TARGET_PROJECT_NAME=""
    #     else
    #         isProjExists=`${CLI_CMD} get project $TARGET_PROJECT_NAME --ignore-not-found | wc -l`  >/dev/null 2>&1

    #         if [ "$isProjExists" -ne 2 ] ; then
    #             echo -e "\x1B[1;31mInvalid project name, please enter a existing project name ...\x1B[0m"
    #             TARGET_PROJECT_NAME=""
    #         else
    #             echo -e "\x1B[1mUsing project ${TARGET_PROJECT_NAME}...\x1B[0m"
    #         fi
    #     fi
    # done

    # Apply separation of operands into final CR
    if ${CLI_CMD} get configMap ibm-cp4ba-common-config -n $CP4BA_SERVICES_NS >/dev/null 2>&1; then
        cp4ba_services_namespace=$(${CLI_CMD} get configMap ibm-cp4ba-common-config -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o jsonpath='{.data.services_namespace}')
        cp4ba_operators_namespace=$(${CLI_CMD} get configMap ibm-cp4ba-common-config -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found -o jsonpath='{.data.operators_namespace}')
        if [[ (! -z $cp4ba_services_namespace) && (! -z $cp4ba_operators_namespace) ]]; then
            if [[ $cp4ba_services_namespace != $cp4ba_operators_namespace ]]; then
                info "This CP4BA deployment is separation of operators and operands"
            fi
            info "the script will set \"$cp4ba_services_namespace\" as the namespace in the final custom resource."
            ${YQ_CMD} w -i ${CP4A_PATTERN_FILE_TMP} metadata.namespace "$cp4ba_services_namespace" --style=double
        else
            warning "\"services_namespace\" was not found in the configMap ibm-cp4ba-common-config in the project \"$CP4BA_SERVICES_NS\""
            info "You need to apply the custom resource in the project for CP4BA operand, not in the project for CP4BA operators."
        fi
    else
        warning "ibm-cp4ba-common-config configmap was not found in the project \"$CP4BA_SERVICES_NS\"."
        # For https://jsw.ibm.com/browse/DBACLD-160661 where we have added remediation steps on how to recreate the configmap
        fail "You NEED to first create the \"ibm-cp4ba-common-config\" configMap in the project (namespace) where you want to deploy or upgrade CP4BA operands (i.e., runtime pods)."
        info "${YELLOW_TEXT}- [NEXT-STEPS]${RESET_TEXT}"
        echo "  - STEP 1 ${RED_TEXT}(Required)${RESET_TEXT}:${GREEN_TEXT} # Execute the cp4a-clusteradmin-setup.sh script with the \"-fix_configmap\" option to re-create the missing \"ibm-cp4ba-common-config\" configMap in the target namespace.For additional information refer to the Troubleshooting page in the Upgrade Section of the Knowledge Center.${RESET_TEXT}"
        exit 1
    fi

    # rename final CR to ibm_content_cr_final.yaml
    if [[ " ${PATTERNS_CR_SELECTED[@]} " =~ "content" && "${#PATTERNS_CR_SELECTED[@]}" -eq "1" ]]; then
        CP4A_PATTERN_FILE_BAK=$FNCM_SEPARATE_PATTERN_FILE_BAK
    fi

    ${COPY_CMD} -rf ${CP4A_PATTERN_FILE_TMP} ${CP4A_PATTERN_FILE_BAK}

    if [[ "$DEPLOYMENT_TYPE" == "starter" && "$INSTALLATION_TYPE" == "new" && !("$SCRIPT_MODE" == "review" || "$SCRIPT_MODE" == "OLM") ]];then
        ${CLI_CMD} delete -f ${CP4A_PATTERN_FILE_TMP} >/dev/null 2>&1
        sleep 5
        printf "\n"
        echo -e "\x1B[1mInstalling the selected Cloud Pak capability...\x1B[0m"

        if [[ "${ALL_NAMESPACE}" == "Yes" ]]; then
            APPLY_CONTENT_CMD="${CLI_CMD} apply -f ${CP4A_PATTERN_FILE_BAK} -n openshift-operators"
        else
            APPLY_CONTENT_CMD="${CLI_CMD} apply -f ${CP4A_PATTERN_FILE_BAK} -n $CP4BA_SERVICES_NS"
        fi
        if $APPLY_CONTENT_CMD ; then
            echo -e "\x1B[1mDone\x1B[0m"
        else
            echo -e "\x1B[1;31mFailed\x1B[0m"
        fi
    elif  [[ "$DEPLOYMENT_TYPE" == "starter" && "$INSTALLATION_TYPE" == "existing" && !("$SCRIPT_MODE" == "review" || "$SCRIPT_MODE" == "OLM") ]]
    then
        echo "Applying the existing Custom Resource file supplied to the cluster...\n"
        # Merging the existing CR sections to the CR that will be applied
        # For https://jsw.ibm.com/browse/DBACLD-159390
        ${YQ_CMD} m -a -i -M ${CP4A_PATTERN_FILE_BAK} ${CP4A_EXISTING_TMP}
        echo -e "\x1B[1mInstalling the selected Cloud Pak capability...\x1B[0m"

        if [[ "${ALL_NAMESPACE}" == "Yes" ]]; then
            APPLY_CONTENT_CMD="${CLI_CMD} apply -f ${CP4A_PATTERN_FILE_BAK} -n openshift-operators"
        else
            APPLY_CONTENT_CMD="${CLI_CMD} apply -f ${CP4A_PATTERN_FILE_BAK} -n $CP4BA_SERVICES_NS"
        fi

        if $APPLY_CONTENT_CMD ; then
            echo -e "\x1B[1mDone\x1B[0m"
        else
            echo -e "\x1B[1;31mFailed\x1B[0m"
        fi
    elif  [[ "$DEPLOYMENT_TYPE" == "production" && "$INSTALLATION_TYPE" == "new" && "$DEPLOYMENT_WITH_PROPERTY" == "Yes" ]]
    then
        ## CP4BA_APPLY_CR is going to be a environment variable to apply the CR for silent install.
        if [[ "$CP4BA_APPLY_CR" == "Yes" || "$CP4BA_APPLY_CR" == "YES" || "$CP4BA_APPLY_CR" == "yes" || "$CP4BA_APPLY_CR" == "True"  || "$CP4BA_APPLY_CR" == "TRUE"  || "$CP4BA_APPLY_CR" == "true" ]]; then
           echo -e "\x1B[1mInstalling the selected Cloud Pak capability...\x1B[0m"
           echo -e "${CP4A_PATTERN_FILE_BAK}"
           APPLY_CUSTOM_RESOURCE_CMD="${CLI_CMD} apply -f ${CP4A_PATTERN_FILE_BAK} -n $CP4BA_SERVICES_NS"
           if $APPLY_CUSTOM_RESOURCE_CMD ; then
               echo -e "\x1B[1mDone\x1B[0m"
           else
               echo -e "\x1B[1;31mFailed\x1B[0m"
           fi
        else
            printf "\n"
            echo -e "${YELLOW_TEXT}[NOTE]${RESET_TEXT} The custom resource (CR) file has been generated, but is not yet deployed (applied).\n"

            echo -e "${YELLOW_TEXT}[ATTENTION]${RESET_TEXT} Before deploying (applying) the custom resource (CR), follow the steps in the Knowledge Center ${BLUE_TEXT}\"Check and complete your custom resource\"${RESET_TEXT} to add or update any additional configuration to the custom resource file for the capabilities you have selected, which are not configured by the script: ${BLUE_TEXT} https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/$CP4BA_RELEASE_BASE?topic=deployment-checking-completing-your-custom-resource ${RESET_TEXT}  \n"

            echo -e "${YELLOW_TEXT}[ATTENTION]${RESET_TEXT} After finishing configuring the custom resource (CR) file, follow the steps in the Knowledge Center to deploy (apply) your custom resource:${BLUE_TEXT} https://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/$CP4BA_RELEASE_BASE?topic=cpd-option-1b-deploying-custom-resource-you-created-deployment-script ${RESET_TEXT} \n"
        fi
    fi

    echo -e "\x1B[1mThe custom resource file is located at: \"${CP4A_PATTERN_FILE_BAK}\"\x1B[0m"
    if [[ "$DEPLOYMENT_TYPE" == "production" && (" ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-workstreams" || " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-runtime" || " ${PATTERNS_CR_SELECTED[@]} " =~ "workflow-authoring") ]]; then
        printf "\n"
        echo -e "\x1B[33;5m[ATTENTION]: \x1B[0m\x1B[1mIf the cluster is running a Linux on Z (s390x)/Power architecture, remove the \x1B[0m\x1B[1;31mbaml_configuration\x1B[0m \x1B[1msection from \"${CP4A_PATTERN_FILE_BAK}\" before applying the custom resource. Business Automation Machine Learning Server (BAML) is not supported on this architecture.\n\x1B[0m"
    fi
    printf "\n"
    echo -e "\x1B[1mTo monitor the deployment status, follow the Operator logs.\x1B[0m"
    echo -e "\x1B[1mFor details, refer to the troubleshooting section in Knowledge Center here: \x1B[0m"
    echo -e "\x1B[1mhttps://www.ibm.com/docs/en/cloud-paks/cp-biz-automation/$CP4BA_RELEASE_BASE?topic=automation-troubleshooting\x1B[0m"
}
# End - Modify FOUNDATION pattern yaml according pattent/components selected

function show_summary_pattern_selected(){
    printf "\n"
    echo -e "\x1B[1m*******************************************************\x1B[0m"
    echo -e "\x1B[1m            Summary of CP4BA capabilities              \x1B[0m"
    echo -e "\x1B[1m*******************************************************\x1B[0m"

    echo -e "\x1B[1;31m1. Cloud Pak capability to deploy: \x1B[0m"
    if [ "${#pattern_arr[@]}" -eq "0" ]; then
        printf '   * %s\n' "None"
    else
        for each_pattern in "${pattern_arr[@]}"
        do
            if [[ "$each_pattern" =~ .*"Workflow Authoring".* || "$each_pattern" =~ .*"Workflow Runtime".* || "$each_pattern" =~ .*"Development Environment".* || "$each_pattern" =~ .*"Runtime Environment".* ]];then
               printf '     %s\n' "${each_pattern}"
            else
                printf '   * %s\n' "${each_pattern}"
            fi
        done
    fi

    echo -e "\x1B[1;31m2. Optional components to deploy: \x1B[0m"
    if [ "${#optional_component_arr[@]}" -eq "0" ]; then
        printf '   * %s\n' "None"
    else
        # printf '   * %s\n' "${OPT_COMPONENTS_SELECTED[@]}"
        for each_opt_component in "${optional_component_arr[@]}"
        do
            if [[ ${each_opt_component} == "ExternalShare" ]]; then
                printf '   * %s\n' "External Share"
            elif [[ ${each_opt_component} == "TaskManager" ]]
            then
                printf '   * %s\n' "Task Manager"
            elif [[ ${each_opt_component} == "ContentSearchServices" ]]
            then
                printf '   * %s\n' "Content Search Services"
            elif [[ ${each_opt_component} == "DecisionCenter" ]]
            then
                printf '   * %s\n' "Decision Center"
            elif [[ ${each_opt_component} == "RuleExecutionServer" ]]
            then
                printf '   * %s\n' "Rule Execution Server"
            elif [[ ${each_opt_component} == "DecisionRunner" ]]
            then
                printf '   * %s\n' "Decision Runner"
            elif [[ ${each_opt_component} == "DecisionDesigner" ]]
            then
                printf '   * %s\n' "Decision Designer"
            elif [[ ${each_opt_component} == "DecisionRuntime" ]]
            then
                printf '   * %s\n' "Decision Runtime"
            elif [[ "${each_opt_component}" == "ContentManagementInteroperabilityServices" ]]
            then
                printf '   * %s\n' "Content Management Interoperability Services"
            elif [[ "${each_opt_component}" == "UserManagementService" ]]
            then
                printf '   * %s\n' "User Management Service"
            elif [[ "${each_opt_component}" == "BusinessAutomationInsights" ]]
            then
                printf '   * %s\n' "Business Automation Insights"
            elif [[ "${each_opt_component}" == "ProcessFederationServer" ]]
            then
                printf '   * %s\n' "Process Federation Server"
            elif [[ "${each_opt_component}" == "DataCollectorandDataIndexer" ]]
            then
                printf '   * %s\n' "Data Collector and Data Indexer"
            elif [[ "${each_opt_component}" == "ExposedKafkaServices" ]]
            then
                printf '   * %s\n' "Exposed Kafka Services"
            elif [[ "${each_opt_component}" == "ExposedOpenSearch" ]]
            then
                printf '   * %s\n' "Exposed OpenSearch"
            elif [[ "${each_opt_component}" == "BusinessAutomationMachineLearning" ]]
            then
                printf '   * %s\n' "Business Automation Machine Learning"
            elif [[ "${each_opt_component}" == "ApplicationDesigner" ]]
            then
                printf '   * %s\n' "Application Designer"
            elif [[ "${each_opt_component}" == "BusinessAutomationApplicationDataPersistence" ]]
            then
                printf '   * %s\n' "Business Automation Application Data Persistence"
            elif [[ "${each_opt_component}" == "IBMEnterpriseRecords" ]]
            then
                printf '   * %s\n' "IBM Enterprise Records"
            elif [[ "${each_opt_component}" == "IBMContentCollectorforSAP" ]]
            then
                printf '   * %s\n' "IBM Content Collector for SAP"
            elif [[ "${each_opt_component}" == "IBMContentNavigator" ]]
            then
                printf '   * %s\n' "IBM Content Navigator"
            elif [[ "${each_opt_component}" == "ContentIntegration" ]]
            then
                printf '   * %s\n' "Content Integration"
            else
                printf '   * %s\n' "${each_opt_component}"
            fi
        done
    fi
    echo -e "\x1B[1m*******************************************************\x1B[0m"
    info "Above CP4BA capabilities is already selected in the cp4a-prerequisites.sh script"
    prompt_press_any_key_to_continue
}

function show_summary(){
    printf "\n"
    echo -e "\x1B[1m*******************************************************\x1B[0m"
    echo -e "\x1B[1m                    Summary of input                   \x1B[0m"
    echo -e "\x1B[1m*******************************************************\x1B[0m"

    echo -e "\x1B[1;31m1. Cloud Pak capability to deploy: \x1B[0m"
    if [ "${#pattern_arr[@]}" -eq "0" ]; then
        printf '   * %s\n' "None"
    else
        for each_pattern in "${pattern_arr[@]}"
        do
            if [[ "$each_pattern" =~ .*"Workflow Authoring".* || "$each_pattern" =~ .*"Workflow Runtime".* || "$each_pattern" =~ .*"Development Environment".* || "$each_pattern" =~ .*"Runtime Environment".* ]];then
               printf '     %s\n' "${each_pattern}"
            else
                printf '   * %s\n' "${each_pattern}"
            fi
        done
    fi

    echo -e "\x1B[1;31m2. Optional components to deploy: \x1B[0m"
    if [ "${#OPT_COMPONENTS_SELECTED[@]}" -eq "0" ]; then
        printf '   * %s\n' "None"
    else
        # printf '   * %s\n' "${OPT_COMPONENTS_SELECTED[@]}"
        for each_opt_component in "${OPT_COMPONENTS_SELECTED[@]}"
        do
            if [[ ${each_opt_component} == "ExternalShare" ]]; then
                printf '   * %s\n' "External Share"
            elif [[ ${each_opt_component} == "TaskManager" ]]
            then
                printf '   * %s\n' "Task Manager"
            elif [[ ${each_opt_component} == "ContentSearchServices" ]]
            then
                printf '   * %s\n' "Content Search Services"
            elif [[ ${each_opt_component} == "DecisionCenter" ]]
            then
                printf '   * %s\n' "Decision Center"
            elif [[ ${each_opt_component} == "RuleExecutionServer" ]]
            then
                printf '   * %s\n' "Rule Execution Server"
            elif [[ ${each_opt_component} == "DecisionRunner" ]]
            then
                printf '   * %s\n' "Decision Runner"
            elif [[ ${each_opt_component} == "DecisionDesigner" ]]
            then
                printf '   * %s\n' "Decision Designer"
            elif [[ ${each_opt_component} == "DecisionRuntime" ]]
            then
                printf '   * %s\n' "Decision Runtime"
            elif [[ "${each_opt_component}" == "ContentManagementInteroperabilityServices" ]]
            then
                printf '   * %s\n' "Content Management Interoperability Services"
            elif [[ "${each_opt_component}" == "UserManagementService" ]]
            then
                printf '   * %s\n' "User Management Service"
            elif [[ "${each_opt_component}" == "BusinessAutomationInsights" ]]
            then
                printf '   * %s\n' "Business Automation Insights"
            elif [[ "${each_opt_component}" == "ProcessFederationServer" ]]
            then
                printf '   * %s\n' "Process Federation Server"
            elif [[ "${each_opt_component}" == "DataCollectorandDataIndexer" ]]
            then
                printf '   * %s\n' "Data Collector and Data Indexer"
            elif [[ "${each_opt_component}" == "ExposedKafkaServices" ]]
            then
                printf '   * %s\n' "Exposed Kafka Services"
            elif [[ "${each_opt_component}" == "ExposedOpenSearch" ]]
            then
                printf '   * %s\n' "Exposed OpenSearch"
            elif [[ "${each_opt_component}" == "BusinessAutomationMachineLearning" ]]
            then
                printf '   * %s\n' "Business Automation Machine Learning"
            elif [[ "${each_opt_component}" == "ApplicationDesigner" ]]
            then
                printf '   * %s\n' "Application Designer"
            elif [[ "${each_opt_component}" == "BusinessAutomationApplicationDataPersistence" ]]
            then
                printf '   * %s\n' "Business Automation Application Data Persistence"
            elif [[ "${each_opt_component}" == "IBMEnterpriseRecords" ]]
            then
                printf '   * %s\n' "IBM Enterprise Records"
            elif [[ "${each_opt_component}" == "IBMContentCollectorforSAP" ]]
            then
                printf '   * %s\n' "IBM Content Collector for SAP"
            elif [[ "${each_opt_component}" == "IBMContentNavigator" ]]
            then
                printf '   * %s\n' "IBM Content Navigator"
            elif [[ "${each_opt_component}" == "ContentIntegration" ]]
            then
                printf '   * %s\n' "Content Integration"
            else
                printf '   * %s\n' "${each_opt_component}"
            fi
        done
    fi

    if [[ $PLATFORM_SELECTED == "other" ]]; then
        echo -e "\x1B[1;31m3. Entitlement Registry key:\x1B[0m" # not show plaintext password
        echo -e "\x1B[1;31m4. Docker registry service name or URL:\x1B[0m ${LOCAL_REGISTRY_SERVER}"
        echo -e "\x1B[1;31m5. Docker registry user name:\x1B[0m ${LOCAL_REGISTRY_USER}"
        # echo -e "\x1B[1;31m5. Docker registry password: ${LOCAL_REGISTRY_PWD}\x1B[0m"
        echo -e "\x1B[1;31m6. Docker registry password:\x1B[0m" # not show plaintext password
    fi
    if  [[ $PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS" ]];
    then
        if [ -z "$existing_infra_name" ]; then
            if  [[ $DEPLOYMENT_TYPE == "starter" && $PLATFORM_SELECTED == "OCP" ]];
            then
                echo -e "\x1B[1;31m3. File storage classname(RWX):\x1B[0m ${STORAGE_CLASS_NAME}"
                echo -e "\x1B[1;31m4. Block storage classname(RWO):\x1B[0m ${BLOCK_STORAGE_CLASS_NAME}"
                if [[ " ${optional_component_cr_arr[@]} " =~ "iccsap" ]]; then
                    echo -e "\x1B[1;31m5. URL to zip file for JDBC and/or ICCSAP drivers:\x1B[0m ${CP4BA_JDBC_URL}"
                fi
            else
                echo -e "\x1B[1;31m3. File storage classname(RWX):\x1B[0m"
                printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Slow:" "${SLOW_STORAGE_CLASS_NAME}"
                printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Medium:" "${MEDIUM_STORAGE_CLASS_NAME}"
                printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Fast:" "${FAST_STORAGE_CLASS_NAME}"
                echo -e "\x1B[1;31m4. Block storage classname(RWO): \x1B[0m${BLOCK_STORAGE_CLASS_NAME}"
                echo -e "\x1B[1;31m5. URL to zip file for JDBC and/or ICCSAP drivers:\x1B[0m ${CP4BA_JDBC_URL}"
            fi
        else
            if  [[ $PLATFORM_SELECTED == "OCP" ]]; then
                echo -e "\x1B[1;31m3. OCP Infrastructure Node:\x1B[0m ${INFRA_NAME}"
            elif [[ $PLATFORM_SELECTED == "ROKS" ]]
            then
                echo -e "\x1B[1;31m3. ROKS Infrastructure Node:\x1B[0m ${INFRA_NAME}"
            fi
            if  [[ $DEPLOYMENT_TYPE == "starter" && $PLATFORM_SELECTED == "OCP" ]];
            then
                echo -e "\x1B[1;31m4. File storage classname(RWX):\x1B[0m ${STORAGE_CLASS_NAME}"
                echo -e "\x1B[1;31m5. Block storage classname(RWO):\x1B[0m ${BLOCK_STORAGE_CLASS_NAME}"
                if [[ " ${optional_component_cr_arr[@]} " =~ "iccsap" ]]; then
                    echo -e "\x1B[1;31m6. URL to zip file for JDBC and/or ICCSAP drivers:\x1B[0m ${CP4BA_JDBC_URL}"
                fi
            else
                echo -e "\x1B[1;31m4. File storage classname(RWX):\x1B[0m"
                printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Slow:" "${SLOW_STORAGE_CLASS_NAME}"
                printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Medium:" "${MEDIUM_STORAGE_CLASS_NAME}"
                printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Fast:" "${FAST_STORAGE_CLASS_NAME}"
                echo -e "\x1B[1;31m5. Block storage classname(RWO): \x1B[0m${BLOCK_STORAGE_CLASS_NAME}"
                echo -e "\x1B[1;31m6. URL to zip file for JDBC and/or ICCSAP drivers:\x1B[0m ${CP4BA_JDBC_URL}"
            fi
        fi
    else
        if  [[ $DEPLOYMENT_TYPE == "starter" ]];
        then
            echo -e "\x1B[1;31m3. File storage classname(RWX):\x1B[0m ${STORAGE_CLASS_NAME}"
            echo -e "\x1B[1;31m4. Block storage classname(RWO):\x1B[0m ${BLOCK_STORAGE_CLASS_NAME}"
            if [[ " ${optional_component_cr_arr[@]} " =~ "iccsap" ]]; then
                echo -e "\x1B[1;31m5. URL to zip file for JDBC and/or ICCSAP drivers:\x1B[0m ${CP4BA_JDBC_URL}"
            fi
        else
            if [[ $PLATFORM_SELECTED == "other" ]]; then
                echo -e "\x1B[1;31m7. File storage classname(RWX):\x1B[0m"
            else
                echo -e "\x1B[1;31m3. File storage classname(RWX):\x1B[0m"
            fi
        fi
        printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Slow:" "${SLOW_STORAGE_CLASS_NAME}"
        printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Medium:" "${MEDIUM_STORAGE_CLASS_NAME}"
        printf '   * \x1B[1;31m%s\x1B[0m %s\n' "Fast:" "${FAST_STORAGE_CLASS_NAME}"
        if [[ $PLATFORM_SELECTED == "other" ]]; then
            echo -e "\x1B[1;31m8. URL to zip file for JDBC and/or ICCSAP drivers:\x1B[0m ${CP4BA_JDBC_URL}"
        fi
        if [[ $PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS" ]]; then
            echo -e "\x1B[1;31m4. Block storage classname(RWO):\x1B[0m ${BLOCK_STORAGE_CLASS_NAME}"
            echo -e "\x1B[1;31m5. URL to zip file for JDBC and/or ICCSAP drivers:\x1B[0m ${CP4BA_JDBC_URL}"
        fi
    fi
    echo -e "\x1B[1m*******************************************************\x1B[0m"
}

function prepare_pattern_file(){
    ${COPY_CMD} -rf "${OPERATOR_FILE}" "${OPERATOR_FILE_BAK}"
    # ${COPY_CMD} -rf "${OPERATOR_PVC_FILE}" "${OPERATOR_PVC_FILE_BAK}"

    if [[ "$DEPLOYMENT_TYPE" == "production" ]];then
        DEPLOY_TYPE_IN_FILE_NAME="production"
    else
        DEPLOY_TYPE_IN_FILE_NAME="starter"
    fi

    FOUNDATION_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_foundation.yaml

    CONTENT_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_content.yaml
    CONTENT_SEPARATE_PATTERN_FILE=${PARENT_DIR}/descriptors/sub-operator/FNCM/ibm_content_cr_${DEPLOY_TYPE_IN_FILE_NAME}.yaml

    CONTENT_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_content_tmp.yaml
    CONTENT_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_content.yaml

    APPLICATION_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_application.yaml
    APPLICATION_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_application_tmp.yaml
    APPLICATION_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_application.yaml

    DECISIONS_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_decisions.yaml
    DECISIONS_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_decisions_tmp.yaml
    DECISIONS_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_decisions.yaml

    ADS_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_decisions_ads.yaml
    ADS_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_decisions_ads_tmp.yaml
    ADS_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_decisions_ads.yaml

    ARIA_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_document_processing.yaml
    ARIA_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_document_processing_tmp.yaml
    ARIA_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_document_processing.yaml

    ${COPY_CMD} -rf "${CONTENT_PATTERN_FILE}" "${CONTENT_PATTERN_FILE_BAK}"
    ${COPY_CMD} -rf "${APPLICATION_PATTERN_FILE}" "${APPLICATION_PATTERN_FILE_BAK}"
    ${COPY_CMD} -rf "${ADS_PATTERN_FILE}" "${ADS_PATTERN_FILE_BAK}"
    ${COPY_CMD} -rf "${DECISIONS_PATTERN_FILE}" "${DECISIONS_PATTERN_FILE_BAK}"
    ${COPY_CMD} -rf "${ARIA_PATTERN_FILE}" "${ARIA_PATTERN_FILE_BAK}"

    ${COPY_CMD} -rf "${FOUNDATION_PATTERN_FILE}" "${CP4A_PATTERN_FILE_TMP}"
    if [[ "$DEPLOYMENT_TYPE" == "starter" ]];then
        WORKFLOW_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow.yaml
        WORKFLOW_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_tmp.yaml
        WORKFLOW_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow.yaml

        WW_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_authoring-workstreams.yaml
        WW_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_authoring-workstreams_tmp.yaml
        WW_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_authoring-workstreams.yaml
        ${COPY_CMD} -rf "${WORKFLOW_PATTERN_FILE}" "${WORKFLOW_PATTERN_FILE_BAK}"
        ${COPY_CMD} -rf "${WW_PATTERN_FILE}" "${WW_PATTERN_FILE_BAK}"
    elif [[ "$DEPLOYMENT_TYPE" == "production" ]]
    then
        WORKFLOW_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow.yaml
        WORKFLOW_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_tmp.yaml
        WORKFLOW_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow.yaml

        WORKSTREAMS_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workstreams.yaml
        WORKSTREAMS_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workstreams_tmp.yaml
        WORKSTREAMS_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workstreams.yaml

        WORKFLOW_AUTHOR_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_authoring.yaml
        WORKFLOW_AUTHOR_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_authoring_tmp.yaml
        WORKFLOW_AUTHOR_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_authoring.yaml

        WFPS_AUTHOR_PATTERN_FILE=${PARENT_DIR}/descriptors/patterns/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_process_service_authoring.yaml
        WFPS_AUTHOR_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_process_service_authoring_tmp.yaml
        WFPS_AUTHOR_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_process_service_authoring.yaml

        # merge workflow with workstreams templat for workflow-workstreams in 4Q
        ${YQ_CMD} m -a -M ${WORKFLOW_PATTERN_FILE} ${WORKSTREAMS_PATTERN_FILE} > $TEMP_FOLDER/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_workstreams.yaml
        WW_PATTERN_FILE=$TEMP_FOLDER/ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_workstreams.yaml
        ${YQ_CMD} d -i ${WW_PATTERN_FILE} spec.initialize_configuration.ic_obj_store_creation.object_stores.[3]
        ${YQ_CMD} d -i ${WW_PATTERN_FILE} spec.datasource_configuration.dc_os_datasources.[3]
        ${YQ_CMD} d -i ${WW_PATTERN_FILE} spec.initialize_configuration.ic_ldap_creation.ic_ldap_admin_user_name.[1]
        ${YQ_CMD} d -i ${WW_PATTERN_FILE} spec.initialize_configuration.ic_ldap_creation.ic_ldap_admins_groups_name.[1]
        ${YQ_CMD} w -i ${WW_PATTERN_FILE} spec.baw_configuration.[0].host_federated_portal false
        ${YQ_CMD} w -i ${WW_PATTERN_FILE} spec.baw_configuration.[1].host_federated_portal false
        ${YQ_CMD} w -i ${WW_PATTERN_FILE} spec.baw_configuration.[0].host_federated_portal true
        WW_PATTERN_FILE_TMP=$TEMP_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_workstreams_tmp.yaml
        WW_PATTERN_FILE_BAK=$BAK_FOLDER/.ibm_cp4a_cr_${DEPLOY_TYPE_IN_FILE_NAME}_workflow_workstreams.yaml

        ${COPY_CMD} -rf "${WORKFLOW_PATTERN_FILE}" "${WORKFLOW_PATTERN_FILE_BAK}"
        ${COPY_CMD} -rf "${WORKSTREAMS_PATTERN_FILE}" "${WORKSTREAMS_PATTERN_FILE_BAK}"
        ${COPY_CMD} -rf "${WORKFLOW_AUTHOR_PATTERN_FILE}" "${WORKFLOW_AUTHOR_PATTERN_FILE_BAK}"
        ${COPY_CMD} -rf "${WW_PATTERN_FILE}" "${WW_PATTERN_FILE_BAK}"
        ${COPY_CMD} -rf "${WFPS_AUTHOR_PATTERN_FILE}" "${WFPS_AUTHOR_PATTERN_FILE_BAK}"
    fi
}


function startup_operator(){
    # scale up CP4BA operators
    local project_name=$1
    local run_mode=$2  # silent
    # Loop through the CP4BA_OPERATOR_LIST defined in common.sh and start up the operator from that list
    # If the operator is ibm-dpe-operator then only start up when architecure is amd64
    info "$CP4BA_OPERATOR_LIST"
    for operator in $CP4BA_OPERATOR_LIST; do
        if [[ "$operator" == "ibm-dpe-operator" ]]; then
            arch_type=$(kubectl get cm cluster-config-v1 -n kube-system -o yaml | grep -i architecture|tail -1| awk '{print $2}')
            if [[ "$arch_type" == "amd64" ]]; then
                info "Scaling up \"$operator\" operator"
                kubectl scale --replicas=1 deployment $operator -n $project_name >/dev/null 2>&1
                if [ $? -eq 0 ]; then
                    sleep 1
                    if [[ -z "$run_mode" ]]; then
                        echo "Done!"
                    fi
                else
                    fail "Failed to scale up \"$operator\" operator"
                fi
            fi
        else
            info "Scaling up \"$operator\" operator"
            kubectl scale --replicas=1 deployment $operator -n $project_name >/dev/null 2>&1
            if [ $? -eq 0 ]; then
                sleep 1
                if [[ -z "$run_mode" ]]; then
                    echo "Done!"
                fi
            else
                fail "Failed to scale up \"$operator\" operator"
            fi
        fi

    done
}

function shutdown_operator(){
    # scale down CP4BA operators base on the CP4BA_OPERATOR_LIST defined in common.sh
    local project_name=$1
    for operator in $CP4BA_OPERATOR_LIST; do
        
        if [[ $operator == "ibm-ads-operator" || $operator == "ibm-cp4a-wfps-operator" || $operator == "ibm-insights-engine-operator" ]]; then
            info "Scaling down \"$operator\" operator"
            kubectl scale --replicas=0 deployment $operator -n $project_name >/dev/null 2>&1
            kubectl scale --replicas=0 deployment ${operator}-controller-manager -n $project_name >/dev/null 2>&1
            sleep 1
            echo "Done!"

        else
            info "Scaling down \"$operator\" operator"
            kubectl scale --replicas=0 deployment $operator -n $project_name >/dev/null 2>&1
            sleep 1
            echo "Done!"
        fi
    done
}

function cncf_install(){
  sed -e '/dba_license/{n;s/value:.*/value: accept/;}' ${CUR_DIR}/../upgradeOperator.yaml > ${CUR_DIR}/../upgradeOperatorsav.yaml ;  mv ${CUR_DIR}/../upgradeOperatorsav.yaml ${CUR_DIR}/../upgradeOperator.yaml
  sed -e '/baw_license/{n;s/value:.*/value: accept/;}' ${CUR_DIR}/../upgradeOperator.yaml > ${CUR_DIR}/../upgradeOperatorsav.yaml ;  mv ${CUR_DIR}/../upgradeOperatorsav.yaml ${CUR_DIR}/../upgradeOperator.yaml
  sed -e '/fncm_license/{n;s/value:.*/value: accept/;}' ${CUR_DIR}/../upgradeOperator.yaml > ${CUR_DIR}/../upgradeOperatorsav.yaml ;  mv ${CUR_DIR}/../upgradeOperatorsav.yaml ${CUR_DIR}/../upgradeOperator.yaml
  sed -e '/ier_license/{n;s/value:.*/value: accept/;}' ${CUR_DIR}/../upgradeOperator.yaml > ${CUR_DIR}/../upgradeOperatorsav.yaml ;  mv ${CUR_DIR}/../upgradeOperatorsav.yaml ${CUR_DIR}/../upgradeOperator.yaml

  if [ ! -z ${IMAGEREGISTRY} ]; then
  # Change the location of the image
  echo "Using the operator image name: $IMAGEREGISTRY"
  sed -e "s|image: .*|image: \"$IMAGEREGISTRY\" |g" ${CUR_DIR}/../upgradeOperator.yaml > ${CUR_DIR}/../upgradeOperatorsav.yaml ;  mv ${CUR_DIR}/../upgradeOperatorsav.yaml ${CUR_DIR}/../upgradeOperator.yaml
  fi

  # Change the pullSecrets if needed
  if [ ! -z ${PULLSECRET} ]; then
      echo "Setting pullSecrets to $PULLSECRET"
      sed -e "s|ibm-entitlement-key|$PULLSECRET|g" ${CUR_DIR}/../upgradeOperator.yaml > ${CUR_DIR}/../upgradeOperatorsav.yaml ;  mv ${CUR_DIR}/../upgradeOperatorsav.yaml ${CUR_DIR}/../upgradeOperator.yaml
  else
      sed -e '/imagePullSecrets:/{N;d;}' ${CUR_DIR}/../upgradeOperator.yaml > ${CUR_DIR}/../upgradeOperatorsav.yaml ;  mv ${CUR_DIR}/../upgradeOperatorsav.yaml ${CUR_DIR}/../upgradeOperator.yaml
  fi
  kubectl apply -f ${CUR_DIR}/../descriptors/service_account.yaml --validate=false
  kubectl apply -f ${CUR_DIR}/../descriptors/role.yaml --validate=false
  kubectl apply -f ${CUR_DIR}/../descriptors/role_binding.yaml --validate=false
  kubectl apply -f ${CUR_DIR}/../upgradeOperator.yaml --validate=false
}

function patch_edb_configmap(){
# DBACLD-166239 -> Update EDB configmap ibm-zen-metastore-edb-cm to add new parameters with CPFS 4.10 or later.
# During upgrade, we check the existence of the configmap along with these 2 new parameters (DATABASE_ENABLE_SSL and DATABASE_SSL_MODE).
# If those parameters exists then we will not patch the configmap as the configmap is same for embedded and external postgres.
# Input:
#   namespace: The namespace that the configmap is located

local namespace=$1

is_edb_missing_ssl_enable_param=$(${CLI_CMD} get configmap ${ZEN_EDB_CFG} -n $namespace -o jsonpath='{.data.DATABASE_ENABLE_SSL}' 2>&1 )
is_edb_missing_ssl_mode_param=$(${CLI_CMD} get configmap ${ZEN_EDB_CFG} -n $namespace -o jsonpath='{.data.DATABASE_SSL_MODE}' 2>&1 )

#Patch cm if DATABASE_ENABLE_SSL does not exist
if [[  -z $is_edb_missing_ssl_enable_param ]]; then
    info "Patching ${ZEN_EDB_CFG} with DATABASE_ENABLE_SSL parameter"
    ${CLI_CMD} patch configmap ${ZEN_EDB_CFG} -n $namespace --type=merge -p '{"data": {"DATABASE_ENABLE_SSL":"'true'"}}'
fi
#Patch cm if DATABASE_SSL_MODE does not exist
if [[  -z $is_edb_missing_ssl_mode_param ]]; then
    info "Patching ${ZEN_EDB_CFG} with DATABASE_SSL_MODE parameter"
    ${CLI_CMD} patch configmap ${ZEN_EDB_CFG} -n $namespace --type=merge -p '{"data": {"DATABASE_SSL_MODE":"require"}}'

fi
}

# Function to patch the elastic search cluster CR with "quiesce":true which is required before upgrading to opensearch 2.19
# https://jsw.ibm.com/browse/DBACLD-166681
function patch_elasticsearch_cr(){
    local cr_namespace=$1
    elasticsearch_cr_name=$(${CLI_CMD} get ElasticsearchCluster -n $cr_namespace --no-headers --ignore-not-found | awk '{print $1}')
    if [[ -n $elasticsearch_cr_name ]]; then
        info "Patching ElasticsearchCluster $elasticsearch_cr_name in namespace $cr_namespace..."
        ${CLI_CMD} patch ElasticsearchCluster $elasticsearch_cr_name -n $cr_namespace --type=merge -p '{"spec": {"quiesce":true}}'
        printf "\n"
    else
        info " Manually patch the Elasticsearch Cluster by executing \" ${CLI_CMD} patch ElasticsearchCluster $elasticsearch_cr_name -n $cr_namespace --type=merge -p '{\"spec\": {\"quiesce\":true}}' \" "
        printf "\n"
    
    fi
}
#DBACLD-166863: This function determines the UPGRADE_MODE
function determine_upgrade_mode () {
############
# This function will set the UPGRADE_MODE.  The main logic will use the following 2 parameters:
# 1. The UPGRADE_MODE: This is the mode that will be set.
# 2. ALLOW_DIRECT_UPGRADE: This is an internal flag that will allow the user to do a direct upgrade from 21.0.3.x/22.0.x/23.0.2.x to 25.0.0.x or later. If it's 1 then we'll use the old orginal logic to determine the UPGRADE_MODE.
## If it set to 0 then the UPGRADE_MODE will be set to "shared2shared" (Allnamespace -> Allnamespace) or "dedicated2dedicated" if the customer is already on 24.0.1.x or later.
############
# This is the original way to check for skip version upgrade such as 21.0.3.x/22.0.x/23.0.2.x to 25.0.0.x or later which we don't officially support in 25.0.0.x unless the ALLOW_DIRECT_UPGRADE is set to 1.
    if [[ -z "$UPGRADE_MODE" && "$ALLOW_DIRECT_UPGRADE" == 1 ]]; then
        cs_dedicated=$(${CLI_CMD} get cm -n ${COMMON_SERVICES_CM_NAMESPACE}  | grep ${COMMON_SERVICES_CM_DEDICATED_NAME} | awk '{print $1}')

        cs_shared=$(${CLI_CMD} get cm -n ${COMMON_SERVICES_CM_NAMESPACE}  | grep ${COMMON_SERVICES_CM_SHARED_NAME} | awk '{print $1}')

        if [[ "$cs_dedicated" != "" || "$cs_shared" != ""  ]] ; then
            control_namespace=$(${CLI_CMD} get cm ${COMMON_SERVICES_CM_DEDICATED_NAME} --no-headers --ignore-not-found -n ${COMMON_SERVICES_CM_NAMESPACE} -o jsonpath='{ .data.common-service-maps\.yaml }' | grep  'controlNamespace' | cut -d':' -f2 )
            control_namespace=$(sed -e 's/^"//' -e 's/"$//' <<<"$control_namespace")
            control_namespace=$(sed "s/ //g" <<< $control_namespace)
        fi


        if [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
            info "IBM Cloud Pak foundational services is working in \"Cluster-scoped\"."
            UPGRADE_MODE="shared2shared"
            ENABLE_PRIVATE_CATALOG=0
            info "Found the IBM Cloud Pak foundational services/IBM Cloud Pak for Business Automation are installed into all namespaces, IBM Cloud Pak foundational services only can be migrated from \"Cluster-scoped\" to \"Cluster-scoped\"!"
        elif [[ "$cs_dedicated" != "" && "$cs_shared" == "" ]]; then
            info "IBM Cloud Pak foundational services is working in \"Namespace-scoped\"."
            UPGRADE_MODE="dedicated2dedicated"
        elif [[ "$cs_dedicated" == "" && "$cs_shared" != "" && $ALL_NAMESPACE_FLAG == "No" ]]; then
            info "IBM Cloud Pak foundational services is working in \"Cluster-scoped\"."
            # select_upgrade_mode
            UPGRADE_MODE="shared2dedicated"
            info "IBM Cloud Pak foundational services will always migrate from \"Cluster-scoped\" to \"Namespace-scoped\"."
            prompt_press_any_key_to_continue
        elif [[ "$cs_dedicated" != "" && "$cs_shared" != "" && "$control_namespace" == "" ]]; then
            info "IBM Cloud Pak foundational services is working in \"Cluster-scoped\"."
            # select_upgrade_mode
            UPGRADE_MODE="shared2dedicated"
            info "IBM Cloud Pak foundational services will always migrate from \"Cluster-scoped\" to \"Namespace-scoped\"."
            prompt_press_any_key_to_continue
        elif [[ "$cs_dedicated" != "" && "$cs_shared" != "" && "$control_namespace" != "" ]]; then
            ${CLI_CMD} get cm ${COMMON_SERVICES_CM_DEDICATED_NAME} --no-headers --ignore-not-found -n ${COMMON_SERVICES_CM_NAMESPACE} -o jsonpath='{ .data.common-service-maps\.yaml }' > /tmp/common-service-maps.yaml
            index=0
            common_service_namespace=`cat /tmp/common-service-maps.yaml | ${YQ_CMD} r - namespaceMapping.[$index].map-to-common-service-namespace`
            while [[ ! -z $common_service_namespace ]]
            do
                if [[ $common_service_namespace == "ibm-common-services" ]]; then
                    # listing all the requested namespaces that are in shared mode
                    # If the CP4BA_SERVICES_NS is in this list under ibm-common-services then it is shared otherwise it means CP4BA_SERVICES_NS is dedicated but there are other deployments in shared mode. Just checking if ibm-common-services is listed in namespaceMapping.map-to-common-service-namespace is not sufficient.
                    # For https://jsw.ibm.com/browse/DBACLD-168119
                    common_service_requested_namespace=`cat /tmp/common-service-maps.yaml | ${YQ_CMD} r - namespaceMapping.[$index].requested-from-namespace`
                    if echo "$common_service_requested_namespace" | grep -q "$CP4BA_SERVICES_NS"; then
                        info "IBM Cloud Pak foundational services is working in \"Cluster-scoped\"."
                        # select_upgrade_mode
                        UPGRADE_MODE="shared2dedicated"
                        info "IBM Cloud Pak foundational services will always migrate from \"Cluster-scoped\" to \"Namespace-scoped\"."
                        prompt_press_any_key_to_continue
                        break
                    else
                        info "IBM Cloud Pak foundational services is working in \"Namespace-scoped\"."
                        UPGRADE_MODE="dedicated2dedicated"
                        prompt_press_any_key_to_continue
                        break
                    fi
                fi
                ((index++))
                common_service_namespace=`cat /tmp/common-service-maps.yaml | ${YQ_CMD} r - namespaceMapping.[$index].map-to-common-service-namespace`
                if [[ -z $common_service_namespace ]]; then
                    info "IBM Cloud Pak foundational services is working in \"Namespace-scoped\"."
                    UPGRADE_MODE="dedicated2dedicated"
                fi
            done
        elif [[ $ALL_NAMESPACE_FLAG == "No" ]]; then
            info "IBM Cloud Pak foundational services is working in \"Cluster-scoped\"."
            # select_upgrade_mode
            UPGRADE_MODE="shared2dedicated"
            info "IBM Cloud Pak foundational services will always migrate from \"Cluster-scoped\" to \"Namespace-scoped\"."
            prompt_press_any_key_to_continue
        fi
    fi
# This scenario is the official support scenario for upgrading to 25.0.0.x or later.
# "all namespaces" ==> "all namespaces"
# "dedicated" ==> "dedicated"
    if [[ -z "$UPGRADE_MODE" && "$ALLOW_DIRECT_UPGRADE" != 1 ]]; then
        if [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
            info "IBM Cloud Pak foundational services is working in \"Cluster-scoped\"."
            UPGRADE_MODE="shared2shared"
            ENABLE_PRIVATE_CATALOG=0
            info "Found the IBM Cloud Pak foundational services/IBM Cloud Pak for Business Automation are installed into all namespaces, IBM Cloud Pak foundational services only can be migrated from \"Cluster-scoped\" to \"Cluster-scoped\"!"
        else
            info "IBM Cloud Pak foundational services is working in \"Namespace-scoped\"."
            UPGRADE_MODE="dedicated2dedicated"
        fi 
    fi

}

################################################
#### Begin - Main step for install operator ####
################################################
save_log "cp4a-script-logs/project/$TARGET_PROJECT_NAME" "cp4a-deployment-log"
trap cleanup_log EXIT

# Import upgrade upgrade_check_version.sh script
source ${CUR_DIR}/helper/upgrade/upgrade_check_status.sh

if [[ ($SCRIPT_MODE == "" && $RUNTIME_MODE == "") || ($SCRIPT_MODE == "dev" && $RUNTIME_MODE == "") || ($SCRIPT_MODE == "review" && $RUNTIME_MODE == "") || ($SCRIPT_MODE == "baw-dev" && $RUNTIME_MODE == "") ]]
then
    prompt_license

    set_script_mode

    input_information

    # Check whether the CP4BA is separation of operators and operands.
    check_cp4ba_separate_operand $TARGET_PROJECT_NAME

    show_summary

    while true; do

        printf "\n"
        printf "\x1B[1mVerify that the information above is correct.\n\x1B[0m"
        printf "\x1B[1mTo proceed with the deployment, enter \"Yes\".\n\x1B[0m"
        printf "\x1B[1mTo make changes, enter \"No\" (default: No): \x1B[0m"
        if [[ -z ${CP4BA_SKIP_SUMMARY} ]]; then
          read -rp "" ans
        else
            ans=${CP4BA_SKIP_SUMMARY}
        fi
        case "$ans" in
        "y"|"Y"|"yes"|"Yes"|"YES")
            if [[ ("$SCRIPT_MODE" != "review") && ("$SCRIPT_MODE" != "OLM") ]]; then
                if [[ $DEPLOYMENT_TYPE == "production" ]];then
                    printf "\n"
                    echo -e "\x1B[1mCreating the Custom Resource of the Cloud Pak for Business Automation operator...\x1B[0m"
                fi
            fi
            printf "\n"
            if [[ "${INSTALLATION_TYPE}"  == "new" ]]; then
                if [[ "$SCRIPT_MODE" == "review" ]]; then
                    echo -e "\x1B[1mReview mode is running. The final CR will be generated, but the operator will not be deployed.\x1B[0m"
                    # prompt_press_any_key_to_continue
                elif [[ "$SCRIPT_MODE" == "OLM" ]]
                then
                    echo -e "\x1B[1mA custom resource file for applying to the OCP Catalog is being generated.\x1B[0m"
                    # prompt_press_any_key_to_continue
                else
                    if [ "$use_entitlement" = "no" ] ; then
                        isReady=$(${CLI_CMD} get secret | grep ibm-entitlement-key)
                        if [[ -z $isReady ]]; then
                            echo "Secret \"ibm-entitlement-key\" not found, exiting..."
                            exit 1
                        else
                            echo "Secret \"ibm-entitlement-key\" found, continuing..."
                        fi
                    fi
                fi
            fi
            apply_pattern_cr
            break
            ;;
        "n"|"N"|"no"|"No"|"NO"|*)
            while true; do
                printf "\n"
                show_summary
                printf "\n"

                if  [[ $PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS" ]];
                then
                    printf "\x1B[1mEnter the number from 1 to 5 that you want to change: \x1B[0m"
                else
                    printf "\x1B[1mEnter the number from 1 to 7 that you want to change: \x1B[0m"
                fi

                read -rp "" ans
                if  [[ $PLATFORM_SELECTED == "OCP" || $PLATFORM_SELECTED == "ROKS" ]];
                then
                    case "$ans" in
                    "1")
                        if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
                            select_pattern
                            select_optional_component
                            if [[ ( -z "$CP4BA_JDBC_URL" ) && (" ${optional_component_cr_arr[@]} " =~ "iccsap") ]]; then
                                get_jdbc_url
                            fi
                        else
                            info "Run cp4a-prerequisites.sh to modify CP4BA capability"
                            prompt_press_any_key_to_continue
                        fi
                        break
                        ;;
                    "2")
                        if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
                            select_optional_component
                            if [[ ( -z "$CP4BA_JDBC_URL" ) && (" ${optional_component_cr_arr[@]} " =~ "iccsap") ]]; then
                                get_jdbc_url
                            fi
                        else
                            info "Run cp4a-prerequisites.sh to modify optional components"
                            prompt_press_any_key_to_continue
                        fi
                        break
                        ;;
                    "3"|"4")
                        if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
                            get_storage_class_name
                        else
                            info "Run cp4a-prerequisites.sh to modify storage class name"
                            prompt_press_any_key_to_continue
                        fi
                        break
                        ;;
                    "5")
                        get_jdbc_url
                        break
                        ;;
                    *)
                        echo -e "\x1B[1mEnter a valid number [1 to 5] \x1B[0m"
                        ;;
                    esac
                else
                    case "$ans" in
                    "1")
                        if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
                            select_pattern
                            select_optional_component
                            if [[ ( -z "$CP4BA_JDBC_URL" ) && (" ${optional_component_cr_arr[@]} " =~ "iccsap") ]]; then
                                get_jdbc_url
                            fi
                        else
                            info "Run cp4a-prerequisites.sh to modify CP4BA pattern"
                            prompt_press_any_key_to_continue
                        fi
                        break
                        ;;
                    "2")
                        if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
                            select_optional_component
                            if [[ ( -z "$CP4BA_JDBC_URL" ) && (" ${optional_component_cr_arr[@]} " =~ "iccsap") ]]; then
                                get_jdbc_url
                            fi
                        else
                            info "Run cp4a-prerequisites.sh to modify optional components"
                            prompt_press_any_key_to_continue
                        fi
                        break
                        ;;
                    "3")
                        get_entitlement_registry
                        break
                        ;;
                    "4")
                        get_local_registry_server
                        break
                        ;;
                    "5")
                        get_local_registry_user
                        break
                        ;;
                    "6")
                        get_local_registry_password
                        break
                        ;;
                    "7")
                        if [[ $DEPLOYMENT_TYPE == "starter" || $DEPLOYMENT_WITH_PROPERTY == "No" ]]; then
                            get_storage_class_name
                        else
                            info "Run cp4a-prerequisites.sh to modify storage class name"
                            prompt_press_any_key_to_continue
                        fi
                        break
                        ;;
                    "8")
                        get_jdbc_url
                        break
                        ;;
                    *)
                        echo -e "\x1B[1mEnter a valid number [1 to 8] \x1B[0m"
                        ;;
                    esac
                fi
            done
            show_summary
            ;;
        esac
    done
else
    ENABLE_PRIVATE_CATALOG=0
    # parse_arguments "$@"
    # if [[ -z "$RUNTIME_MODE" ]]; then
    #     echo -e "\x1B[1;31mPlease input value for \"-m <MODE_NAME>\" option.\n\x1B[0m"
    #     exit 1
    # fi
    # if [[ -z "$TARGET_PROJECT_NAME" ]]; then
    #     echo -e "\x1B[1;31mPlease input value for \"-n <NAME_SPACE>\" option.\n\x1B[0m"
    #     exit 1
    # fi
fi
############## Start - Migration CPfs mode and upgrade CP4BA Operators ##############

if [ "$RUNTIME_MODE" == "upgradeOperator" ]; then
    upgrade_operator_project_name=$TARGET_PROJECT_NAME

    # check current cp4ba version
    check_cp4ba_operator_version $TARGET_PROJECT_NAME $ALLOW_DIRECT_UPGRADE

    ################## Start - Detect all namespace or not ##################
    cp4a_operator_csv_name_target_ns=$(${CLI_CMD} get csv -n $TARGET_PROJECT_NAME --no-headers --ignore-not-found | grep "IBM Cloud Pak for Business Automation" | awk '{print $1}')
    cp4a_operator_csv_name_allnamespace_ns=$(${CLI_CMD} get csv -n $ALL_NAMESPACE_NAME --no-headers --ignore-not-found | grep "IBM Cloud Pak for Business Automation" | awk '{print $1}')

    if [[ -z $cp4a_operator_csv_name_allnamespace_ns && (! -z $cp4a_operator_csv_name_target_ns) ]]; then
        success "The IBM Cloud Pak for Business Automation Operator was found deployed in the project \"$TARGET_PROJECT_NAME\"."
        ALL_NAMESPACE_FLAG="No"
        TEMP_OPERATOR_PROJECT_NAME=$TARGET_PROJECT_NAME
    elif [[ (! -z $cp4a_operator_csv_name_allnamespace_ns) && (! -z $cp4a_operator_csv_name_target_ns) ]]; then
        success "The IBM Cloud Pak for Business Automation Operator was found deployed in AllNamespace mode in the project \"$ALL_NAMESPACE_NAME\"."
        ALL_NAMESPACE_FLAG="Yes"
        upgrade_operator_project_name="openshift-operators"
        TEMP_OPERATOR_PROJECT_NAME="openshift-operators"
    fi
    ################## End - Detect all namespace or not ##################

    # Check whether the CP4BA is separation of operators and operands, namely seperation of duty.
#DBACLD-175890: There is no need to for CSV version in 25.0.0.x or later.
    if [[ $ALL_NAMESPACE_FLAG != "Yes" ]]; then
        check_cp4ba_separate_operand $TARGET_PROJECT_NAME
    else
        CP4BA_SERVICES_NS=$TARGET_PROJECT_NAME
    fi

    ######## Start - Define variables cp4ba-upgrade for supporting multiple CP4BA instances ########
    UPGRADE_DEPLOYMENT_FOLDER=${CUR_DIR}/cp4ba-upgrade/project/$CP4BA_SERVICES_NS
    UPGRADE_DEPLOYMENT_PROPERTY_FILE=${UPGRADE_DEPLOYMENT_FOLDER}/cp4ba_upgrade.property

    UPGRADE_DEPLOYMENT_CR=${UPGRADE_DEPLOYMENT_FOLDER}/custom_resource
    UPGRADE_DEPLOYMENT_CR_BAK=${UPGRADE_DEPLOYMENT_CR}/backup
    UPGRADE_DEPLOYMENT_CONTENT_CR=${UPGRADE_DEPLOYMENT_CR}/content.yaml
    UPGRADE_DEPLOYMENT_CONTENT_CR_TMP=${UPGRADE_DEPLOYMENT_CR}/.content_tmp.yaml
    UPGRADE_DEPLOYMENT_CONTENT_CR_BAK=${UPGRADE_DEPLOYMENT_CR_BAK}/content_cr_backup.yaml

    UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR=${UPGRADE_DEPLOYMENT_CR}/icp4acluster.yaml
    UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP=${UPGRADE_DEPLOYMENT_CR}/.icp4acluster_tmp.yaml
    UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_BAK=${UPGRADE_DEPLOYMENT_CR_BAK}/icp4acluster_cr_backup.yaml

    UPGRADE_DEPLOYMENT_BAI_TMP=${UPGRADE_DEPLOYMENT_CR}/.bai_tmp.yaml
    mkdir -p ${UPGRADE_DEPLOYMENT_CR} >/dev/null 2>&1
    mkdir -p ${TEMP_FOLDER} >/dev/null 2>&1
    ######## End - Define variables cp4ba-upgrade variables for supporting multiple CP4BA instances ########

    ################## Start - Apply third-party WORKAROUND ####################
    # As workaround for https://github.ibm.com/IBMPrivateCloud/roadmap/issues/64017
    # require to remove ibm-operator-catalog first, and then to add back after upgrading.
    # This check has been commented out since 24.0.0 IF001 as upgrade can now be completed with ibm-operator-catalog
    #info "Checking \"ibm-operator-catalog\" exist or not in project \"openshift-marketplace\""
    #printf "\n"
    #if ${CLI_CMD} get catalogsource -n openshift-marketplace --no-headers --ignore-not-found | grep ibm-operator-catalog >/dev/null 2>&1; then
    #    echo "${YELLOW_TEXT}[ATTENTION]: ${RESET_TEXT}${RED_TEXT}Detected \"ibm-operator-catalog\" catalog source in project \"openshift-marketplace\".  This configuration is not yet supported for upgrade and will be supported in a future interim fix.  The upgrade will terminate.${RESET_TEXT}"
    #    exit 1
    #fi

    # As workaround for https://github.ibm.com/IBMPrivateCloud/roadmap/issues/64207
    # update secret postgresql-operator-controller-manager-config in <cp4ba> namespace and/or ibm-common-services namespace and add this annotation ibm-bts/skip-updates: "true"
    if ${CLI_CMD} get secret -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | grep postgresql-operator-controller-manager-config >/dev/null 2>&1; then
        ${CLI_CMD} patch secret postgresql-operator-controller-manager-config -n $CP4BA_SERVICES_NS -p '{"metadata": {"annotations": {"ibm-bts/skip-updates": "true"}}}' >/dev/null 2>&1
    fi
    if ${CLI_CMD} get secret -n ibm-common-services --no-headers --ignore-not-found | grep postgresql-operator-controller-manager-config >/dev/null 2>&1; then
        ${CLI_CMD} patch secret postgresql-operator-controller-manager-config -n ibm-common-services -p '{"metadata": {"annotations": {"ibm-bts/skip-updates": "true"}}}' >/dev/null 2>&1
    fi
    ################## End - Apply third-party WORKAROUND ####################

    ################## Start of workaround for https://jsw.ibm.com/browse/DBACLD-167061  
    # We're setting spec.enableSuperuserAccess to true for our postgres-cp4ba edb instance
    info "Determinig if EnterpriseDB PostgreSQL \"$EDB_INSTANCE_CP4BA_NAME\" is installed for IBM Cloud Pak for Business Automation."
    edb_instance_cp4ba_cr=$( ${CLI_CMD} get cluster.postgresql.k8s.enterprisedb.io -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found $EDB_INSTANCE_CP4BA_NAME | awk '{print $1}' )
    if [[ $edb_instance_cp4ba_cr == $EDB_INSTANCE_CP4BA_NAME  ]]; then
      info "Found EnterpriseDB PostgreSQL instance \"$EDB_INSTANCE_CP4BA_NAME\""

      info "In certain deployment scenarios, such as if IBM Automation Document Processing deployed"
      info "Patching the EnterpriseDB PostgreSQL \"$EDB_INSTANCE_CP4BA_NAME\" with '{\"spec\": {\"enableSuperuserAccess\":true}}' is required"

      enable_superuser_access=$( ${CLI_CMD} get cluster.postgresql.k8s.enterprisedb.io -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found $EDB_INSTANCE_CP4BA_NAME -o jsonpath='{.spec.enableSuperuserAccess}' )
      if [[ ${enable_superuser_access,,} != "true" ]]; then
        ${CLI_CMD} patch cluster.postgresql.k8s.enterprisedb.io  $EDB_INSTANCE_CP4BA_NAME -n $CP4BA_SERVICES_NS --type=merge -p '{"spec": {"enableSuperuserAccess":true}}'  >/dev/null 2>&1
      else
        info "The EnterpriseDB PostgreSQL instance \"$EDB_INSTANCE_CP4BA_NAME\" already as the field \"enableSuperuserAccess\" set to true."
      fi
    else 
      info "Unable to find EnterpriseDB PostgreSQL instance \"$EDB_INSTANCE_CP4BA_NAME\""
      info "If the EnterpriseDB PostgreSQL instance \"$EDB_INSTANCE_CP4BA_NAME\" exists, execute the command: "
      info "${CLI_CMD} patch cluster.postgresql.k8s.enterprisedb.io  $EDB_INSTANCE_CP4BA_NAME -n $CP4BA_SERVICES_NS --type=merge -p '{\"spec\": {\"enableSuperuserAccess\":true}}'"
    fi
    ################## End of setting spec.enableSuperuserAccess to true for our postgres-cp4ba edb instance


    ############## Start - Prepare definition for ibm-cp4ba-shared-info/ibm-cp4ba-content-shared-info/ibm-cp4ba-common-config configMap ##############
    if [[ $SEPARATE_OPERAND_FLAG == "Yes" ]]; then
        source ${CUR_DIR}/helper/upgrade/upgrade_merge_yaml.sh $CP4BA_SERVICES_NS $ALLOW_DIRECT_UPGRADE
    else
        source ${CUR_DIR}/helper/upgrade/upgrade_merge_yaml.sh $TARGET_PROJECT_NAME $ALLOW_DIRECT_UPGRADE
    fi
    ############## End - Prepare definition for ibm-cp4ba-shared-info/ibm-cp4ba-content-shared-info/ibm-cp4ba-common-config configMap ##############


    ############## Start - Setup opensearch and show dynamic tips for migration es to os ##############
    # Confirm finish migration from Elasticsearch to Opensearch.
    info "Checking legacy Elasticsearch is installed for this CP4BA deployment or not."

    ${CLI_CMD} get crd |grep contents.icp4a.ibm.com >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        content_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        if [ ! -z $content_cr_name ]; then
            cr_type="content"
            cp4ba_cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
            owner_ref=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)
            if [[ ${owner_ref} == "ICP4ACluster" ]]; then
                echo
            else
                ${CLI_CMD} get $cr_type $content_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_CONTENT_CR_TMP}
                bai_flag=`cat $UPGRADE_DEPLOYMENT_CONTENT_CR_TMP | ${YQ_CMD} r - spec.content_optional_components.bai`
                bai_flag=$(echo $bai_flag | tr '[:upper:]' '[:lower:]')
                CONTENT_CR_EXIST="Yes"
                cp4ba_root_ca_secret_name=`cat $UPGRADE_DEPLOYMENT_CONTENT_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.root_ca_secret`
            fi
        fi
    fi

    icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
    if [ ! -z $icp4acluster_cr_name ]; then
        cr_type="icp4acluster"
        cp4ba_cr_metaname=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
        ${CLI_CMD} get $cr_type $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}

        cp4ba_root_ca_secret_name=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.root_ca_secret`
        convert_olm_cr "${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}"
        if [[ $olm_cr_flag == "No" ]]; then
            # Get EXISTING_PATTERN_ARR/EXISTING_OPT_COMPONENT_ARR
            existing_pattern_list=""
            existing_opt_component_list=""
            EXISTING_PATTERN_ARR=()
            EXISTING_OPT_COMPONENT_ARR=()
            existing_pattern_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_patterns`
            existing_opt_component_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_optional_components`

            OIFS=$IFS
            IFS=',' read -r -a EXISTING_PATTERN_ARR <<< "$existing_pattern_list"
            IFS=',' read -r -a EXISTING_OPT_COMPONENT_ARR <<< "$existing_opt_component_list"
            IFS=$OIFS
        fi
    fi
    if [[ -z $content_cr_name && -z $icp4acluster_cr_name ]]; then
        fail "No CP4BA custom resource found in the cluster in the project \"$CP4BA_SERVICES_NS\"."
        exit 1
    fi

    ${CLI_CMD} get crd |grep elasticsearches.elastic.automation.ibm.com >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        old_elastic_cr_name=$(${CLI_CMD} get Elasticsearch.elastic.automation.ibm.com -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
    fi
    # if [[ (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "bai") || $bai_flag == "True" || $bai_flag == "true" ]]; then
    if [[  ! -z $old_elastic_cr_name  ]]; then
        info "Found the legacy Elasticsearch deployment in the project \"$CP4BA_SERVICES_NS\"."
        if [[ (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "bai") || $bai_flag == "true" ]]; then
        # test_flag="true"
        # if [[ $test_flag == "false" ]]; then
            info "Found the BAI component selected for this CP4BA deployment in the project \"$CP4BA_SERVICES_NS\"."
            es_to_os_migration_flag=$(${CLI_CMD} get configmap ibm-cp4ba-os-migration-status --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath={.data.opensearch_migration_done})
            es_to_os_migration_flag=$(echo $es_to_os_migration_flag | tr '[:upper:]' '[:lower:]')
            if [[ -z $es_to_os_migration_flag || $es_to_os_migration_flag == "no" ]]; then
                check_es_to_os_migration
            elif [[ $es_to_os_migration_flag == "yes" ]]; then
                # the script still need check ElasticsearchCluster cr even opensearch_migration_done is yes
                ${CLI_CMD} get crd |grep elasticsearch.opencontent.ibm.com >/dev/null 2>&1
                if [ $? -eq 0 ]; then
                    opensearch_cr_name=$(${CLI_CMD} get ElasticsearchCluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
                    if [ -z $opensearch_cr_name ]; then
                        warning "Opensearch is not installed (ElasticsearchCluster custom resource could not be found in project \"$CP4BA_SERVICES_NS\")."
                        check_es_to_os_migration
                    else
                        success "Found that \"opensearch_migration_done\" is set to \"Yes\" in the ibm-cp4ba-os-migration-status configMap in the project \"$CP4BA_SERVICES_NS\""
                        sleep 5
                    fi
                else
                    warning "ElasticsearchCluster CRD not found. You need to install Opensearch cluster before migration from Elasticserch to Opensearch."
                    check_es_to_os_migration
                fi
            fi
        else
            info "The BAI component selected for this CP4BA deployment was not found in the project \"$CP4BA_SERVICES_NS\"."
            check_selection_migration
            if [[ $ES_TO_OS_MIGRATION_SELECTED == "Yes" ]]; then
                es_to_os_migration_flag=$(${CLI_CMD} get configmap ibm-cp4ba-os-migration-status --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath={.data.opensearch_migration_done})
                es_to_os_migration_flag=$(echo $es_to_os_migration_flag | tr '[:upper:]' '[:lower:]')
                if [[ -z $es_to_os_migration_flag || $es_to_os_migration_flag == "no" ]]; then
                    check_es_to_os_migration
                elif [[ $es_to_os_migration_flag == "yes" ]]; then
                    # the script still need check ElasticsearchCluster cr even opensearch_migration_done is yes
                    ${CLI_CMD} get crd |grep elasticsearch.opencontent.ibm.com >/dev/null 2>&1
                    if [ $? -eq 0 ]; then
                        opensearch_cr_name=$(${CLI_CMD} get ElasticsearchCluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
                        if [ -z $opensearch_cr_name ]; then
                            warning "Opensearch is not installed (ElasticsearchCluster custom resource could not be found in project \"$CP4BA_SERVICES_NS\")."
                            check_es_to_os_migration
                        else
                            success "Found that \"opensearch_migration_done\" is set to \"Yes\" in the ibm-cp4ba-os-migration-status configMap in the project \"$CP4BA_SERVICES_NS\""
                            sleep 5
                        fi
                    else
                        warning "ElasticsearchCluster CRD not found. You need to install Opensearch cluster before migration from Elasticserch to Opensearch."
                        check_es_to_os_migration
                    fi
                fi
            elif [[ $ES_TO_OS_MIGRATION_SELECTED == "No" ]]; then
                MIGRATE_ES_TO_OS_DONE="NONEED"
                create_configmap_os_migration
            fi
        fi
    fi
    ############## End - Setup opensearch and show dynamic tips for migration es to os ##############

    # Updating the elastic search CR to switch the quiesce flag from false to true
    # https://jsw.ibm.com/browse/DBACLD-166681
    if [[ (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "bai") || (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "pfs") || $bai_flag == "true" ]]; then
        patch_elasticsearch_cr "$CP4BA_SERVICES_NS"
    fi

    info "Starting to upgrade CP4BA operators and IBM Cloud Pak foundational services"

    ############## Start - Check CP4BA operator is already upgrade and how to rerun upgradeOperator ##############
    if [[ "$cp4a_operator_csv_version" == "${CP4BA_CSV_VERSION//v/}" ]]; then
        warning "The ClusterServiceVersion (CSV) of CP4BA operator already is $CP4BA_CSV_VERSION."
        printf "\n"
        source ${CUR_DIR}/helper/messages.sh
        while true; do
            printf "\n"
            printf "\x1B[1mDo you want to continue to do upgrade? (Yes/No, default: No): \x1B[0m"
            read -rp "" ans
            case "$ans" in
            "y"|"Y"|"yes"|"Yes"|"YES")
                displayUpgradeOperatorMessage '' $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                exit 1
                ;;
            "n"|"N"|"no"|"No"|"NO"|"")
                echo "Exiting..."
                exit 1
                ;;
            *)
                echo -e "Answer must be \"Yes\" or \"No\"\n"
                ;;
            esac
        done
    fi
    ############## End - Check CP4BA operator is already upgrade and how to rerun upgradeOperator ##############

    ############## Start - Create bm-cp4ba-shared-info/ibm-cp4ba-content-shared-info configMap ##############
    info "Checking if the ibm-cp4ba-shared-info/ibm-cp4ba-content-shared-info configMap exists in the project \"$CP4BA_SERVICES_NS\""
    ibm_cp4ba_shared_info_cm=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.data.cp4ba_operator_of_last_reconcile}')
    ibm_cp4ba_content_shared_info_cm=$(${CLI_CMD} get configmap ibm-cp4ba-content-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.data.content_operator_of_last_reconcile}')
    # flag to track which shared-info configmap to use. By default this flag will be true, it will turn false only if it is a content based CR 
    cp4ba_shared_info_used=true 

    # Create ibm-cp4ba-shared-info configMap if not exist
    icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
    if [ ! -z $icp4acluster_cr_name ]; then
        cr_version=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.appVersion)
        cr_metaname=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
        cr_uid=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.uid)
        if [[ -z $ibm_cp4ba_shared_info_cm ]]; then
            info "Not found ibm-cp4ba-shared-info configMap,creating it."
            create_ibm_cp4ba_shared_info_cm_yaml
            ${SED_COMMAND} "s|<cp4a_namespace>|$CP4BA_SERVICES_NS|g" ${UPGRADE_ICP4A_SHARED_INFO_CM_FILE}
            ${SED_COMMAND} "s|<cr_metaname>|$cr_metaname|g" ${UPGRADE_ICP4A_SHARED_INFO_CM_FILE}
            ${SED_COMMAND} "s|<cr_uid>|$cr_uid|g" ${UPGRADE_ICP4A_SHARED_INFO_CM_FILE}
            ${SED_COMMAND} "s|<csv_version>|$cp4a_operator_csv_version|g" ${UPGRADE_ICP4A_SHARED_INFO_CM_FILE}
            ${SED_COMMAND} "s|<cr_version>|$cr_version|g" ${UPGRADE_ICP4A_SHARED_INFO_CM_FILE}

            ${CLI_CMD} apply -f $UPGRADE_ICP4A_SHARED_INFO_CM_FILE  >/dev/null 2>&1
            if [ $? -eq 0 ]; then
                success "Created ibm-cp4ba-shared-info configMap in the project \"$CP4BA_SERVICES_NS\"!"
                ${CLI_CMD} patch configmap ibm-cp4ba-shared-info -n $CP4BA_SERVICES_NS --type=json -p="[{'op': 'add', 'path': '/data/cp4ba_original_csv_ver_for_upgrade_script', 'value': '$(echo $cp4a_operator_csv_version)'}]" >/dev/null 2>&1
                ${CLI_CMD} patch configmap ibm-cp4ba-shared-info -n $CP4BA_SERVICES_NS --type=json -p="[{'op': 'add', 'path': '/data/cpfs_original_csv_ver_for_upgrade_script', 'value': '$(echo $cpfs_operator_csv_version)'}]" >/dev/null 2>&1
                cp4ba_original_csv_ver_for_upgrade_script=$cp4a_operator_csv_version
            else
                fail "Failed to create ibm-cp4ba-shared-info configMap in the project \"$CP4BA_SERVICES_NS\"!"
            fi
        else
            success "Found ibm-cp4ba-shared-info configMap under \"$CP4BA_SERVICES_NS\"!"
            ${CLI_CMD} patch configmap ibm-cp4ba-shared-info -n $CP4BA_SERVICES_NS --type=json -p="[{'op': 'add', 'path': '/data/cp4ba_original_csv_ver_for_upgrade_script', 'value': '$(echo $cp4a_operator_csv_version)'}]" >/dev/null 2>&1
            ${CLI_CMD} patch configmap ibm-cp4ba-shared-info -n $CP4BA_SERVICES_NS --type=json -p="[{'op': 'add', 'path': '/data/cpfs_original_csv_ver_for_upgrade_script', 'value': '$(echo $cpfs_operator_csv_version)'}]" >/dev/null 2>&1
            cp4ba_original_csv_ver_for_upgrade_script=$cp4a_operator_csv_version
        fi
    fi

    # Create ibm-cp4ba-content-shared-info configMap if not exist
    ${CLI_CMD} get crd |grep contents.icp4a.ibm.com >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        content_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        if [ ! -z $content_cr_name ]; then
            cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
            owner_ref=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)
            if [[ ${owner_ref} != "ICP4ACluster" ]]; then
                # Setting this variable to false so that we know the cp4ba-content-shared-info is to be used
                cp4ba_shared_info_used=false 
                cr_version=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.appVersion)
                cr_uid=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.uid)
                if [[ -z $ibm_cp4ba_content_shared_info_cm ]]; then
                    info "ibm-cp4ba-content-shared-info configMap not found, creating it now."
                    create_ibm_cp4ba_content_shared_info_cm_yaml
                    ${SED_COMMAND} "s|<content_namespace>|$CP4BA_SERVICES_NS|g" ${UPGRADE_ICP4A_CONTENT_SHARED_INFO_CM_FILE}
                    ${SED_COMMAND} "s|<cr_metaname>|$cr_metaname|g" ${UPGRADE_ICP4A_CONTENT_SHARED_INFO_CM_FILE}
                    ${SED_COMMAND} "s|<cr_uid>|$cr_uid|g" ${UPGRADE_ICP4A_CONTENT_SHARED_INFO_CM_FILE}
                    ${SED_COMMAND} "s|<csv_version>|${cp4a_operator_csv_version}|g" ${UPGRADE_ICP4A_CONTENT_SHARED_INFO_CM_FILE}
                    ${SED_COMMAND} "s|<cr_version>|${cr_version}|g" ${UPGRADE_ICP4A_CONTENT_SHARED_INFO_CM_FILE}

                    ${CLI_CMD} apply -f $UPGRADE_ICP4A_CONTENT_SHARED_INFO_CM_FILE  >/dev/null 2>&1
                    if [ $? -eq 0 ]; then
                        success "Created ibm-cp4ba-content-shared-info configMap in the project \"$CP4BA_SERVICES_NS\"!"
                        ${CLI_CMD} patch configmap ibm-cp4ba-content-shared-info -n $CP4BA_SERVICES_NS --type=json -p="[{'op': 'add', 'path': '/data/cp4ba_original_csv_ver_for_upgrade_script', 'value': '$(echo $cp4a_operator_csv_version)'}]" >/dev/null 2>&1
                        ${CLI_CMD} patch configmap ibm-cp4ba-content-shared-info -n $CP4BA_SERVICES_NS --type=json -p="[{'op': 'add', 'path': '/data/cpfs_original_csv_ver_for_upgrade_script', 'value': '$(echo $cpfs_operator_csv_version)'}]" >/dev/null 2>&1
                        cp4ba_original_csv_ver_for_upgrade_script=$cp4a_operator_csv_version
                    else
                        fail "Failed to create ibm-cp4ba-content-shared-info configMap in the project \"$CP4BA_SERVICES_NS\"!"
                    fi
                else
                    success "Found ibm-cp4ba-content-shared-info configMap under \"$CP4BA_SERVICES_NS\"!"
                    ${CLI_CMD} patch configmap ibm-cp4ba-content-shared-info -n $CP4BA_SERVICES_NS --type=json -p="[{'op': 'add', 'path': '/data/cp4ba_original_csv_ver_for_upgrade_script', 'value': '$(echo $cp4a_operator_csv_version)'}]" >/dev/null 2>&1
                    ${CLI_CMD} patch configmap ibm-cp4ba-content-shared-info -n $CP4BA_SERVICES_NS --type=json -p="[{'op': 'add', 'path': '/data/cpfs_original_csv_ver_for_upgrade_script', 'value': '$(echo $cpfs_operator_csv_version)'}]" >/dev/null 2>&1
                    cp4ba_original_csv_ver_for_upgrade_script=$cp4a_operator_csv_version
                fi
            fi
        fi
    fi
    ############## End - Create bm-cp4ba-shared-info/ibm-cp4ba-content-shared-info configMap ##############


    ############## Start - Check existing icp4acluster/content cr or not ##############
    PLATFORM_SELECTED=$(eval echo $(${CLI_CMD} get icp4acluster $(${CLI_CMD} get icp4acluster --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS | grep NAME -v | awk '{print $1}') --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o yaml | grep sc_deployment_platform | tail -1 | cut -d ':' -f 2))
    if [[ -z $PLATFORM_SELECTED ]]; then
        PLATFORM_SELECTED=$(eval echo $(${CLI_CMD} get content $(${CLI_CMD} get content --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS | grep NAME -v | awk '{print $1}') --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o yaml | grep sc_deployment_platform | tail -1 | cut -d ':' -f 2))
        if [[ -z $PLATFORM_SELECTED ]]; then
            fail "No custom resource for CP4BA found in the project \"$CP4BA_SERVICES_NS\", exiting"
            exit 1
        fi
    fi
    ############## End - Check existing icp4acluster/content cr or not ##############

    ############## Start - Decide which CPfs migration mode should be used ##############
    #DBACLD-166863: Calling determine_upgrade_mode function
    determine_upgrade_mode
    ############## End - Decide which CPfs migration mode should be used ##############

    ############## Start - Decide which catalog source (GNC/Private) should be used ##############
    if [[ "$PLATFORM_SELECTED" != "others" ]]; then
        # checking existing catalog type
        if ${CLI_CMD} get catalogsource -n openshift-marketplace --no-headers --ignore-not-found | grep ibm-cp4a-operator-catalog >/dev/null 2>&1; then
            CATALOG_FOUND="Yes"
            PINNED="Yes"
        elif ${CLI_CMD} get catalogsource -n openshift-marketplace --no-headers --ignore-not-found | grep ibm-operator-catalog >/dev/null 2>&1; then
            CATALOG_FOUND="Yes"
            PINNED="No"
        else
            CATALOG_FOUND="No"
            PINNED="Yes" # Fresh install use pinned catalog source
        fi

        # Check --enable-private-catalog is set or not and whether reasonable for shared2shared
        # To call select_private_catalog_cp4ba if not set --enable-private-catalog option.
        if ${CLI_CMD} get catalogsource -n $TARGET_PROJECT_NAME --no-headers --ignore-not-found | grep ibm-cp4a-operator-catalog >/dev/null 2>&1; then
            PRIVATE_CATALOG_FOUND="Yes"
            ENABLE_PRIVATE_CATALOG=1
            info "Found this CP4BA deployment is installed using private catalog in the project \"$TARGET_PROJECT_NAME\""
        elif ${CLI_CMD} get catalogsource -n openshift-marketplace --no-headers --ignore-not-found | grep ibm-cp4a-operator-catalog >/dev/null 2>&1; then
            PRIVATE_CATALOG_FOUND="No"
            info "Found this CP4BA deployment is installed using global catalog in the project \"openshift-marketplace\""
            if [[ $ENABLE_PRIVATE_CATALOG -eq 1 && $UPGRADE_MODE == "shared2shared" ]]; then
                ENABLE_PRIVATE_CATALOG=0
                warning "Can NOT switch catalog source from global catalog namespace (GCN) to private catalog (namespace-scoped) when migrating IBM Cloud Pak foundational services from \"Cluster-scoped to Namespace-scoped\"."
                prompt_press_any_key_to_continue
            elif [[ $ENABLE_PRIVATE_CATALOG -eq 1 && ($UPGRADE_MODE == "shared2dedicated" || $UPGRADE_MODE == "dedicated2dedicated") ]]; then
                info "You have set the option \"--enable-private-catalog\" for this CP4BA deployment to use a private catalog"
            elif [[ $ENABLE_PRIVATE_CATALOG -eq 0 || -z $ENABLE_PRIVATE_CATALOG ]]; then
                if [[ $UPGRADE_MODE == "shared2dedicated" || $UPGRADE_MODE == "dedicated2dedicated" ]]; then
                    select_private_catalog_cp4ba
                elif [[ $UPGRADE_MODE == "shared2shared" ]]; then
                    info "Keep using the global catalog namespace (GCN) for this CP4BA deployment when migrating IBM Cloud Pak foundational services from \"Cluster-scoped\" to \"Cluster-scoped\"."
                    sleep 2
                fi
            fi
        fi

        # Checking whether can switch GCN to namesapce scoped catalog source if CPfs/CP4BA is in AllNamespace
        if [[ $ENABLE_PRIVATE_CATALOG -eq 1 && $ALL_NAMESPACE_FLAG == "Yes" ]]; then
            ENABLE_PRIVATE_CATALOG=0
            warning "Can NOT switch catalog source from global catalog namespace (GCN) to private catalog (namespace-scoped) when CP4BA is in AllNamespaces."
            prompt_press_any_key_to_continue
        elif [[ $ENABLE_PRIVATE_CATALOG -eq 1 && $PRIVATE_CATALOG_FOUND == "No" && ($UPGRADE_MODE == "shared2dedicated" || $UPGRADE_MODE == "dedicated2dedicated") ]]; then
            info "The global catalog namespace (GCN) will be switched to private catalog (namespace-scoped)."
            sleep 2
        elif [[ $PRIVATE_CATALOG_FOUND == "Yes" ]]; then
            ENABLE_PRIVATE_CATALOG=1
            info "Keep using the private catalog (namespace-scoped) for this CP4BA deployment."
            sleep 2
        fi

        # For shared->dedicated upgrade, we should allow user option to keep "global catalog"
        if [[ $ENABLE_PRIVATE_CATALOG -eq 0 && $UPGRADE_MODE == "shared2dedicated" ]]; then
            echo "${RED_TEXT}[WARNING]${RESET_TEXT}: ${YELLOW_TEXT}Before proceeding with the upgrade: if you have multiple CP4BA deployments on this cluster and you don't want them to be updated, update installPlan approval for BTS and EDB PostgreSQL on the other CP4BA deployments from \"Automatic\" to \"Manual\".${RESET_TEXT}"
            prompt_press_any_key_to_continue
        fi
    fi
    ############## End - Decide which catalog source (GNC/Private) should be used ##############


    ############## Start - Check CSS selected or not ##############
    # Retrieve existing Content CR
    ${CLI_CMD} get crd |grep contents.icp4a.ibm.com >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        content_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        if [ ! -z $content_cr_name ]; then
            cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
            owner_ref=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)

            if [[ ${owner_ref} != "ICP4ACluster" ]]; then
                CONTENT_CR_EXIST="Yes"
                css_flag=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.content_optional_components.css)
                css_flag=$(echo $css_flag | tr '[:upper:]' '[:lower:]')
                # Check fncm_secret_name for default ibm-fncm-secret
                fncm_secret_name_val=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.ecm_configuration.fncm_secret_name)
                if [[ ! -z $fncm_secret_name_val ]]; then
                    CP4BA_IBM_FNCM_SECRET_NAME=$fncm_secret_name_val
                fi
            fi
        fi
    fi

    # Retrieve existing ICP4ACluster CR
    icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')

    if [ ! -z $icp4acluster_cr_name ]; then
        cr_metaname=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
        ${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}

        convert_olm_cr "${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}"
        if [[ $olm_cr_flag == "No" ]]; then
            existing_pattern_list=""
            existing_opt_component_list=""
            EXISTING_PATTERN_ARR=()
            EXISTING_OPT_COMPONENT_ARR=()
            existing_pattern_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_patterns`
            existing_opt_component_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_optional_components`
            OIFS=$IFS
            IFS=',' read -r -a EXISTING_PATTERN_ARR <<< "$existing_pattern_list"
            IFS=',' read -r -a EXISTING_OPT_COMPONENT_ARR <<< "$existing_opt_component_list"
            IFS=$OIFS
            # Check fncm_secret_name for default ibm-fncm-secret
            fncm_secret_name_val=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.ecm_configuration.fncm_secret_name`
            if [[ ! -z $fncm_secret_name_val ]]; then
                CP4BA_IBM_FNCM_SECRET_NAME=$fncm_secret_name_val
            fi
        fi
    fi
    ############## End - Check CSS selected or not ##############

    ######### START - THE Check to see if SCIM is configured in the Domain ########

    # This check is only when Content pattern exists and can be skipped otherwise
    # https://jsw.ibm.com/browse/DBACLD-157386 https://jsw.ibm.com/browse/DBACLD-178101 https://jsw.ibm.com/browse/DBACLD-177550 https://jsw.ibm.com/browse/DBACLD-177742
    if [[ $CONTENT_CR_EXIST == "Yes" || (" ${EXISTING_PATTERN_ARR[@]} " =~ "content") || ((" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") && (! " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-process-service")) || (" ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing") || (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "baw_authoring") || (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "ae_data_persistence") ]]; then
        info "Determining if the directory provider type is SCIM for Content Process Engine."
        CONTENT_DEPLOYMENT_NAME=$(${CLI_CMD} get deployment ibm-content-operator --no-headers --ignore-not-found -n $TARGET_PROJECT_NAME -o name)
        if [[ ! -z $CONTENT_DEPLOYMENT_NAME ]]; then
            ${CLI_CMD} scale --replicas=1 deployment ibm-content-operator -n $TARGET_PROJECT_NAME >/dev/null 2>&1
            info "Waiting for ibm-content-operator pod to be ready in the project \"$TARGET_PROJECT_NAME\"."
            maxRetry=10
            for ((retry=0;retry<=${maxRetry};retry++)); do
                pod_name=$(${CLI_CMD} get pod -l=name=ibm-content-operator -n $TARGET_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                if [[ -z $pod_name ]]; then
                    if [[ $retry -eq ${maxRetry} ]]; then
                        printf "\n"
                        if [[ -z $pod_name ]]; then
                            warning "Timeout waiting for ibm-content-operator pod to be ready in the project \"$TARGET_PROJECT_NAME\""
                        fi
                        exit 1
                    else
                        sleep 15
                        echo -n "..."
                        continue
                    fi
                else
                    break
                fi
            done
        fi
        # Calling function to check the SCIM configuration
        is_scim_enabled
        #echo " SCIM DETAILS ->>> $IS_SCIM_ENABLED"
        # Based on the shared-info configmap we are using , we will patch the scim_configured value in that CM to later be used in upgradeDeployment mode
        if [[ "$cp4ba_shared_info_used" == true ]]; then
            shared_info_configmap="ibm-cp4ba-shared-info"
        else
            shared_info_configmap="ibm-cp4ba-content-shared-info"
        fi
        #Patching configmap with the SCIM configuration value
        if ! ${CLI_CMD} patch configmap "$shared_info_configmap" -n "$CP4BA_SERVICES_NS" --type=json -p="[{'op': 'add', 'path': '/data/scim_configured', 'value': '$(echo $IS_SCIM_ENABLED)'}]" >/dev/null 2>&1; then
            warning "Failed to patch ConfigMap '$shared_info_configmap' in namespace '$CP4BA_SERVICES_NS' with the SCIM configuration details."
        fi
    fi
    ######### END - THE Check to see if SCIM is configured in the Domain ########

    ############## Start - Decide whether to create savepoint for Flink job ##############
    # NOTES: No need to create save point for upgrade IFIX by IFIX
    # Checking CSV for cp4ba-operator/content-operator/bai-operator to decide whether to do BAI save point during IFIX to IFIX upgrade
    sub_inst_list=$(${CLI_CMD} get subscriptions.operators.coreos.com -n $TEMP_OPERATOR_PROJECT_NAME|grep ibm-cp4a-operator-catalog|awk '{if(NR>0){if(NR==1){ arr=$1; }else{ arr=arr" "$1; }} } END{ print arr }')
    if [[ -z $sub_inst_list ]]; then
        info "No existing CP4BA subscriptions found, continuing ..."
        # exit 1
    fi
    sub_array=($sub_inst_list)
    target_csv_version=${CP4BA_CSV_VERSION//v/}
    for i in ${!sub_array[@]}; do
        if [[ ! -z "${sub_array[i]}" ]]; then
            if [[ ${sub_array[i]} = ibm-cp4a-operator* || ${sub_array[i]} = ibm-content-operator* || ${sub_array[i]} = ibm-insights-engine-operator* || ${sub_array[i]} = ibm-workflow-operator* ]]; then
                current_version=$(${CLI_CMD} get subscriptions.operators.coreos.com ${sub_array[i]} --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                installed_version=$(${CLI_CMD} get subscriptions.operators.coreos.com ${sub_array[i]} --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                if [[ -z $current_version || -z $installed_version ]]; then
                    error "Failed to retrieve the installed or current CSV. Aborting the upgrade procedure. Check the subscription status of ${sub_array[i]}."
                    exit 1
                fi
                case "${sub_array[i]}" in
                "ibm-cp4a-operator"*)
                    prefix_sub="ibm-cp4a-operator.v"
                    ;;
                "ibm-content-operator"*)
                    prefix_sub="ibm-content-operator.v"
                    ;;
                "ibm-insights-engine-operator"*)
                    prefix_sub="ibm-insights-engine-operator.v"
                    ;;
                esac
                current_version=${current_version#"$prefix_sub"}
                installed_version=${installed_version#"$prefix_sub"}
                if [[ $current_version != $installed_version || $current_version != $target_csv_version || $installed_version != $target_csv_version ]]; then
                    RUN_BAI_SAVEPOINT="Yes"
                fi
            fi
        else
            fail "Subscription '${sub_array[i]}' not found! Exiting now..."
            exit 1
        fi
    done

    # No need to create Flink job savepoint for upgrading from IFIX to IFIX
    # The previous condition if [[ $cp4a_operator_csv_version != "$CP4BA_RELEASE_BASE" ]]; will not work for an upgrade from 24.0.0 IF001 to 24.0.1 since the csv version of 24.0.0 IF001 is the same as the release base of 24.0.1
    # Updated this condition to use the is_ifix_to_ifix_upgrade flag which can correctly detect if it is an n-1 to n upgrade or ifix to ifix upgrade
    if [[ "$is_ifix_to_ifix_upgrade" == "false" ]]; then
 
        if [[ "$ALLOW_DIRECT_UPGRADE" == 1 ]]; then # Only execute this block when it's a skip-version upgrade.
            # In 24.0.0, only merge bai.json into yaml but not call savepoint RestAPI because that savepoint should be done during migration from ES to OS
            if [[ $RERUN_UPGRADE_DEPLOYMENT == "Yes" ]]; then
                RUN_BAI_SAVEPOINT="Yes"
            fi
            if [[ $RUN_BAI_SAVEPOINT == "Yes" ]]; then
                # Retrieve existing Content CR for Create BAI save points
                ${CLI_CMD} get crd |grep contents.icp4a.ibm.com >/dev/null 2>&1
                if [ $? -eq 0 ]; then
                    content_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
                    if [ ! -z $content_cr_name ]; then
                        info "Retrieving existing CP4BA Content (Kind: content.icp4a.ibm.com) Custom Resource"
                        cr_type="content"
                        cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
                        owner_ref=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)

                        if [[ ${owner_ref} != "ICP4ACluster" ]]; then
                            ${CLI_CMD} get $cr_type $content_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_CONTENT_CR_TMP}

                            # Backup existing content CR
                            mkdir -p ${UPGRADE_DEPLOYMENT_CR_BAK} >/dev/null 2>&1
                            ${COPY_CMD} -rf ${UPGRADE_DEPLOYMENT_CONTENT_CR_TMP} ${UPGRADE_DEPLOYMENT_CONTENT_CR_BAK}

                            mkdir -p ${TEMP_FOLDER} >/dev/null 2>&1
                            bai_flag=`cat $UPGRADE_DEPLOYMENT_CONTENT_CR_TMP | ${YQ_CMD} r - spec.content_optional_components.bai`
                            if [[ $bai_flag == "True" || $bai_flag == "true" ]]; then
                                if [[ "$machine" == "Mac" ]]; then
                                    which jq &>/dev/null
                                    [[ $? -ne 0 ]] && \
                                    echo -e  "\x1B[1;31mUnable to locate the jq CLI. You must install it to run this script on macOS.\x1B[0m" && \
                                    exit 1
                                fi

                                if [[ -e ${UPGRADE_DEPLOYMENT_CR}/bai.json ]]; then
                                    # Check if bai.json is empty
                                    info "Found the Flink savepoint in the temp file \"${UPGRADE_DEPLOYMENT_CR}/bai.json\""
                                    info "Starting to parse and merge into temp custom resource file \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                    touch ${UPGRADE_DEPLOYMENT_BAI_TMP} >/dev/null 2>&1
                                    if [[ -e ${UPGRADE_DEPLOYMENT_CR}/bai.json ]]; then
                                        [ "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" != "[]" ] && mkdir -p ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup && cp ${UPGRADE_DEPLOYMENT_CR}/bai.json ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup/bai_$(date +'%Y%m%d%H%M%S').json
                                    fi
                                    # json_file_content="[]"
                                    if [ "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" == "[]" ] ;then
                                        warning "Fetch Flink job savepoints for the recovery path using above REST API manually, then place the JSON file (bai.json) under the directory \"${TEMP_FOLDER}/\""
                                    else
                                        ##########################################################################################################################
                                        ## In 24.0.1 and later, we'll only support n-1 upgrade therefore we're back to the old way of saving content event-forwarder savepoint and bai-content savepoint UNLESS the ALLOW_DIRECT_UPGRADE == 1 .
                                        ##########################################################################################################################
                                        # if [[ "$ALLOW_DIRECT_UPGRADE" != 1 ]]; then
                                        #     if [[ "$machine" == "Mac" ]]; then
                                        #         tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-event-forwarder)
                                        #     else
                                        #         tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-event-forwarder |cut -d':' -f2)
                                        #     fi
                                        #     tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                        #     if [ ! -z "$tmp_recovery_path" ]; then
                                        #         ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.event-forwarder.recovery_path ${tmp_recovery_path}
                                        #         success "Create savepoint for Event-forwarder: \"$tmp_recovery_path\""
                                        #         info "When run \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.event-forwarder.recovery_path."
                                        #     fi
                                        #     if [[ "$machine" == "Mac" ]]; then
                                        #         tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-content)
                                        #     else
                                        #         tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-content |cut -d':' -f2)
                                        #     fi
                                        #     tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                        #     if [ ! -z "$tmp_recovery_path" ]; then
                                        #         ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.content.recovery_path ${tmp_recovery_path}
                                        #         success "Merged Flink savepoint for Content: \"$tmp_recovery_path\""
                                        #         info "When run \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.content.recovery_path."
                                        #     fi
                                        # fi
                                        if [[ "$machine" == "Mac" ]]; then
                                            tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-icm)
                                        else
                                            tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-icm |cut -d':' -f2)
                                        fi
                                        tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                        if [ ! -z "$tmp_recovery_path" ]; then
                                            ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.icm.recovery_path ${tmp_recovery_path}
                                            success "Merged Flink savepoint for ICM: \"$tmp_recovery_path\" into \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                            info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.icm.recovery_path."
                                        fi
                                        if [[ "$machine" == "Mac" ]]; then
                                            tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-odm)
                                        else
                                            tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-odm |cut -d':' -f2)
                                        fi
                                        tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                        if [ ! -z "$tmp_recovery_path" ]; then
                                            ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.odm.recovery_path ${tmp_recovery_path}
                                            success "Merged Flink savepoint for ODM: \"$tmp_recovery_path\" into \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                            info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.odm.recovery_path."
                                        fi
                                        if [[ "$machine" == "Mac" ]]; then
                                            tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-bawadv)
                                        else
                                            tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-bawadv |cut -d':' -f2)
                                        fi
                                        tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                        if [ ! -z "$tmp_recovery_path" ]; then
                                            ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.bawadv.recovery_path ${tmp_recovery_path}
                                            success "Merged Flink savepoint for BAW ADV: \"$tmp_recovery_path\" into \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                            info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.bawadv.recovery_path."
                                        fi
                                        if [[ "$machine" == "Mac" ]]; then
                                            tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-bpmn)
                                        else
                                            tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-bpmn |cut -d':' -f2)
                                        fi
                                        tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                        if [ ! -z "$tmp_recovery_path" ]; then
                                            ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.bpmn.recovery_path ${tmp_recovery_path}
                                            success "Merged Flink savepoint for BPMN: \"$tmp_recovery_path\" into \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                            info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.bpmn.recovery_path."
                                        fi
                                    fi
                                else
                                    fail "Could not find \"${UPGRADE_DEPLOYMENT_CR}/bai.json\" for Flink savepoint."
                                    msg "Fetch Flink job savepoints for the recovery path using above REST API manually, then place the JSON file (bai.json) under the directory \"${TEMP_FOLDER}/\""
                                fi
                            fi
                        fi
                    fi
                fi

                # Retrieve existing ICP4ACluster CR for Create BAI save points
                icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
                if [ ! -z $icp4acluster_cr_name ]; then
                    info "Retrieving existing CP4BA ICP4ACluster (Kind: icp4acluster.icp4a.ibm.com) Custom Resource"
                    cr_type="icp4acluster"
                    cr_metaname=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
                    ${CLI_CMD} get $cr_type $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}

                    # Backup existing icp4acluster CR
                    mkdir -p ${UPGRADE_DEPLOYMENT_CR_BAK}
                    ${COPY_CMD} -rf ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP} ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_BAK}


                    convert_olm_cr "${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}"
                    if [[ $olm_cr_flag == "No" ]]; then
                        # Get EXISTING_PATTERN_ARR/EXISTING_OPT_COMPONENT_ARR
                        existing_pattern_list=""
                        existing_opt_component_list=""

                        EXISTING_PATTERN_ARR=()
                        EXISTING_OPT_COMPONENT_ARR=()
                        existing_pattern_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_patterns`
                        existing_opt_component_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_optional_components`

                        OIFS=$IFS
                        IFS=',' read -r -a EXISTING_PATTERN_ARR <<< "$existing_pattern_list"
                        IFS=',' read -r -a EXISTING_OPT_COMPONENT_ARR <<< "$existing_opt_component_list"
                        IFS=$OIFS
                    fi

                    # Check BAI save points json
                    mkdir -p ${TEMP_FOLDER} >/dev/null 2>&1
                    if [[ (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "bai") ]]; then
                        # Check the jq install on MacOS
                        if [[ "$machine" == "Mac" ]]; then
                            which jq &>/dev/null
                            [[ $? -ne 0 ]] && \
                            echo -e  "\x1B[1;31mUnable to locate the jq CLI. You must install it to run this script on macOS.\x1B[0m" && \
                            exit 1
                        fi

                        if [[ -e ${UPGRADE_DEPLOYMENT_CR}/bai.json ]]; then
                            info "Found the Flink savepoint in the temp file \"${UPGRADE_DEPLOYMENT_CR}/bai.json\""
                            info "Starting to parse and merge into temp custom resource file \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                            touch ${UPGRADE_DEPLOYMENT_BAI_TMP} >/dev/null 2>&1
                            if [[ -e ${UPGRADE_DEPLOYMENT_CR}/bai.json ]]; then
                                [ "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" != "[]" ] && mkdir -p ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup && cp ${UPGRADE_DEPLOYMENT_CR}/bai.json ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup/bai_$(date +'%Y%m%d%H%M%S').json
                            fi
                            # json_file_content="[]"
                            if [ "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" == "[]" ] ;then
                                warning "Fetch Flink job savepoints for the recovery path using above REST API manually, then place the JSON file (bai.json) under the directory \"${TEMP_FOLDER}/\""
                            else
                                ##########################################################################################################################
                                ## In 24.0.1 and later, we'll only support n-1 upgrade therefore we're back to the old way of saving content event-forwarder savepoint and bai-content savepoint UNLESS the ALLOW_DIRECT_UPGRADE == 1 .
                                ##########################################################################################################################
                                # if [[ "$ALLOW_DIRECT_UPGRADE" != 1 ]]; then
                                #     if [[ "$machine" == "Mac" ]]; then
                                #         tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-event-forwarder)
                                #     else
                                #         tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-event-forwarder |cut -d':' -f2)
                                #     fi
                                #     tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                #     if [ ! -z "$tmp_recovery_path" ]; then
                                #         ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.event-forwarder.recovery_path ${tmp_recovery_path}
                                #         success "Create savepoint for Event-forwarder: \"$tmp_recovery_path\""
                                #         info "When run \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.event-forwarder.recovery_path."
                                #     fi
                                #     if [[ "$machine" == "Mac" ]]; then
                                #         tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-content)
                                #     else
                                #         tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-content |cut -d':' -f2)
                                #     fi
                                #     tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                #     if [ ! -z "$tmp_recovery_path" ]; then
                                #         ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.content.recovery_path ${tmp_recovery_path}
                                #         success "Merged Flink savepoint for Content: \"$tmp_recovery_path\""
                                #         info "When run \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.content.recovery_path."
                                #     fi
                                # fi

                                if [[ "$machine" == "Mac" ]]; then
                                    tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-icm)
                                else
                                    tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-icm |cut -d':' -f2)
                                fi
                                tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                if [ ! -z "$tmp_recovery_path" ]; then
                                    ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.icm.recovery_path ${tmp_recovery_path}
                                    success "Merged Flink savepoint for ICM: \"$tmp_recovery_path\" into \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                    info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.icm.recovery_path."
                                fi

                                if [[ "$machine" == "Mac" ]]; then
                                    tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-odm)
                                else
                                    tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-odm |cut -d':' -f2)
                                fi
                                tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                if [ ! -z "$tmp_recovery_path" ]; then
                                    ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.odm.recovery_path ${tmp_recovery_path}
                                    success "Merged Flink savepoint for ODM: \"$tmp_recovery_path\" into \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                    info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.odm.recovery_path."
                                fi

                                if [[ "$machine" == "Mac" ]]; then
                                    tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-bawadv)
                                else
                                    tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-bawadv |cut -d':' -f2)
                                fi
                                tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                if [ ! -z "$tmp_recovery_path" ]; then
                                    ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.bawadv.recovery_path ${tmp_recovery_path}
                                    success "Merged Flink savepoint for BAW ADV: \"$tmp_recovery_path\" into \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                    info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.bawadv.recovery_path."
                                fi

                                if [[ "$machine" == "Mac" ]]; then
                                    tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-bpmn)
                                else
                                    tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-bpmn |cut -d':' -f2)
                                fi
                                tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                if [ ! -z "$tmp_recovery_path" ]; then
                                    ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.bpmn.recovery_path ${tmp_recovery_path}
                                    success "Merged Flink savepoint for BPMN: \"$tmp_recovery_path\" into \"${UPGRADE_DEPLOYMENT_BAI_TMP}\""
                                    info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.bpmn.recovery_path."
                                fi
                            fi
                        else
                            fail "Could not find \"${UPGRADE_DEPLOYMENT_CR}/bai.json\" for Flink savepoint."
                            msg "Fetch Flink job savepoints for the recovery path using above REST API manually, then place the JSON file (bai.json) under the directory \"${TEMP_FOLDER}/\""
                        fi 
                    fi
                fi # End of checking for bai.json for ICP4ACluster CR
            fi
        fi #  End of block for skip-version upgrade. This is BAI related block when $ALLOW_DIRECT_UPGRADE =1 
        # In 24.0.0, follow the flow of migration from  Elasticsearch to Opensearch, the bai savepoint creation already done before upgrade CP4BA
        # So do not rerun savepoint. But need to covert bai json into UPGRADE_DEPLOYMENT_BAI_TMP for next upgradeDeployment mode.
        # Keep below logic for future IFIX to IFX upgrade.  Setting the RUN_BAI_SAVEPOINT="No" which will skip the savepoint creation in IFIX to IFIX upgrade
        if [[ "$is_ifix_to_ifix_upgrade" == "true" ]]; then
            RUN_BAI_SAVEPOINT="No"
        fi
        # This section is for normal increment, n-1, upgrade like 24.0.0 to 24.0.1 for BAI.
        if [[ $RUN_BAI_SAVEPOINT == "Yes" ]]; then
            # Retrieve existing Content CR for Create BAI save points
            ${CLI_CMD} get crd |grep contents.icp4a.ibm.com >/dev/null 2>&1
            if [ $? -eq 0 ]; then
                content_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
                if [ ! -z $content_cr_name ]; then
                    info "Retrieving existing CP4BA Content (Kind: content.icp4a.ibm.com) Custom Resource"
                    cr_type="content"
                    cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
                    owner_ref=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)
                    if [[ ${owner_ref} == "ICP4ACluster" ]]; then
                        echo
                    else
                        ${CLI_CMD} get $cr_type $content_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_CONTENT_CR_TMP}

                        # Backup existing content CR
                        mkdir -p ${UPGRADE_DEPLOYMENT_CR_BAK} >/dev/null 2>&1
                        ${COPY_CMD} -rf ${UPGRADE_DEPLOYMENT_CONTENT_CR_TMP} ${UPGRADE_DEPLOYMENT_CONTENT_CR_BAK}

                        # Create BAI save points
                        info "Checking whether BAI install in this CP4BA deployment."

                        mkdir -p ${TEMP_FOLDER} >/dev/null 2>&1
                        bai_flag=`cat $UPGRADE_DEPLOYMENT_CONTENT_CR_TMP | ${YQ_CMD} r - spec.content_optional_components.bai`
                        if [[ $bai_flag == "True" || $bai_flag == "true" ]]; then
                            info "Found BAI installed in this CP4BA deployment."
                            # Check the jq install on MacOS
                            if [[ "$machine" == "Mac" ]]; then
                                which jq &>/dev/null
                                [[ $? -ne 0 ]] && \
                                echo -e  "\x1B[1;31mUnable to locate the jq CLI. You must install it to run this script on macOS.\x1B[0m" && \
                                exit 1
                            fi

                            info "Create the BAI savepoints for recovery path when merge custom resource"
                            ${CLI_CMD} get crd |grep insightsengines.icp4a.ibm.com >/dev/null 2>&1
                            if [ $? -eq 0 ]; then
                                INSIGHTS_ENGINE_CR=$(${CLI_CMD} get insightsengines.icp4a.ibm.com --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o name)
                            fi
                            if [[ -z $INSIGHTS_ENGINE_CR ]]; then
                                INSIGHTS_ENGINE_CR=$(${CLI_CMD} get insightsengines.insightsengine.automation.ibm.com --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o name)
                                if [[ -z $INSIGHTS_ENGINE_CR ]]; then
                                    error "insightsengines custom resource instance was not found in the project \"${CP4BA_SERVICES_NS}\"."
                                fi
                                # exit 1
                            fi
                            if [[ ! -z $INSIGHTS_ENGINE_CR ]]; then
                                MANAGEMENT_URL=$(${CLI_CMD} get ${INSIGHTS_ENGINE_CR} --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o jsonpath='{.status.components.management.endpoints[?(@.scope=="External")].uri}')
                                MANAGEMENT_AUTH_SECRET=$(${CLI_CMD} get ${INSIGHTS_ENGINE_CR} --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o jsonpath='{.status.components.management.endpoints[?(@.scope=="External")].authentication.secret.secretName}')
                                MANAGEMENT_USERNAME=$(${CLI_CMD} get secret ${MANAGEMENT_AUTH_SECRET} --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o jsonpath='{.data.username}' | base64 -d)
                                MANAGEMENT_PASSWORD=$(${CLI_CMD} get secret ${MANAGEMENT_AUTH_SECRET} --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o jsonpath='{.data.password}' | base64 -d)
                                if [[ -z "$MANAGEMENT_URL" || -z "$MANAGEMENT_AUTH_SECRET" || -z "$MANAGEMENT_USERNAME" || -z "$MANAGEMENT_PASSWORD" ]]; then
                                    error "Can not create the BAI savepoints for recovery path."
                                    # exit 1
                                else
                                    # rm -rf ${UPGRADE_DEPLOYMENT_CR}/bai.json >/dev/null 2>&1
                                    touch ${UPGRADE_DEPLOYMENT_BAI_TMP} >/dev/null 2>&1
                                    if [[ -e ${UPGRADE_DEPLOYMENT_CR}/bai.json ]]; then
                                        [ "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" != "[]" ] && mkdir -p ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup && cp ${UPGRADE_DEPLOYMENT_CR}/bai.json ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup/bai_$(date +'%Y%m%d%H%M%S').json
                                    fi
                                    curl -X POST -k -u ${MANAGEMENT_USERNAME}:${MANAGEMENT_PASSWORD} "${MANAGEMENT_URL}/api/v1/processing/jobs/savepoints" -o ${UPGRADE_DEPLOYMENT_CR}/bai.json >/dev/null 2>&1

                                    json_file_content="[]"
                                    if [ "$json_file_content" == "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" ] ;then
                                        fail "None return in \"${UPGRADE_DEPLOYMENT_CR}/bai.json\" when request BAI savepoint through REST API: curl -X POST -k -u ${MANAGEMENT_USERNAME}:${MANAGEMENT_PASSWORD} \"${MANAGEMENT_URL}/api/v1/processing/jobs/savepoints\" "
                                        warning "Fetch Flink job savepoints for the recovery path using above REST API manually, then place the JSON file (bai.json) under the directory \"${TEMP_FOLDER}/\""
                                        prompt_press_any_key_to_continue
                                    fi
                                    ##########################################################################################################################
                                    ## In 24.0.1 and later, we'll only support n-1 upgrade therefore we're back to the old way of saving content event-forwarder savepoint and bai-content savepoint UNLESS the ALLOW_DIRECT_UPGRADE == 1 .
                                    ##########################################################################################################################
                                    if [[ "$ALLOW_DIRECT_UPGRADE" != 1 ]]; then
                                        if [[ "$machine" == "Mac" ]]; then
                                            tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-event-forwarder)
                                        else
                                            tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-event-forwarder |cut -d':' -f2)
                                        fi
                                        tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                        if [ ! -z "$tmp_recovery_path" ]; then
                                            ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.event-forwarder.recovery_path ${tmp_recovery_path}
                                            success "Create savepoint for Event-forwarder: \"$tmp_recovery_path\""
                                            info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.event-forwarder.recovery_path."
                                        fi
                                        if [[ "$machine" == "Mac" ]]; then
                                            tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-content)
                                        else
                                            tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-content |cut -d':' -f2)
                                        fi
                                        tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                        if [ ! -z "$tmp_recovery_path" ]; then
                                            ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.content.recovery_path ${tmp_recovery_path}
                                            success "Merged Flink savepoint for Content: \"$tmp_recovery_path\""
                                            info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.content.recovery_path."
                                        fi
                                    fi
                                    if [[ "$machine" == "Mac" ]]; then
                                        tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-icm)
                                    else
                                        tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-icm |cut -d':' -f2)
                                    fi
                                    tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                    if [ ! -z "$tmp_recovery_path" ]; then
                                        ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.icm.recovery_path ${tmp_recovery_path}
                                        success "Merged Flink savepoint for ICM: \"$tmp_recovery_path\""
                                        info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.icm.recovery_path."
                                    fi

                                    if [[ "$machine" == "Mac" ]]; then
                                        tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-odm)
                                    else
                                        tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-odm |cut -d':' -f2)
                                    fi
                                    tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                    if [ ! -z "$tmp_recovery_path" ]; then
                                        ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.odm.recovery_path ${tmp_recovery_path}
                                        success "Merged Flink savepoint for ODM: \"$tmp_recovery_path\""
                                        info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.odm.recovery_path."
                                    fi

                                    if [[ "$machine" == "Mac" ]]; then
                                        tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-bawadv)
                                    else
                                        tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-bawadv |cut -d':' -f2)
                                    fi
                                    tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                    if [ ! -z "$tmp_recovery_path" ]; then
                                        ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.bawadv.recovery_path ${tmp_recovery_path}
                                        success "Merged Flink savepoint for BAW ADV: \"$tmp_recovery_path\""
                                        info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.bawadv.recovery_path."
                                    fi

                                    if [[ "$machine" == "Mac" ]]; then
                                        tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-bpmn)
                                    else
                                        tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-bpmn |cut -d':' -f2)
                                    fi
                                    tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")

                                    if [ ! -z "$tmp_recovery_path" ]; then
                                        ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.bpmn.recovery_path ${tmp_recovery_path}
                                        success "Merged Flink savepoint for BPMN: \"$tmp_recovery_path\""
                                        info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.bpmn.recovery_path."
                                    fi

                                    # Adding Navigator's recovery path
                                    if [[ "$machine" == "Mac" ]]; then
                                        tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-navigator)
                                    else
                                        tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-navigator |cut -d':' -f2)
                                    fi
                                    tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                    if [ ! -z "$tmp_recovery_path" ]; then
                                        ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.navigator.recovery_path ${tmp_recovery_path}
                                        success "Merged Flink savepoint for Navigator: \"$tmp_recovery_path\""
                                        info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.navigator.recovery_path."
                                    fi

                                    # Adding ADS's recovery path
                                    if [[ "$machine" == "Mac" ]]; then
                                        tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-ads)
                                    else
                                        tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-ads |cut -d':' -f2)
                                    fi
                                    tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                    if [ ! -z "$tmp_recovery_path" ]; then
                                        ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.ads.recovery_path ${tmp_recovery_path}
                                        success "Merged Flink savepoint for ADS: \"$tmp_recovery_path\""
                                        info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.ads.recovery_path."
                                    fi
                                fi
                            fi
                        fi
                    fi
                fi
            fi

            # Retrieve existing ICP4ACluster CR for Create BAI save points
            icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
            if [ ! -z $icp4acluster_cr_name ]; then
                info "Retrieving existing CP4BA ICP4ACluster (Kind: icp4acluster.icp4a.ibm.com) Custom Resource"
                cr_type="icp4acluster"
                cr_metaname=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
                ${CLI_CMD} get $cr_type $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}

                # Backup existing icp4acluster CR
                mkdir -p ${UPGRADE_DEPLOYMENT_CR_BAK}
                ${COPY_CMD} -rf ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP} ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_BAK}

                convert_olm_cr "${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}"
                if [[ $olm_cr_flag == "No" ]]; then
                    # Get EXISTING_PATTERN_ARR/EXISTING_OPT_COMPONENT_ARR
                    existing_pattern_list=""
                    existing_opt_component_list=""

                    EXISTING_PATTERN_ARR=()
                    EXISTING_OPT_COMPONENT_ARR=()
                    existing_pattern_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_patterns`
                    existing_opt_component_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_optional_components`

                    OIFS=$IFS
                    IFS=',' read -r -a EXISTING_PATTERN_ARR <<< "$existing_pattern_list"
                    IFS=',' read -r -a EXISTING_OPT_COMPONENT_ARR <<< "$existing_opt_component_list"
                    IFS=$OIFS
                fi

                # Create BAI save points
                info "Checking whether BAI install in this CP4BA deployment."
                mkdir -p ${TEMP_FOLDER} >/dev/null 2>&1
                if [[ (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "bai") ]]; then
                    info "Found BAI installed in this CP4BA deployment."
                    # Check the jq install on MacOS
                    if [[ "$machine" == "Mac" ]]; then
                        which jq &>/dev/null
                        [[ $? -ne 0 ]] && \
                        echo -e  "\x1B[1;31mUnable to locate the jq CLI. You must install it to run this script on macOS.\x1B[0m" && \
                        exit 1
                    fi
                    info "Create the BAI savepoints for recovery path when merge custom resource"
                    ${CLI_CMD} get crd |grep insightsengines.icp4a.ibm.com >/dev/null 2>&1
                    if [ $? -eq 0 ]; then
                        INSIGHTS_ENGINE_CR=$(${CLI_CMD} get insightsengines.icp4a.ibm.com --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o name)
                    fi
                    if [[ -z $INSIGHTS_ENGINE_CR ]]; then
                        INSIGHTS_ENGINE_CR=$(${CLI_CMD} get insightsengines.insightsengine.automation.ibm.com --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o name)
                        if [[ -z $INSIGHTS_ENGINE_CR ]]; then
                            error "insightsengines custom resource instance was not found in the project \"${CP4BA_SERVICES_NS}\"."
                        fi
                        # exit 1
                    fi
                    if [[ ! -z $INSIGHTS_ENGINE_CR ]]; then
                        MANAGEMENT_URL=$(${CLI_CMD} get ${INSIGHTS_ENGINE_CR} --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o jsonpath='{.status.components.management.endpoints[?(@.scope=="External")].uri}')
                        MANAGEMENT_AUTH_SECRET=$(${CLI_CMD} get ${INSIGHTS_ENGINE_CR} --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o jsonpath='{.status.components.management.endpoints[?(@.scope=="External")].authentication.secret.secretName}')
                        MANAGEMENT_USERNAME=$(${CLI_CMD} get secret ${MANAGEMENT_AUTH_SECRET} --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o jsonpath='{.data.username}' | base64 -d)
                        MANAGEMENT_PASSWORD=$(${CLI_CMD} get secret ${MANAGEMENT_AUTH_SECRET} --no-headers --ignore-not-found -n ${CP4BA_SERVICES_NS} -o jsonpath='{.data.password}' | base64 -d)
                        if [[ -z "$MANAGEMENT_URL" || -z "$MANAGEMENT_AUTH_SECRET" || -z "$MANAGEMENT_USERNAME" || -z "$MANAGEMENT_PASSWORD" ]]; then
                            error "Can not create the BAI savepoints for recovery path."
                            # exit 1
                        else
                            # rm -rf ${UPGRADE_DEPLOYMENT_CR}/bai.json >/dev/null 2>&1
                            touch ${UPGRADE_DEPLOYMENT_BAI_TMP} >/dev/null 2>&1
                            if [[ -e ${UPGRADE_DEPLOYMENT_CR}/bai.json ]]; then
                                [ "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" != "[]" ] && mkdir -p ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup && cp ${UPGRADE_DEPLOYMENT_CR}/bai.json ${UPGRADE_DEPLOYMENT_CR}/bai-json-backup/bai_$(date +'%Y%m%d%H%M%S').json
                            fi
                            curl -X POST -k -u ${MANAGEMENT_USERNAME}:${MANAGEMENT_PASSWORD} "${MANAGEMENT_URL}/api/v1/processing/jobs/savepoints" -o ${UPGRADE_DEPLOYMENT_CR}/bai.json >/dev/null 2>&1

                            json_file_content="[]"
                            if [ "$json_file_content" == "$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json)" ] ;then
                                fail "None return in \"${UPGRADE_DEPLOYMENT_CR}/bai.json\" when request BAI savepoint through REST API: curl -X POST -k -u ${MANAGEMENT_USERNAME}:${MANAGEMENT_PASSWORD} \"${MANAGEMENT_URL}/api/v1/processing/jobs/savepoints\" "
                                warning "Fetch Flink job savepoints for the recovery path using above REST API manually, then place the JSON file (bai.json) under the directory \"${TEMP_FOLDER}/\""
                                prompt_press_any_key_to_continue
                            fi
                            ##########################################################################################################################
                            ## In 24.0.1 and later, we'll only support n-1 upgrade therefore we're back to the old way of saving content event-forwarder savepoint and bai-content savepoint UNLESS the ALLOW_DIRECT_UPGRADE == 1 .
                            ##########################################################################################################################
                            if [[ "$ALLOW_DIRECT_UPGRADE" != 1 ]]; then
                                if [[ "$machine" == "Mac" ]]; then
                                    tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-event-forwarder)
                                else
                                    tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-event-forwarder |cut -d':' -f2)
                                fi
                                tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                if [ ! -z "$tmp_recovery_path" ]; then
                                    ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.event-forwarder.recovery_path ${tmp_recovery_path}
                                    success "Create savepoint for Event-forwarder: \"$tmp_recovery_path\""
                                    info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.event-forwarder.recovery_path."
                                fi
                                if [[ "$machine" == "Mac" ]]; then
                                    tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-content)
                                else
                                    tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-content |cut -d':' -f2)
                                fi
                                tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                                if [ ! -z "$tmp_recovery_path" ]; then
                                    ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.content.recovery_path ${tmp_recovery_path}
                                    success "Merged Flink savepoint for Content: \"$tmp_recovery_path\""
                                    info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.content.recovery_path."
                                fi
                            fi

                            if [[ "$machine" == "Mac" ]]; then
                                tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-icm)
                            else
                                tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-icm |cut -d':' -f2)
                            fi
                            tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                            if [ ! -z "$tmp_recovery_path" ]; then
                                ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.icm.recovery_path ${tmp_recovery_path}
                                success "Merged Flink savepoint for ICM: \"$tmp_recovery_path\""
                                info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.icm.recovery_path."
                            fi

                            if [[ "$machine" == "Mac" ]]; then
                                tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-odm)
                            else
                                tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-odm |cut -d':' -f2)
                            fi
                            tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                            if [ ! -z "$tmp_recovery_path" ]; then
                                ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.odm.recovery_path ${tmp_recovery_path}
                                success "Merged Flink savepoint for ODM: \"$tmp_recovery_path\""
                                info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.odm.recovery_path."
                            fi

                            if [[ "$machine" == "Mac" ]]; then
                                tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-bawadv)
                            else
                                tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-bawadv |cut -d':' -f2)
                            fi
                            tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                            if [ ! -z "$tmp_recovery_path" ]; then
                                ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.bawadv.recovery_path ${tmp_recovery_path}
                                success "Merged Flink savepoint for BAW ADV: \"$tmp_recovery_path\""
                                info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.bawadv.recovery_path."
                            fi

                            if [[ "$machine" == "Mac" ]]; then
                                tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-bpmn)
                            else
                                tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-bpmn |cut -d':' -f2)
                            fi
                            tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                            if [ ! -z "$tmp_recovery_path" ]; then
                                ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.bpmn.recovery_path ${tmp_recovery_path}
                                success "Merged Flink savepoint for BPMN: \"$tmp_recovery_path\""
                                info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.bpmn.recovery_path."
                            fi
                            # Adding Navigator's recovery path
                            if [[ "$machine" == "Mac" ]]; then
                                tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-navigator)
                            else
                                tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-navigator |cut -d':' -f2)
                            fi
                            tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                            if [ ! -z "$tmp_recovery_path" ]; then
                                ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.navigator.recovery_path ${tmp_recovery_path}
                                success "Merged Flink savepoint for Navigator: \"$tmp_recovery_path\""
                                info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.navigator.recovery_path."
                            fi
                            # Adding ADS's recovery path
                            if [[ "$machine" == "Mac" ]]; then
                                tmp_recovery_path=$(cat ${UPGRADE_DEPLOYMENT_CR}/bai.json | jq '.[].location' | grep bai-ads)
                            else
                                tmp_recovery_path=$(grep -Po '"location":.*?[^\\]"' ${UPGRADE_DEPLOYMENT_CR}/bai.json | grep bai-ads |cut -d':' -f2)
                            fi
                            tmp_recovery_path=$(sed -e 's/^"//' -e 's/"$//' <<<"$tmp_recovery_path")
                            if [ ! -z "$tmp_recovery_path" ]; then
                                ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_BAI_TMP} spec.bai_configuration.ads.recovery_path ${tmp_recovery_path}
                                success "Merged Flink savepoint for ADS: \"$tmp_recovery_path\""
                                info "When running \"cp4a-deployment -m upgradeDeployment\", this savepoint will be auto-filled into spec.bai_configuration.ads.recovery_path."
                            fi
                        fi
                    fi
                fi
            fi
        fi
    fi
    ############## End - Decide whether to create savepoint for Flink job ##############

    ############## Start - Migration CPfs mode and upgrade CP4BA Operators ##############
    if [[ "$PLATFORM_SELECTED" == "others" ]]; then
        [ -f ${UPGRADE_DEPLOYMENT_FOLDER}/upgradeOperator.yaml ] && rm ${UPGRADE_DEPLOYMENT_FOLDER}/upgradeOperator.yaml
        cp ${CUR_DIR}/../descriptors/operator.yaml ${UPGRADE_DEPLOYMENT_FOLDER}/upgradeOperator.yaml
        cncf_install
        
        # DBACLD-168537: need to re-create {{meta.name}}-fncm-custom-ssl-secret to add CSS DNSName (in case they are missing from previous deployment) which will be included in FNCM's keystores
        cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        if [[ -z $cp4ba_cr_name ]]; then
            cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        fi
        local fncm_custom_ssl_secret=$(${CLI_CMD} get secret --no-headers --ignore-not-found ${cr_name}-fncm-custom-ssl-secret -n $CP4BA_SERVICES_NS | awk '{print $1}') 
        if [[ -z $fncm_custom_ssl_secret ]]; then
            info "${cr_name}-fncm-custom-ssl-secret is not found."
        else
            info "Found ${cr_name}-fncm-custom-ssl-secret and the script will now delete the secret."
            ${CLI_CMD} delete secret ${cr_name}-fncm-custom-ssl-secret -n $CP4BA_SERVICES_NS
        fi
        # DBACLD-178263: need to re-create {{meta.name}}-ban-custom-ssl-secret to add localhost to Navigator's keystore
        local ban_custom_ssl_secret=$(${CLI_CMD} get secret --no-headers --ignore-not-found ${cr_name}-ban-custom-ssl-secret -n $CP4BA_SERVICES_NS | awk '{print $1}') 
        if [[ -z $ban_custom_ssl_secret ]]; then
            info "${cr_name}-ban-custom-ssl-secret is not found."
        else
            info "Found ${cr_name}-ban-custom-ssl-secret and the script will now delete the secret."
            ${CLI_CMD} delete secret ${cr_name}-ban-custom-ssl-secret -n $CP4BA_SERVICES_NS
        fi

    else
        #  Switch CP4BA Operator to private catalog source
        if [ $ENABLE_PRIVATE_CATALOG -eq 1 ]; then
            # switch CP4BA
            sub_inst_list=$(${CLI_CMD} get subscriptions.operators.coreos.com -n $TARGET_PROJECT_NAME|grep ibm-cp4a-operator-catalog|awk '{if(NR>0){if(NR==1){ arr=$1; }else{ arr=arr" "$1; }} } END{ print arr }')
            if [[ -z $sub_inst_list ]]; then
                info "No existing CP4BA subscriptions found, continuing ..."
                # exit 1
            fi

            sub_array=($sub_inst_list)
            for i in ${!sub_array[@]}; do
                if [[ ! -z "${sub_array[i]}" ]]; then
                    if [[ ${sub_array[i]} = ibm-cp4a-operator* || ${sub_array[i]} = ibm-cp4a-wfps-operator* || ${sub_array[i]} = ibm-content-operator* || ${sub_array[i]} = icp4a-foundation-operator* || ${sub_array[i]} = ibm-pfs-operator* || ${sub_array[i]} = ibm-ads-operator* || ${sub_array[i]} = ibm-dpe-operator* || ${sub_array[i]} = ibm-odm-operator* || ${sub_array[i]} = ibm-insights-engine-operator* || ${sub_array[i]} = ibm-workflow-operator* ]]; then
                        ${CLI_CMD} patch subscriptions.operators.coreos.com ${sub_array[i]} -n $TARGET_PROJECT_NAME -p '{"spec":{"sourceNamespace":"'"$TARGET_PROJECT_NAME"'"}}' --type=merge >/dev/null 2>&1
                        if [ $? -eq 0 ]
                        then
                            sleep 1
                            success "Switched the CatalogSource of subscription '${sub_array[i]}' to project \"$TARGET_PROJECT_NAME\"!"
                            printf "\n"
                        else
                            fail "Failed to switch the CatalogSource of subscription '${sub_array[i]}' to project \"$TARGET_PROJECT_NAME\"!"
                        fi
                    fi
                else
                    fail "Subscription '${sub_array[i]}' not found in the project \"$TARGET_PROJECT_NAME\"! Exiting now..."
                    exit 1
                fi
            done

            # switch CPfs for 24.0.0 IFIX

        fi

        #  Patch CP4BA channel to latest version, wait for all the operators are upgraded before applying operandRequest.
        sub_inst_list=$(${CLI_CMD} get subscriptions.operators.coreos.com -n $TEMP_OPERATOR_PROJECT_NAME|grep ibm-cp4a-operator-catalog|awk '{if(NR>0){if(NR==1){ arr=$1; }else{ arr=arr" "$1; }} } END{ print arr }')
        if [[ -z $sub_inst_list ]]; then
            info "No existing CP4BA subscriptions found, continuing..."
            # exit 1
        fi

        sub_array=($sub_inst_list)
        for i in ${!sub_array[@]}; do
            if [[ ! -z "${sub_array[i]}" ]]; then
                if [[ ${sub_array[i]} = ibm-cp4a-operator* || ${sub_array[i]} = ibm-cp4a-wfps-operator* || ${sub_array[i]} = ibm-content-operator* || ${sub_array[i]} = icp4a-foundation-operator* || ${sub_array[i]} = ibm-pfs-operator* || ${sub_array[i]} = ibm-ads-operator* || ${sub_array[i]} = ibm-dpe-operator* || ${sub_array[i]} = ibm-odm-operator* || ${sub_array[i]} = ibm-insights-engine-operator* || ${sub_array[i]} = ibm-workflow-operator* ]]; then
                    ${CLI_CMD} patch subscriptions.operators.coreos.com ${sub_array[i]} -n $TEMP_OPERATOR_PROJECT_NAME -p "{\"spec\":{\"channel\":\"$CP4BA_CHANNEL_VERSION\"}}" --type=merge >/dev/null 2>&1
                    if [ $? -eq 0 ]
                    then
                        success "Updated the channel of subscription '${sub_array[i]}' to $CP4BA_CHANNEL_VERSION"
                        printf "\n"
                    else
                        fail "Failed to update the channel of subscription '${sub_array[i]}' to $CP4BA_CHANNEL_VERSION! exiting now..."
                        exit 1
                    fi
                fi
            else
                fail "Subscription '${sub_array[i]}' not found! Exiting now..."
                exit 1
            fi
        done

        success "Completed to switch the channel of subscription for CP4BA operators"

        # Apply the new catalog source
        if [[ ($CATALOG_FOUND == "Yes" && $PINNED == "Yes") || $PRIVATE_CATALOG_FOUND == "Yes" ]]; then
            # switch catalog from "global" to "namespace" catalog or keep private catalog source
            if [ $ENABLE_PRIVATE_CATALOG -eq 1 ]; then
                TEMP_CATALOG_PROJECT_NAME=${TARGET_PROJECT_NAME}
                OLM_CATALOG=${PARENT_DIR}/descriptors/op-olm/catalog_source.yaml
                OLM_CATALOG_TMP=${TEMP_FOLDER}/.catalog_source.yaml

                info "Creating project \"$CERT_MANAGER_PROJECT\" for IBM Cert Manager operator catalog."
                create_project "$CERT_MANAGER_PROJECT"
                if [[ $? -eq 0 ]]; then
                    success "Created project \"$CERT_MANAGER_PROJECT\" for IBM Cert Manager operator catalog."
                fi

                info "Creating project \"$LICENSE_MANAGER_PROJECT\" for IBM Licensing operator catalog."
                create_project "$LICENSE_MANAGER_PROJECT"
                if [[ $? -eq 0 ]]; then
                    success "Created project \"$LICENSE_MANAGER_PROJECT\" for IBM Licensing operator catalog."
                    printf "\n"
                fi

                # Additionally, we would check if cs-control namespace exists.
                isProjExists=`${CLI_CMD} get project $DEDICATED_CS_PROJECT --no-headers --ignore-not-found | wc -l`  >/dev/null 2>&1
                if [ $isProjExists -eq 1 ] ; then
                    # If it exists, we will deploy the same ibm-licensing-catalog into cs-control namespace.
                    if [[ $machine == "Linux" ]]; then
                        TMP_LICENSING_OLM_CATALOG=$(mktemp --suffix=.yaml)
                    elif [[ $machine == "Mac" ]]; then
                        TMP_LICENSING_OLM_CATALOG=$(mktemp -t licensing_olm_catalog).yaml
                    fi
                    start_num="# IBM License Manager"
                    end_num="interval: 45m"

                    reading_section=false

                    while IFS= read -r line; do
                        if [[ "$line" == *"$start_num"* ]]; then
                            reading_section=true
                        fi

                        if $reading_section; then
                            echo "$line" >> "$TMP_LICENSING_OLM_CATALOG"
                        fi

                        if [[ "$line" == *"$end_num"* ]]; then
                            reading_section=false
                        fi
                    done < "${OLM_CATALOG}"

                    # replace openshift-marketplace for ibm-licensing-catalog with cs-control
                    ${SED_COMMAND} "/name: ibm-licensing-catalog/{n;s/namespace: .*/namespace: \"$DEDICATED_CS_PROJECT\"/;}" ${TMP_LICENSING_OLM_CATALOG}

                    ${CLI_CMD} apply -f $TMP_LICENSING_OLM_CATALOG >/dev/null 2>&1
                    if [ $? -eq 0 ]; then
                        echo "Create IBM License Manager Catalog source in project \"$DEDICATED_CS_PROJECT\"!"
                    else
                        echo "Generic Operator catalog source update failed"
                        exit 1
                    fi
                    rm -rf $TMP_LICENSING_OLM_CATALOG >/dev/null 2>&1
                fi

                sed "s/REPLACE_CATALOG_SOURCE_NAMESPACE/$CATALOG_NAMESPACE/g" ${OLM_CATALOG} > ${OLM_CATALOG_TMP}
                # replace all other catalogs with <CP4BA NS> namespaces
                ${SED_COMMAND} "s|namespace: .*|namespace: \"$TARGET_PROJECT_NAME\"|g" ${OLM_CATALOG_TMP}
                # replace openshift-marketplace for ibm-cert-manager-catalog with ibm-cert-manager
                ${SED_COMMAND} "/name: ibm-cert-manager-catalog/{n;s/namespace: .*/namespace: $CERT_MANAGER_PROJECT/;}" ${OLM_CATALOG_TMP}
                # replace openshift-marketplace for ibm-licensing-catalog with ibm-licensing
                ${SED_COMMAND} "/name: ibm-licensing-catalog/{n;s/namespace: .*/namespace: $LICENSE_MANAGER_PROJECT/;}" ${OLM_CATALOG_TMP}

                # CPFS suggestion to delete the old BTS Catalogs after the new catalog source for BTS is applied. 
                # Saving the name of the current BTS Catalog sources so that it can be deleted once the new catalog sources are applied
                # Any existing bts catalogs in the stream v3-35 must be deleted and only what was applied must be kept
                # To be dynamic this loop will check for any v3-35-1 or v3-35-2 catalog source names and then accordingly remove them.
                # It will not delete any older BTS catalogs like v3-33 or v3-34 
                # Moving forward from the latest refresh of the public 24.0.1 IF002 this BTS catalog will be v3-35 for any catalog source in that stream
                # In 24.0.1 IF001 , 24.0.1 GA , 24.0.0 IF005,24.0.0 IF004 we have v3-35-1 and in 24.0.1 IF002 we will have v3-35 so it will remove the older one when we upgrade
                # In 24.0.0 IF003 we have v3-34 so there will be no problems with keeping that catalog and we will not delete it.
                # https://jsw.ibm.com/browse/DBACLD-176790
                pre_upgrade_bts_catalog_names=$(${CLI_CMD} get catalogsource -n "$TARGET_PROJECT_NAME" --no-headers 2>/dev/null | awk '$1 ~ /^ibm-bts-operator-catalog/ { print $1 }')
                pre_upgrade_bts_catalog_names_to_delete=()
                for pre_upgrade_bts_catalog_name in $pre_upgrade_bts_catalog_names; do
                    #echo "here catalog ->$pre_upgrade_bts_catalog_name"
                    if [[ "$pre_upgrade_bts_catalog_name" == *"ibm-bts-operator-catalog-v3-35"* && "$pre_upgrade_bts_catalog_name" != "ibm-bts-operator-catalog-v3-35" ]]; then
                        #echo "addng this to Deleting catalog source: $pre_upgrade_bts_catalog_name"
                        pre_upgrade_bts_catalog_names_to_delete+=("$pre_upgrade_bts_catalog_name")
                        #${CLI_CMD} delete catalogsource "$pre_upgrade_bts_catalog_name" -n "$TARGET_PROJECT_NAME"
                    fi
                done

                ${CLI_CMD} apply -f $OLM_CATALOG_TMP
                if [ $? -eq 0 ]; then
                    echo "IBM Operator Catalog source updated!"
                else
                    echo "Generic Operator catalog source update failed"
                    exit 1
                fi

                # Delete BTS catalog sources that are no longer required and would cause problems with upgrade
                # https://jsw.ibm.com/browse/DBACLD-176790
                for cs in "${pre_upgrade_bts_catalog_names_to_delete[@]}"; do
                    ${CLI_CMD} delete catalogsource "$cs" -n "$TARGET_PROJECT_NAME"
                done


            else
                TEMP_CATALOG_PROJECT_NAME="openshift-marketplace"

                # CPFS suggestion to delete the old BTS Catalogs after the new catalog source for BTS is applied. 
                # Saving the name of the current BTS Catalog sources so that it can be deleted once the new catalog sources are applied
                # Any existing bts catalogs in the stream v3-35 must be deleted and only what was applied must be kept
                # To be dynamic this loop will check for any v3-35-1 or v3-35-2 catalog source names and then accordingly remove them.
                # It will not delete any older BTS catalogs like v3-33 or v3-34 
                # Moving forward from the latest refresh of the public 24.0.1 IF002 this BTS catalog will be v3-35 for any catalog source in that stream
                # In 24.0.1 IF001 , 24.0.1 GA , 24.0.0 IF005,24.0.0 IF004 we have v3-35-1 and in 24.0.1 IF002 we will have v3-35 so it will remove the older one when we upgrade
                # In 24.0.0 IF003 we have v3-34 so there will be no problems with keeping that catalog and we will not delete it.
                # https://jsw.ibm.com/browse/DBACLD-176790
                pre_upgrade_bts_catalog_names=$(${CLI_CMD} get catalogsource -n "$TEMP_CATALOG_PROJECT_NAME" --no-headers 2>/dev/null | awk '$1 ~ /^ibm-bts-operator-catalog/ { print $1 }')
                pre_upgrade_bts_catalog_names_to_delete=()
                for pre_upgrade_bts_catalog_name in $pre_upgrade_bts_catalog_names; do
                    #echo "here catalog ->$pre_upgrade_bts_catalog_name"
                    if [[ "$pre_upgrade_bts_catalog_name" == *"ibm-bts-operator-catalog-v3-35"* && "$pre_upgrade_bts_catalog_name" != "ibm-bts-operator-catalog-v3-35" ]]; then
                        #echo "addng this to Deleting catalog source: $pre_upgrade_bts_catalog_name"
                        pre_upgrade_bts_catalog_names_to_delete+=("$pre_upgrade_bts_catalog_name")
                        #${CLI_CMD} delete catalogsource "$pre_upgrade_bts_catalog_name" -n "$TARGET_PROJECT_NAME"
                    fi
                done

                info "Apply latest CP4BA catalog source ..."
                OLM_CATALOG=${PARENT_DIR}/descriptors/op-olm/catalog_source.yaml
                ${CLI_CMD} apply -f $OLM_CATALOG >/dev/null 2>&1
                if [ $? -ne 0 ]; then
                    echo "IBM Cloud Pak for Business Automation Operator catalog source update failed"
                    exit 1
                fi
                echo "Done!"

                # Delete BTS catalog sources that are no longer required and would cause problems with upgrade
                # https://jsw.ibm.com/browse/DBACLD-176790
                for cs in "${pre_upgrade_bts_catalog_names_to_delete[@]}"; do
                    #echo "Deleting catalog source: $cs"
                    ${CLI_CMD} delete catalogsource "$cs" -n "$TEMP_CATALOG_PROJECT_NAME"
                done
            fi

            # Checking ibm-cp4a-operator catalog source pod
            info "Checking CP4BA operator catalog pod ready or not in the project \"$TEMP_CATALOG_PROJECT_NAME\""
            maxRetry=50
            for ((retry=0;retry<=${maxRetry};retry++)); do
                cp4a_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=ibm-cp4a-operator-catalog -n $TEMP_CATALOG_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                fncm_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=ibm-fncm-operator-catalog -n $TEMP_CATALOG_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                postgresql_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=cloud-native-postgresql-catalog -n $TEMP_CATALOG_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                cs_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=$CS_CATALOG_VERSION -n $TEMP_CATALOG_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                if [ $ENABLE_PRIVATE_CATALOG -eq 1 ]; then
                    cert_mgr_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=ibm-cert-manager-catalog -n $CERT_MANAGER_PROJECT -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                    license_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=ibm-licensing-catalog -n $LICENSE_MANAGER_PROJECT -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                else
                    cert_mgr_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=ibm-cert-manager-catalog -n openshift-marketplace -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                    license_catalog_pod_name=$(${CLI_CMD} get pod -l=olm.catalogSource=ibm-licensing-catalog -n openshift-marketplace -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                fi
                if [[ ( -z $cert_mgr_catalog_pod_name) || ( -z $license_catalog_pod_name) || ( -z $cs_catalog_pod_name) || ( -z $cp4a_catalog_pod_name) || (-z $fncm_catalog_pod_name) || (-z $postgresql_catalog_pod_name) ]]; then
                    if [[ $retry -eq ${maxRetry} ]]; then
                        printf "\n"
                        if [[ -z $cp4a_catalog_pod_name ]]; then
                            warning "Timeout waiting for ibm-cp4a-operator-catalog catalog pod to be ready in the project \"$TEMP_CATALOG_PROJECT_NAME\""
                        elif [[ -z $fncm_catalog_pod_name ]]; then
                            warning "Timeout waiting for ibm-fncm-operator-catalog catalog pod to be ready in the project \"$TEMP_CATALOG_PROJECT_NAME\""
                        elif [[ -z $postgresql_catalog_pod_name ]]; then
                            warning "Timeout waiting for cloud-native-postgresql-catalog catalog pod to be ready in the project \"$TEMP_CATALOG_PROJECT_NAME\""
                        elif [[ -z $cs_catalog_pod_name ]]; then
                            warning "Timeout waiting for $CS_CATALOG_VERSION catalog pod to be ready in the project \"$TEMP_CATALOG_PROJECT_NAME\""
                        elif [[ -z $cert_mgr_catalog_pod_name ]]; then
                            warning "Timeout waiting for ibm-cert-manager-catalog catalog pod to be ready in the project \"openshift-marketplace\""
                        elif [[ -z $license_catalog_pod_name ]]; then
                            warning "Timeout waiting for ibm-licensing-catalog catalog pod to be ready in the project \"openshift-marketplace\""
                        fi
                        exit 1
                    else
                        sleep 30
                        echo -n "..."
                        continue
                    fi
                else
                    success "CP4BA operator catalog pod ready in the project \"$TEMP_CATALOG_PROJECT_NAME\"!"
                    break
                fi
            done
        else
            fail "IBM Cloud Pak for Business Automation catalog source not found!"
            exit 1
        fi

        # check_cp4ba_operator_version $TARGET_PROJECT_NAME
        # check_content_operator_version $TARGET_PROJECT_NAME
        # if [ -z "$UPDATE_APPROVAL_STRATEGY" ]; then
        #     # info "The default value is [automatic] for \"-s <UPDATE_APPROVAL_STRATEGY>\" option. "
        #     # info "run script with -h option for help. "
        #     # read -rsn1 -p"Press Enter/Return to continue or CTRL+C to break";echo
        #     UPDATE_APPROVAL_STRATEGY="automatic"
        # fi

        # Upgrade CP4BA operator
        info "Starting to upgrade CP4BA operator"

        # Check ibm-bts-operator/cloud-native-postgresql already upgrade to latest version
        if [[ $UPGRADE_MODE == "dedicated2dedicated"  ]]; then
            cs_service_target_namespace="$TARGET_PROJECT_NAME"
        elif [[ $UPGRADE_MODE == "shared2shared" || $UPGRADE_MODE == "shared2dedicated" ]]; then
            cs_service_target_namespace="ibm-common-services"
        fi

        # Check cloud-native-postgresql/ibm-bts-operator
        if [[ $ENABLE_PRIVATE_CATALOG -eq 0 ]]; then
            cloud_native_postgresql_ready="Yes"
            ibm_bts_operator_ready="Yes"

            if [[ "$cp4a_operator_csv_version" == "21.3."* || "$cp4a_operator_csv_version" == "22.2."* ]]; then
                ibm_cp4a_wfps_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-cp4a-wfps-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_wfps_operator_flag -ne 0 ]; then
                    ibm_cp4a_wfps_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-cp4a-wfps-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_wfps_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_wfps=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_wfps_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_wfps=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_wfps_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="ibm-cp4a-wfps-operator.v"
                        current_version_wfps=${current_version_wfps#"$prefix_bts"}
                        installed_version_wfps=${installed_version_wfps#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_wfps" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_wfps" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_wfps_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_wfps_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_wfps_sub_name' is v$current_version_wfps."
                            ibm_cp4a_wfps_operator_ready="Yes"
                            break
                        fi
                    done
                fi
            fi

            if [[ "$cp4a_operator_csv_version" == "22.2."* ]]; then
                # Check ADS Operator upgrade done or not
                ibm_cp4a_ads_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-ads-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_ads_operator_flag -ne 0 ]; then
                    ibm_cp4a_ads_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-ads-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_ads_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_ads=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_ads_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_ads=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_ads_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="ibm-ads-operator.v"
                        current_version_ads=${current_version_ads#"$prefix_bts"}
                        installed_version_ads=${installed_version_ads#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_ads" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_ads" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_ads_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_ads_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_ads_sub_name' is v$current_version_ads."
                            ibm_cp4a_ads_operator_ready="Yes"
                            break
                        fi
                    done
                fi
                # Check Content Operator upgrade done or not
                ibm_cp4a_content_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-content-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_content_operator_flag -ne 0 ]; then
                    ibm_cp4a_content_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-content-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_content_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_content=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_content_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_content=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_content_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="ibm-content-operator.v"
                        current_version_content=${current_version_content#"$prefix_bts"}
                        installed_version_content=${installed_version_content#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_content" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_content" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_content_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_content_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_content_sub_name' is v$current_version_content."
                            ibm_cp4a_content_operator_ready="Yes"
                            break
                        fi
                    done
                fi
                # Check PFS Operator upgrade done or not
                ibm_cp4a_pfs_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-pfs-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_pfs_operator_flag -ne 0 ]; then
                    ibm_cp4a_pfs_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-pfs-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_pfs_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_pfs=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_pfs_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_pfs=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_pfs_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="ibm-pfs-operator.v"
                        current_version_pfs=${current_version_pfs#"$prefix_bts"}
                        installed_version_pfs=${installed_version_pfs#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_pfs" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_pfs" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_pfs_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_pfs_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_pfs_sub_name' is v$current_version_pfs."
                            ibm_cp4a_pfs_operator_ready="Yes"
                            break
                        fi
                    done
                fi
                # Check Foundation Operator upgrade done or not
                ibm_cp4a_foundation_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/icp4a-foundation-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_foundation_operator_flag -ne 0 ]; then
                    ibm_cp4a_foundation_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/icp4a-foundation-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_foundation_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_foundation=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_foundation_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_foundation=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_foundation_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="icp4a-foundation-operator.v"
                        current_version_foundation=${current_version_foundation#"$prefix_bts"}
                        installed_version_foundation=${installed_version_foundation#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_foundation" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_foundation" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_foundation_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_foundation_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_foundation_sub_name' is v$current_version_foundation."
                            ibm_cp4a_foundation_operator_ready="Yes"
                            break
                        fi
                    done
                fi

            fi
        elif [[ $ENABLE_PRIVATE_CATALOG -eq 1 ]]; then
            # For shared2dedicated/dedicated2dedicated enable private catalog, we do not switch common serivce catalog source in ibm-common-services project.
            ibm_bts_operator_ready="Yes"
            cloud_native_postgresql_ready="Yes"

            if [[ "$cp4a_operator_csv_version" == "21.3."* || "$cp4a_operator_csv_version" == "22.2."* ]]; then
                ibm_cp4a_wfps_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-cp4a-wfps-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_wfps_operator_flag -ne 0 ]; then
                    ibm_cp4a_wfps_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-cp4a-wfps-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_wfps_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_wfps=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_wfps_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_wfps=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_wfps_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="ibm-cp4a-wfps-operator.v"
                        current_version_wfps=${current_version_wfps#"$prefix_bts"}
                        installed_version_wfps=${installed_version_wfps#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_wfps" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_wfps" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_wfps_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_wfps_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_wfps_sub_name' is v$current_version_wfps."
                            ibm_cp4a_wfps_operator_ready="Yes"
                            break
                        fi
                    done
                fi
            fi

            if [[ "$cp4a_operator_csv_version" == "22.2."* ]]; then
                # Check ADS Operator upgrade done or not
                ibm_cp4a_ads_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-ads-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_ads_operator_flag -ne 0 ]; then
                    ibm_cp4a_ads_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-ads-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_ads_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_ads=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_ads_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_ads=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_ads_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="ibm-ads-operator.v"
                        current_version_ads=${current_version_ads#"$prefix_bts"}
                        installed_version_ads=${installed_version_ads#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_ads" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_ads" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_ads_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_ads_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_ads_sub_name' is v$current_version_ads."
                            ibm_cp4a_ads_operator_ready="Yes"
                            break
                        fi
                    done
                fi
                # Check Content Operator upgrade done or not
                ibm_cp4a_content_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-content-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_content_operator_flag -ne 0 ]; then
                    ibm_cp4a_content_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-content-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_content_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_content=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_content_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_content=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_content_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="ibm-content-operator.v"
                        current_version_content=${current_version_content#"$prefix_bts"}
                        installed_version_content=${installed_version_content#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_content" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_content" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_content_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_content_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_content_sub_name' is v$current_version_content."
                            ibm_cp4a_content_operator_ready="Yes"
                            break
                        fi
                    done
                fi
                # Check PFS Operator upgrade done or not
                ibm_cp4a_pfs_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-pfs-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_pfs_operator_flag -ne 0 ]; then
                    ibm_cp4a_pfs_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/ibm-pfs-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_pfs_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_pfs=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_pfs_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_pfs=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_pfs_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="ibm-pfs-operator.v"
                        current_version_pfs=${current_version_pfs#"$prefix_bts"}
                        installed_version_pfs=${installed_version_pfs#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_pfs" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_pfs" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_pfs_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_pfs_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_pfs_sub_name' is v$current_version_pfs."
                            ibm_cp4a_pfs_operator_ready="Yes"
                            break
                        fi
                    done
                fi
                # Check Foundation Operator upgrade done or not
                ibm_cp4a_foundation_operator_flag=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/icp4a-foundation-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | wc -l)

                if [ $ibm_cp4a_foundation_operator_flag -ne 0 ]; then
                    ibm_cp4a_foundation_sub_name=$(${CLI_CMD} get subscriptions.operators.coreos.com -l=operators.coreos.com/icp4a-foundation-operator.$TEMP_OPERATOR_PROJECT_NAME --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME | awk '{print $1}')

                    info "Checking the version of subscription '$ibm_cp4a_foundation_sub_name' in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                    for ((retry=0;retry<=${maxRetry};retry++)); do
                        current_version_foundation=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_foundation_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                        installed_version_foundation=$(${CLI_CMD} get subscriptions.operators.coreos.com $ibm_cp4a_foundation_sub_name --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                        prefix_bts="icp4a-foundation-operator.v"
                        current_version_foundation=${current_version_foundation#"$prefix_bts"}
                        installed_version_foundation=${installed_version_foundation#"$prefix_bts"}
                        REQUIREDVER_VERSION="${CP4BA_CSV_VERSION//v/}"
                        if [[ (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$current_version_foundation" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") || (! "$(printf '%s\n' "$REQUIREDVER_VERSION" "$installed_version_foundation" | sort -V | head -n1)" = "$REQUIREDVER_VERSION") ]]; then
                            if [[ $retry -eq ${maxRetry} ]]; then
                                info "Timeout Checking for the version of $ibm_cp4a_foundation_sub_name subscription in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                ibm_cp4a_foundation_operator_ready="No"
                                break
                            else
                                sleep 30
                                echo -n "..."
                                continue
                            fi
                        else
                            success "The version of subscription '$ibm_cp4a_foundation_sub_name' is v$current_version_foundation."
                            ibm_cp4a_foundation_operator_ready="Yes"
                            break
                        fi
                    done
                fi

            fi
        fi

        # Migrate CPfs from 3.x to 4.6 for upgrade CP4BA 21.0.3.x/22.0.2 to latest
        if [[ "$cp4a_operator_csv_version" == "21.3."* && (("$ibm_bts_operator_ready" == "Yes" && "$cloud_native_postgresql_ready" == "Yes" )) && "$ibm_cp4a_wfps_operator_ready" == "Yes" ]]; then
            READY_FOR_DIRECT_UPGRADE="Yes"
        elif [[ "$cp4a_operator_csv_version" == "22.2."* && (("$ibm_bts_operator_ready" == "Yes" && "$cloud_native_postgresql_ready" == "Yes" )) && "$ibm_cp4a_wfps_operator_ready" == "Yes" && "$ibm_cp4a_pfs_operator_ready" == "Yes" && "$ibm_cp4a_ads_operator_ready" == "Yes" && "$ibm_cp4a_content_operator_ready" == "Yes" && "$ibm_cp4a_foundation_operator_ready" == "Yes" ]]; then
            READY_FOR_DIRECT_UPGRADE="Yes"
        elif [[ "$cp4a_operator_csv_version" == "23.2."* || "$cp4a_operator_csv_version" == "24."* ]]; then
            READY_FOR_DIRECT_UPGRADE="Yes"
        else
            READY_FOR_DIRECT_UPGRADE="No"
            fail "Prerequisite for upgrade did not complete, exiting..."
        fi

        if [[ $READY_FOR_DIRECT_UPGRADE == "Yes" ]]; then
            # Check IAF operator and remove IAF
            info "Prerequisite for upgrade passed, continue..."
            mkdir -p $UPGRADE_DEPLOYMENT_IAF_LOG_FOLDER >/dev/null 2>&1
            info "Checking IBM Automation Foundation components in the project \"$TEMP_OPERATOR_PROJECT_NAME\"."
            iaf_core_operator_pod_name=$(${CLI_CMD} get pod -l=app.kubernetes.io/name=iaf-core-operator,app.kubernetes.io/instance=iaf-core-operator -n $TEMP_OPERATOR_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep '<none>' | head -1 | awk '{print $1}')
            iaf_operator_pod_name=$(${CLI_CMD} get pod -l=app.kubernetes.io/name=iaf-operator,app.kubernetes.io/instance=iaf-operator -n $TEMP_OPERATOR_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep '<none>' | head -1 | awk '{print $1}')

            # remove IAF components from CP4BA deployment
            # Can be skipped for 24.0.0 to 24.0.1 upgrade since there's no IAF
            if [[ (! -z "$iaf_core_operator_pod_name") || (! -z "$iaf_operator_pod_name") ]]; then
                info "Starting to remove IAF components from CP4BA deployment in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                cp4ba_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')

                if [[ -z $cp4ba_cr_name ]]; then
                    cp4ba_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
                    cr_type="contents.icp4a.ibm.com"
                else
                    cr_type="icp4aclusters.icp4a.ibm.com"
                fi

                if [[ -z $cp4ba_cr_name ]]; then
                    fail "No custom resource found for CP4BA deployment in the project \"$CP4BA_SERVICES_NS\", exit..."
                    exit 1
                else
                    cp4ba_cr_metaname=$(${CLI_CMD} get $cr_type $cp4ba_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
                fi


                if [[ $UPGRADE_MODE == "dedicated2dedicated" ]]; then
                    control_namespace=$(${CLI_CMD} get cm ${COMMON_SERVICES_CM_DEDICATED_NAME} --no-headers --ignore-not-found -n ${COMMON_SERVICES_CM_NAMESPACE} -o jsonpath='{ .data.common-service-maps\.yaml }' | grep  'controlNamespace' | cut -d':' -f2 )
                    control_namespace=$(sed -e 's/^"//' -e 's/"$//' <<<"$control_namespace")
                    control_namespace=$(sed "s/ //g" <<< $control_namespace)

                    if [[ ! -z $control_namespace ]]; then
                        msg "All arguments passed into the script: ${CUR_DIR}/helper/upgrade/remove_iaf.sh $cr_type $cp4ba_cr_metaname $TARGET_PROJECT_NAME $control_namespace \"icp4ba\" \"none\""
                        source ${CUR_DIR}/helper/upgrade/remove_iaf.sh $cr_type $cp4ba_cr_metaname $TARGET_PROJECT_NAME $control_namespace "icp4ba" "none" >/dev/null 2>&1
                    else
                        fail "Not found \"controlNamespace\" in configMap \"${COMMON_SERVICES_CM_DEDICATED_NAME}\" in the project \"${COMMON_SERVICES_CM_NAMESPACE}\""
                        exit 1
                    fi
                elif [[ $UPGRADE_MODE == "shared2dedicated" || $UPGRADE_MODE == "shared2shared" ]]; then
                    msg "All arguments passed into the script: ${CUR_DIR}/helper/upgrade/remove_iaf.sh $cr_type $cp4ba_cr_metaname $TARGET_PROJECT_NAME ibm-common-services \"icp4ba\" \"none\""
                    source ${CUR_DIR}/helper/upgrade/remove_iaf.sh $cr_type $cp4ba_cr_metaname $TARGET_PROJECT_NAME ibm-common-services "icp4ba" "none" >/dev/null 2>&1
                fi
                info "Checking if IAF components be removed from the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                maxRetry=10
                for ((retry=0;retry<=${maxRetry};retry++)); do
                    iaf_core_operator_pod_name=$(${CLI_CMD} get pod -l=app.kubernetes.io/name=iaf-core-operator,app.kubernetes.io/instance=iaf-core-operator -n $TEMP_OPERATOR_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep '<none>' | head -1 | awk '{print $1}')
                    iaf_operator_pod_name=$(${CLI_CMD} get pod -l=app.kubernetes.io/name=iaf-operator,app.kubernetes.io/instance=iaf-operator -n $TEMP_OPERATOR_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers | grep 'Running' | grep '<none>' | head -1 | awk '{print $1}')

                    # if [[ -z $isReadyWebhook || -z $isReadyCertmanager || -z $isReadyCainjector || -z $isReadyCertmanagerOperator ]]; then
                    if [[ (! -z $iaf_core_operator_pod_name) || (! -z $iaf_operator_pod_name) ]]; then
                        if [[ $retry -eq ${maxRetry} ]]; then
                            printf "\n"
                            warning "Timeout waiting for IBM Automation Foundation be removed from the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                            echo -e "\x1B[1mRemove IAF manually using the command: \"${CUR_DIR}/helper/upgrade/remove_iaf.sh $cr_type $cp4ba_cr_metaname $TARGET_PROJECT_NAME $control_namespace \"icp4ba\" \"none\"\"\x1B[0m"
                            exit 1
                        else
                            sleep 30
                            echo -n "..."
                            continue
                        fi
                    else
                        success "IBM Automation Foundation was removed successfully!"
                        break
                    fi
                done
            else
                success "IBM Automation Foundation components already were removed from the project \"$TEMP_OPERATOR_PROJECT_NAME\"!"
            fi
            # # Sourcing the message.sh to use the displayUpgradeOperatorMessage
            source ${CUR_DIR}/helper/messages.sh $COMMON_SERVICES_SCRIPT_FOLDER
             # Can be skipped for 24.0.0 to 24.0.1 upgrade since the csv version will NOT be 21.3.x or 22.2.x.  Only execute when allow_direct_upgrade flag is 1 
            if [[ ("$cp4a_operator_csv_version" == "21.3."* || "$cp4a_operator_csv_version" == "22.2."* ) && ("$ALLOW_DIRECT_UPGRADE" == 1) ]]; then
                info "Starting to migrate IBM Cloud Pak foundational services from 3.x to $CS_OPERATOR_VERSION"

                # Check if without option --enable-private-catalog, the catalog is in target project, set the private catalog as default.
                info "Checking ibm-cp4a-operator-catalog catalog source is global or private namespace scoped"
                if [[ $ENABLE_PRIVATE_CATALOG -eq 0 ]]; then
                    if ${CLI_CMD} get catalogsource -n $TARGET_PROJECT_NAME --no-headers --ignore-not-found | grep ibm-cp4a-operator-catalog >/dev/null 2>&1; then
                        ENABLE_PRIVATE_CATALOG=1
                    else
                        info "ibm-cp4a-operator-catalog catalog source not found under the target project \"$TARGET_PROJECT_NAME\""
                    fi
                fi

                if [[ $UPGRADE_MODE == "dedicated2dedicated" && $ENABLE_PRIVATE_CATALOG -eq 1 ]]; then
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TARGET_PROJECT_NAME --cert-manager-source ibm-cert-manager-catalog --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept --enable-private-catalog"
                    # switch catalog from GCN to private
                    $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TARGET_PROJECT_NAME --cert-manager-source ibm-cert-manager-catalog --enable-licensing --yq "$CPFS_YQ_PATH" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept --enable-private-catalog
                    if [ $? -ne 0 ]; then                        
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TARGET_PROJECT_NAME --cert-manager-source ibm-cert-manager-catalog --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept --enable-private-catalog"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                elif [[ $UPGRADE_MODE == "dedicated2dedicated" && $ENABLE_PRIVATE_CATALOG -eq 0 ]]; then
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TARGET_PROJECT_NAME --cert-manager-source ibm-cert-manager-catalog --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept"
                    # keep GCN catalog
                    $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TARGET_PROJECT_NAME --cert-manager-source ibm-cert-manager-catalog --enable-licensing --yq "$CPFS_YQ_PATH" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TARGET_PROJECT_NAME --cert-manager-source ibm-cert-manager-catalog --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                elif [[ $UPGRADE_MODE == "shared2dedicated" && $ENABLE_PRIVATE_CATALOG -eq 1 && $ALL_NAMESPACE_FLAG == "No" ]]; then
                    ## Apply workaround for https://jsw.ibm.com/browse/DBACLD-137719 before start migration CPfs
                    ## Shutdown ibm-bts-operator-controller-manager in ibm-common-services project
                    bts_operator_name=$(${CLI_CMD} get deployment ibm-bts-operator-controller-manager --no-headers --ignore-not-found -n ibm-common-services -o name)
                    if [[ ! -z $bts_operator_name ]]; then
                        ${CLI_CMD} scale --replicas=0 deployment ibm-bts-operator-controller-manager -n ibm-common-services >/dev/null 2>&1
                        if [[ $? -ne 0 ]]; then
                            warning "Failed to scale down the ibm-bts-operator-controller-manager operator in the project \"ibm-common-services\". Scale down ibm-bts-operator-controller-manager operator manually."
                        fi
                    fi
                    # switch catalog from GCN to private
                    # shared2dedicated dose not support All namespace
                    ## Isolated CP4BA instance in $TARGET_PROJECT_NAME namespace
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/isolate.sh --original-cs-ns ibm-common-services --control-ns cs-control --excluded-ns $TARGET_PROJECT_NAME --yq \"$CPFS_YQ_PATH\""
                    $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/isolate.sh --original-cs-ns ibm-common-services --control-ns cs-control --excluded-ns $TARGET_PROJECT_NAME --yq "$CPFS_YQ_PATH"
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/isolate.sh --original-cs-ns ibm-common-services --control-ns cs-control --excluded-ns $TARGET_PROJECT_NAME --yq \"$CPFS_YQ_PATH\""
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi

                    # Additionally, we would check if cs-control namespace exists.
                    isProjExists=`${CLI_CMD} get project $DEDICATED_CS_PROJECT --no-headers --ignore-not-found | wc -l`  >/dev/null 2>&1
                    if [ $isProjExists -eq 1 ] ; then
                        # If it exists, we will deploy the same ibm-licensing-catalog into cs-control namespace.
                        if [[ $machine == "Linux" ]]; then
                            TMP_LICENSING_OLM_CATALOG=$(mktemp --suffix=.yaml)
                        elif [[ $machine == "Mac" ]]; then
                            TMP_LICENSING_OLM_CATALOG=$(mktemp -t licensing_olm_catalog).yaml
                        fi
                        start_num="# IBM License Manager"
                        end_num="interval: 45m"

                        reading_section=false

                        while IFS= read -r line; do
                            if [[ "$line" == *"$start_num"* ]]; then
                                reading_section=true
                            fi

                            if $reading_section; then
                                echo "$line" >> "$TMP_LICENSING_OLM_CATALOG"
                            fi

                            if [[ "$line" == *"$end_num"* ]]; then
                                reading_section=false
                            fi
                        done < "${OLM_CATALOG}"

                        # replace openshift-marketplace for ibm-licensing-catalog with cs-control
                        ${SED_COMMAND} "/name: ibm-licensing-catalog/{n;s/namespace: .*/namespace: \"$DEDICATED_CS_PROJECT\"/;}" ${TMP_LICENSING_OLM_CATALOG}

                        ${CLI_CMD} apply -f $TMP_LICENSING_OLM_CATALOG >/dev/null 2>&1
                        if [ $? -eq 0 ]; then
                            echo "Created IBM License Manager Catalog source in project \"$DEDICATED_CS_PROJECT\"!"
                        else
                            echo "Generic Operator catalog source update failed"
                            exit 1
                        fi
                        rm -rf $TMP_LICENSING_OLM_CATALOG >/dev/null 2>&1
                    fi

                    ## Migrate Cert-Manager and Licensing service from v3.x to v4.x
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --operator-namespace ibm-common-services --enable-licensing --license-accept --enable-private-catalog --yq \"$CPFS_YQ_PATH\" -v 1"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --operator-namespace ibm-common-services --enable-licensing --license-accept --enable-private-catalog --yq "$CPFS_YQ_PATH" -v 1
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --operator-namespace ibm-common-services --enable-licensing --license-accept --enable-private-catalog --yq \"$CPFS_YQ_PATH\" -v 1"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    ## Preload data
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/preload_data.sh --original-cs-ns ibm-common-services --services-ns $TARGET_PROJECT_NAME --yq \"$CPFS_YQ_PATH\""
                    $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/preload_data.sh --original-cs-ns ibm-common-services --services-ns $TARGET_PROJECT_NAME --yq "$CPFS_YQ_PATH"
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/preload_data.sh --original-cs-ns ibm-common-services --services-ns $TARGET_PROJECT_NAME --yq \"$CPFS_YQ_PATH\""
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    ## Fresh install foundational services version 4.x
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --enable-private-catalog --license-accept -s $CS_CATALOG_VERSION -c $CS_CHANNEL_VERSION --yq \"$CPFS_YQ_PATH\" -v 1"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --enable-private-catalog --license-accept -s $CS_CATALOG_VERSION -c $CS_CHANNEL_VERSION --yq "$CPFS_YQ_PATH" -v 1
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --enable-private-catalog --license-accept -s $CS_CATALOG_VERSION -c $CS_CHANNEL_VERSION --yq \"$CPFS_YQ_PATH\" -v 1"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                elif [[ $UPGRADE_MODE == "shared2dedicated" && $ENABLE_PRIVATE_CATALOG -eq 0 && $ALL_NAMESPACE_FLAG == "No" ]]; then
                    ## Apply workaround for https://jsw.ibm.com/browse/DBACLD-137719 before start migration CPfs
                    ## Shutdown ibm-bts-operator-controller-manager in ibm-common-services project
                    bts_operator_name=$(${CLI_CMD} get deployment ibm-bts-operator-controller-manager --no-headers --ignore-not-found -n ibm-common-services -o name)
                    if [[ ! -z $bts_operator_name ]]; then
                        ${CLI_CMD} scale --replicas=0 deployment ibm-bts-operator-controller-manager -n ibm-common-services >/dev/null 2>&1
                        if [[ $? -ne 0 ]]; then
                            warning "Failed to scale down the ibm-bts-operator-controller-manager operator in the project \"ibm-common-services\". Scale down ibm-bts-operator-controller-manager operator manually."
                        fi
                    fi
                    # keep GCN catalog
                    # shared2dedicated dose not support All namespace
                    ## Isolated CP4BA instance in $TARGET_PROJECT_NAME namespace
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/isolate.sh --original-cs-ns ibm-common-services --control-ns cs-control --excluded-ns $TARGET_PROJECT_NAME --yq \"$CPFS_YQ_PATH\""
                    $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/isolate.sh --original-cs-ns ibm-common-services --control-ns cs-control --excluded-ns $TARGET_PROJECT_NAME --yq "$CPFS_YQ_PATH"
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/isolate.sh --original-cs-ns ibm-common-services --control-ns cs-control --excluded-ns $TARGET_PROJECT_NAME --yq \"$CPFS_YQ_PATH\""
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi

                    # Additionally, we would check if cs-control namespace exists.
                    isProjExists=`${CLI_CMD} get project $DEDICATED_CS_PROJECT --no-headers --ignore-not-found | wc -l`  >/dev/null 2>&1
                    if [ $isProjExists -eq 1 ] ; then
                        # If it exists, we will deploy the same ibm-licensing-catalog into cs-control namespace.
                        if [[ $machine == "Linux" ]]; then
                            TMP_LICENSING_OLM_CATALOG=$(mktemp --suffix=.yaml)
                        elif [[ $machine == "Mac" ]]; then
                            TMP_LICENSING_OLM_CATALOG=$(mktemp -t licensing_olm_catalog).yaml
                        fi
                        start_num="# IBM License Manager"
                        end_num="interval: 45m"

                        reading_section=false

                        while IFS= read -r line; do
                            if [[ "$line" == *"$start_num"* ]]; then
                                reading_section=true
                            fi

                            if $reading_section; then
                                echo "$line" >> "$TMP_LICENSING_OLM_CATALOG"
                            fi

                            if [[ "$line" == *"$end_num"* ]]; then
                                reading_section=false
                            fi
                        done < "${OLM_CATALOG}"

                        # replace openshift-marketplace for ibm-licensing-catalog with cs-control
                        ${SED_COMMAND} "/name: ibm-licensing-catalog/{n;s/namespace: .*/namespace: \"$DEDICATED_CS_PROJECT\"/;}" ${TMP_LICENSING_OLM_CATALOG}

                        ${CLI_CMD} apply -f $TMP_LICENSING_OLM_CATALOG >/dev/null 2>&1
                        if [ $? -eq 0 ]; then
                            echo "Created IBM License Manager Catalog source in project \"$DEDICATED_CS_PROJECT\"!"
                        else
                            echo "Generic Operator catalog source update failed"
                            exit 1
                        fi
                        rm -rf $TMP_LICENSING_OLM_CATALOG >/dev/null 2>&1
                    fi

                    ## Migrate Cert-Manager and Licensing service from v3.x to v4.x
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --operator-namespace ibm-common-services --enable-licensing --license-accept --yq \"$CPFS_YQ_PATH\" -v 1"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --operator-namespace ibm-common-services --enable-licensing --license-accept --yq "$CPFS_YQ_PATH" -v 1
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --operator-namespace ibm-common-services --enable-licensing --license-accept --yq \"$CPFS_YQ_PATH\" -v 1"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    ## Preload data
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/preload_data.sh --original-cs-ns ibm-common-services --services-ns $TARGET_PROJECT_NAME --yq \"$CPFS_YQ_PATH\""
                    $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/preload_data.sh --original-cs-ns ibm-common-services --services-ns $TARGET_PROJECT_NAME --yq "$CPFS_YQ_PATH"
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_PARENT_FOLDER/preload_data.sh --original-cs-ns ibm-common-services --services-ns $TARGET_PROJECT_NAME --yq \"$CPFS_YQ_PATH\""
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    ## Fresh install foundational services version 4.x
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --license-accept -s $CS_CATALOG_VERSION -c $CS_CHANNEL_VERSION --yq \"$CPFS_YQ_PATH\" -v 1"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --license-accept -s $CS_CATALOG_VERSION -c $CS_CHANNEL_VERSION --yq "$CPFS_YQ_PATH" -v 1
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --operator-namespace $TARGET_PROJECT_NAME --license-accept -s $CS_CATALOG_VERSION -c $CS_CHANNEL_VERSION --yq \"$CPFS_YQ_PATH\" -v 1"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                elif [[ $UPGRADE_MODE == "shared2shared" && $ALL_NAMESPACE_FLAG == "Yes" ]]; then
                    # keep GCN catalog
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace openshift-operators --cert-manager-source ibm-cert-manager-catalog --licensing-source ibm-licensing-catalog --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept"
                    $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace openshift-operators --cert-manager-source ibm-cert-manager-catalog --licensing-source ibm-licensing-catalog --enable-licensing --yq "$CPFS_YQ_PATH" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace openshift-operators --cert-manager-source ibm-cert-manager-catalog --licensing-source ibm-licensing-catalog --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    # workaround for https://github.ibm.com/IBMPrivateCloud/roadmap/issues/63676
                    cs_namespace="ibm-common-services"
                    info "Removing ibm-events-operator from the project \"$cs_namespace\"."
                    ${CLI_CMD} delete csv -l operators.coreos.com/ibm-events-operator.$cs_namespace='' -n $cs_namespace >/dev/null 2>&1
                    if [ $? -eq 0 ]; then
                        success "Removed ibm-events-operator from the project \"$cs_namespace\"."
                    else
                        fail "Faild to remove ibm-events-operator from the project \"$cs_namespace\"."
                    fi
                elif [[ $UPGRADE_MODE == "shared2shared" && $ALL_NAMESPACE_FLAG == "No" ]]; then
                    # keep GCN catalog
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace ibm-common-services --services-namespace ibm-common-services --cert-manager-source ibm-cert-manager-catalog --licensing-source ibm-licensing-catalog --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept"
                    $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace ibm-common-services --services-namespace ibm-common-services --cert-manager-source ibm-cert-manager-catalog --licensing-source ibm-licensing-catalog --enable-licensing --yq "$CPFS_YQ_PATH" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/migrate_tenant.sh --operator-namespace ibm-common-services --services-namespace ibm-common-services --cert-manager-source ibm-cert-manager-catalog --licensing-source ibm-licensing-catalog --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --license-accept"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                fi
            elif [[ "$cp4a_operator_csv_version" == "23.2."* || "$cp4a_operator_csv_version" == "24.0."* || "$cp4a_operator_csv_version" == "24.1."* ]]; then
                info "Starting to upgrade IBM Cloud Pak foundational services to $CS_OPERATOR_VERSION"
                # Check if without option --enable-private-catalog, the catalog is in target project, set the private catalog as default.
                info "Checking ibm-cp4a-operator-catalog catalog source is global or private namespace scoped"
                if [[ $ENABLE_PRIVATE_CATALOG -eq 0 ]]; then
                    if ${CLI_CMD} get catalogsource -n $TARGET_PROJECT_NAME --no-headers --ignore-not-found | grep ibm-cp4a-operator-catalog >/dev/null 2>&1; then
                        ENABLE_PRIVATE_CATALOG=1
                    else
                        info "Not found ibm-cp4a-operator-catalog catalog source under target project \"$TARGET_PROJECT_NAME\""
                    fi
                fi
                # This is still a valid scenario in 24.0.0 upgrading to 24.0.1.
                if [[ $UPGRADE_MODE == "dedicated2dedicated" && $ENABLE_PRIVATE_CATALOG -eq 1 ]]; then
                    # Additionally, we would check if cs-control namespace exists.
                    isProjExists=`${CLI_CMD} get project $DEDICATED_CS_PROJECT --no-headers --ignore-not-found | wc -l`  >/dev/null 2>&1
                    if [ $isProjExists -eq 1 ] ; then
                        # If it exists, we will deploy the same ibm-licensing-catalog into cs-control namespace.
                        if [[ $machine == "Linux" ]]; then
                            TMP_LICENSING_OLM_CATALOG=$(mktemp --suffix=.yaml)
                        elif [[ $machine == "Mac" ]]; then
                            TMP_LICENSING_OLM_CATALOG=$(mktemp -t licensing_olm_catalog).yaml
                        fi
                        start_num="# IBM License Manager"
                        end_num="interval: 45m"

                        reading_section=false

                        while IFS= read -r line; do
                            if [[ "$line" == *"$start_num"* ]]; then
                                reading_section=true
                            fi

                            if $reading_section; then
                                echo "$line" >> "$TMP_LICENSING_OLM_CATALOG"
                            fi

                            if [[ "$line" == *"$end_num"* ]]; then
                                reading_section=false
                            fi
                        done < "${OLM_CATALOG}"

                        # replace openshift-marketplace for ibm-licensing-catalog with cs-control
                        ${SED_COMMAND} "/name: ibm-licensing-catalog/{n;s/namespace: .*/namespace: \"$DEDICATED_CS_PROJECT\"/;}" ${TMP_LICENSING_OLM_CATALOG}

                        ${CLI_CMD} apply -f $TMP_LICENSING_OLM_CATALOG >/dev/null 2>&1
                        if [ $? -eq 0 ]; then
                            echo "Created IBM License Manager Catalog source in project \"$DEDICATED_CS_PROJECT\"!"
                        else
                            echo "Generic Operator catalog source update failed"
                            exit 1
                        fi
                        rm -rf $TMP_LICENSING_OLM_CATALOG >/dev/null 2>&1
                    fi

                    # Upgrading Cert-Manager and Licensing Service.
                    # This command still valid when upgrading from 24.0.0 to 24.0.1
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --enable-private-catalog --yq \"$CPFS_YQ_PATH\" -c $CERT_LICENSE_CHANNEL_VERSION"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --enable-private-catalog --yq "$CPFS_YQ_PATH" -c $CERT_LICENSE_CHANNEL_VERSION
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --enable-private-catalog --yq \"$CPFS_YQ_PATH\" -c $CERT_LICENSE_CHANNEL_VERSION"
                        # #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    # set --service-namespace property to CP4BA_SERVICES_NS when it's seperation of duty
                    if [[ $SEPARATE_OPERAND_FLAG == "Yes" ]]; then
                        TMP_SERVICES_NAMESPACE=$CP4BA_SERVICES_NS
                    # set --service-namespace property to TARGET_PROJECT_NAME when it's not seperation of duty
                    else
                        TMP_SERVICES_NAMESPACE=$TARGET_PROJECT_NAME
                    fi
                    # switch catalog from GCN to private when it's seperation of duty.
                    # Upgrading CPFS
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TMP_SERVICES_NAMESPACE --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --enable-private-catalog -v 1"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TMP_SERVICES_NAMESPACE --yq "$CPFS_YQ_PATH" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --enable-private-catalog -v 1
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TMP_SERVICES_NAMESPACE --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION --enable-private-catalog -v 1"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME  $cp4a_operator_csv_version
                        exit 1
                    fi
                # This is still a valid scenario in 24.0.0 upgrading to 24.0.1.
                # Why does the setup_singleton and setup_tennant looks the same w/ the above `if` condition?
                elif [[ $UPGRADE_MODE == "dedicated2dedicated" && $ENABLE_PRIVATE_CATALOG -eq 0 ]]; then
                    # Upgrading Cert-Manager and Licensing Service
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CERT_LICENSE_CHANNEL_VERSION"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq "$CPFS_YQ_PATH" -c $CERT_LICENSE_CHANNEL_VERSION
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CERT_LICENSE_CHANNEL_VERSION"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    
                    # set --service-namespace property to CP4BA_SERVICES_NS when it's seperation of duty
                    if [[ $SEPARATE_OPERAND_FLAG == "Yes" ]]; then
                        TMP_SERVICES_NAMESPACE=$CP4BA_SERVICES_NS
                    
                    # set --service-namespace property to TARGET_PROJECT_NAME when it's not seperation of duty
                    else
                        TMP_SERVICES_NAMESPACE=$TARGET_PROJECT_NAME
                    fi
                    # keep GCN catalog
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TMP_SERVICES_NAMESPACE --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TMP_SERVICES_NAMESPACE --yq "$CPFS_YQ_PATH" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE= "Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace $TMP_SERVICES_NAMESPACE --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
             
                # This is still a valid scenario in 24.0.0 upgrading to 24.0.1.
                elif [[ $UPGRADE_MODE == "shared2shared" && $ALL_NAMESPACE_FLAG == "Yes" ]]; then
                    # It is not recommended to install 23.0.2 in all namespace. but script keep coverage for it
                    # Upgrading Cert-Manager and Licensing Service
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CERT_LICENSE_CHANNEL_VERSION"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq "$CPFS_YQ_PATH" -c $CERT_LICENSE_CHANNEL_VERSION
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CERT_LICENSE_CHANNEL_VERSION"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    # workaround for https://github.ibm.com/IBMPrivateCloud/roadmap/issues/63676
                    cs_namespace="ibm-common-services"
                    info "Removing ibm-events-operator from the project \"$cs_namespace\"."
                    ${CLI_CMD} delete csv -l operators.coreos.com/ibm-events-operator.$cs_namespace='' -n $cs_namespace >/dev/null 2>&1
                    if [ $? -eq 0 ]; then
                        success "Removed ibm-events-operator from the project \"$cs_namespace\"."
                    else
                        fail "Faild to remove ibm-events-operator from the project \"$cs_namespace\"."
                    fi
                    # keep GCN catalog
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace openshift-operators --services-namespace ibm-common-services --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace openshift-operators --services-namespace ibm-common-services --yq "$CPFS_YQ_PATH" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace openshift-operators --services-namespace ibm-common-services --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                # This is not a valid scenario unless allow_direct_upgrade == 1
                elif [[ $UPGRADE_MODE == "shared2shared" && $ALL_NAMESPACE_FLAG == "No" && "$ALLOW_DIRECT_UPGRADE" == 1 ]]; then
                    # Upgrading Cert-Manager and Licensing Service
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CERT_LICENSE_CHANNEL_VERSION"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq "$CPFS_YQ_PATH" -c $CERT_LICENSE_CHANNEL_VERSION
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_singleton.sh --license-accept --enable-licensing --yq \"$CPFS_YQ_PATH\" -c $CERT_LICENSE_CHANNEL_VERSION"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                    # keep GCN catalog
                    msg "All arguments passed into the CPfs script: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace ibm-common-services --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1"
                    $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace ibm-common-services --yq "$CPFS_YQ_PATH" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1
                    if [ $? -ne 0 ]; then
                        TMP_MESSAGE="Failed to execute command: $COMMON_SERVICES_SCRIPT_FOLDER/setup_tenant.sh --license-accept --enable-licensing --operator-namespace $TARGET_PROJECT_NAME --services-namespace ibm-common-services --yq \"$CPFS_YQ_PATH\" -c $CS_CHANNEL_VERSION -s $CS_CATALOG_VERSION -v 1"
                        #source ${CUR_DIR}/helper/messages.sh
                        displayUpgradeOperatorMessage "$TMP_MESSAGE" $TARGET_PROJECT_NAME $cp4a_operator_csv_version
                        exit 1
                    fi
                fi
            fi
        fi

        # Check IBM Cloud Pak foundational services Operator $CS_OPERATOR_VERSION
        maxRetry=30
        echo "****************************************************************************"
        info "Checking for IBM Cloud Pak foundational operator pod initialization"
        for ((retry=0;retry<=${maxRetry};retry++)); do
            isReady=$(${CLI_CMD} get csv ibm-common-service-operator.$CS_OPERATOR_VERSION --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o jsonpath='{.status.phase}')
            # isReady=$(${CLI_CMD} exec $cpe_pod_name -c ${meta_name}-cpe-deploy -n $upgrade_operator_project_name -- cat /opt/ibm/version.txt |grep -F "P8 Content Platform Engine $CP4BA_RELEASE_BASE")
            if [[ $isReady != "Succeeded" ]]; then
                if [[ $retry -eq ${maxRetry} ]]; then
                printf "\n"
                warning "Timeout waiting for IBM Cloud Pak foundational operator to start"
                echo -e "\x1B[1mCheck the status of Pod by issuing the following command:\x1B[0m"
                echo "${CLI_CMD} describe pod $(${CLI_CMD} get pod -n $TEMP_OPERATOR_PROJECT_NAME|grep ibm-common-service-operator|awk '{print $1}') -n $TEMP_OPERATOR_PROJECT_NAME"
                printf "\n"
                echo -e "\x1B[1mCheck the status of ReplicaSet by issuing the following command:\x1B[0m"
                echo "${CLI_CMD} describe rs $(${CLI_CMD} get rs -n $TEMP_OPERATOR_PROJECT_NAME|grep ibm-common-service-operator|awk '{print $1}') -n $TEMP_OPERATOR_PROJECT_NAME"
                printf "\n"
                exit 1
                else
                sleep 30
                echo -n "..."
                continue
                fi
            elif [[ $isReady == "Succeeded" ]]; then
                pod_name=$(${CLI_CMD} get pod -l=name=ibm-common-service-operator -n $TEMP_OPERATOR_PROJECT_NAME -o 'custom-columns=NAME:.metadata.name,PHASE:.status.phase,READY:.status.containerStatuses[0].ready,DELETED:.metadata.deletionTimestamp' --no-headers --ignore-not-found | grep 'Running' | grep 'true' | grep '<none>' | head -1 | awk '{print $1}')
                if [ -z $pod_name ]; then
                    error "IBM Cloud Pak foundational Operator pod is NOT running"
                    CHECK_CP4BA_OPERATOR_RESULT=( "${CHECK_CP4BA_OPERATOR_RESULT[@]}" "FAIL" )
                    break
                else
                    success "IBM Cloud Pak foundational Operator is running"
                    info "Pod: $pod_name"
                    CHECK_CP4BA_OPERATOR_RESULT=( "${CHECK_CP4BA_OPERATOR_RESULT[@]}" "PASS" )
                    break
                fi
            fi
        done
        echo "****************************************************************************"

        # Checking CP4BA operator CSV
        # change this value for $CP4BA_RELEASE_BASE-IFIX
        sub_inst_list=$(${CLI_CMD} get subscriptions.operators.coreos.com -n $TEMP_OPERATOR_PROJECT_NAME|grep ibm-cp4a-operator-catalog|awk '{if(NR>0){if(NR==1){ arr=$1; }else{ arr=arr" "$1; }} } END{ print arr }')
        if [[ -z $sub_inst_list ]]; then
            fail "No existing CP4BA subscriptions (version $CP4BA_CSV_VERSION) found! Exiting ..."
            exit 1
        fi
        sub_array=($sub_inst_list)
        target_csv_version=${CP4BA_CSV_VERSION//v/}
        for i in ${!sub_array[@]}; do
            if [[ ! -z "${sub_array[i]}" ]]; then
                if [[ ${sub_array[i]} = ibm-cp4a-operator* || ${sub_array[i]} = ibm-cp4a-wfps-operator* || ${sub_array[i]} = ibm-content-operator* || ${sub_array[i]} = icp4a-foundation-operator* || ${sub_array[i]} = ibm-pfs-operator* || ${sub_array[i]} = ibm-ads-operator* || ${sub_array[i]} = ibm-dpe-operator* || ${sub_array[i]} = ibm-odm-operator* || ${sub_array[i]} = ibm-insights-engine-operator* || ${sub_array[i]} = ibm-workflow-operator* ]]; then
                info "Checking the channel of subscription '${sub_array[i]}'!"
                currentChannel=$(${CLI_CMD} get subscriptions.operators.coreos.com ${sub_array[i]} -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.spec.channel}') >/dev/null 2>&1
                    if [[ "$currentChannel" == "$CP4BA_CHANNEL_VERSION" ]]
                    then
                        success "The channel of subscription '${sub_array[i]}' is $currentChannel!"
                        printf "\n"
                        maxRetry=40
                        info "Waiting for the \"${sub_array[i]}\" subscription be upgraded to the ClusterServiceVersions(CSV) \"v$target_csv_version\""
                        for ((retry=0;retry<=${maxRetry};retry++)); do
                            current_version=$(${CLI_CMD} get subscriptions.operators.coreos.com ${sub_array[i]} --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.currentCSV}') >/dev/null 2>&1
                            installed_version=$(${CLI_CMD} get subscriptions.operators.coreos.com ${sub_array[i]} --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o 'jsonpath={.status.installedCSV}') >/dev/null 2>&1
                            if [[ -z $current_version || -z $installed_version ]]; then
                                error "Failed to retrieve installed or current CSV. Aborting the upgrade procedure. Check the subscription status of ${sub_array[i]}."
                                exit 1
                            fi
                            case "${sub_array[i]}" in
                            "ibm-cp4a-operator"*)
                                prefix_sub="ibm-cp4a-operator.v"
                                ;;
                            "ibm-cp4a-wfps-operator"*)
                                prefix_sub="ibm-cp4a-wfps-operator.v"
                                ;;
                            "ibm-content-operator"*)
                                prefix_sub="ibm-content-operator.v"
                                ;;
                            "icp4a-foundation-operator"*)
                                prefix_sub="icp4a-foundation-operator.v"
                                ;;
                            "ibm-pfs-operator"*)
                                prefix_sub="ibm-pfs-operator.v"
                                ;;
                            "ibm-ads-operator"*)
                                prefix_sub="ibm-ads-operator.v"
                                ;;
                            "ibm-dpe-operator"*)
                                prefix_sub="ibm-dpe-operator.v"
                                ;;
                            "ibm-odm-operator"*)
                                prefix_sub="ibm-odm-operator.v"
                                ;;
                            "ibm-insights-engine-operator"*)
                                prefix_sub="ibm-insights-engine-operator.v"
                                ;;
                            "ibm-workflow-operator"*)
                                prefix_sub="ibm-workflow-operator.v"
                                ;;
                            esac

                            current_version=${current_version#"$prefix_sub"}
                            installed_version=${installed_version#"$prefix_sub"}
                            if [[ $current_version != $installed_version || $current_version != $target_csv_version || $installed_version != $target_csv_version ]]; then
                                approval_mode=$(${CLI_CMD} get subscription.operators.coreos.com ${sub_array[i]} --no-headers --ignore-not-found -n $TEMP_OPERATOR_PROJECT_NAME -o jsonpath={.spec.installPlanApproval})
                                if [[ $approval_mode == "Manual" ]]; then
                                    error "${sub_array[i]} subscription is set to Manual Approval mode. Approve the installPlan to proceed with the upgrade."
                                    exit 1
                                fi
                                if [[ $retry -eq ${maxRetry} ]]; then
                                    warning "Timeout waiting for upgrading \"${sub_array[i]}\" subscription from ${installed_version} to ${target_csv_version} in the project \"$TEMP_OPERATOR_PROJECT_NAME\""
                                    break
                                else
                                    sleep 10
                                    echo -n "..."
                                    continue
                                fi
                            else
                                success "ClusterServiceVersions ${installed_version} is now the latest available version in ${currentChannel} channel."
                                break
                            fi
                        done

                    else
                        fail "Failed to update the channel of subscription '${sub_array[i]}' to $CP4BA_CHANNEL_VERSION! Exiting now..."
                        exit 1
                    fi
                fi
            else
                fail "No subscription found for '${sub_array[i]}'! Exiting now..."
                exit 1
            fi
        done
        success "Completed to check the channel of subscription for CP4BA operators"

        # DBACLD-166239 -> Update EDB configmap ibm-zen-metastore-edb-cm to add new parameters with CPFS 4.10 or later by calling patch_edb_configmap()
        patch_edb_configmap $TMP_SERVICES_NAMESPACE

        # DBACLD-168537: need to re-create {{meta.name}}-fncm-custom-ssl-secret to add CSS DNSName (in case they are missing from previous deployment) which will be included in FNCM's keystores
        cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        if [[ -z $cp4ba_cr_name ]]; then
            cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        fi
        fncm_custom_ssl_secret=$(${CLI_CMD} get secret --no-headers --ignore-not-found ${cr_name}-fncm-custom-ssl-secret -n $CP4BA_SERVICES_NS | awk '{print $1}') 
        if [[ -z $fncm_custom_ssl_secret ]]; then
            info "${cr_name}-fncm-custom-ssl-secret is not found."
        else
            info "Found ${cr_name}-fncm-custom-ssl-secret and the script will now delete the secret."
            ${CLI_CMD} delete secret ${cr_name}-fncm-custom-ssl-secret -n $CP4BA_SERVICES_NS
        fi
        # DBACLD-178263: need to re-create {{meta.name}}-ban-custom-ssl-secret to add localhost to Navigator's keystore
        ban_custom_ssl_secret=$(${CLI_CMD} get secret --no-headers --ignore-not-found ${cr_name}-ban-custom-ssl-secret -n $CP4BA_SERVICES_NS | awk '{print $1}') 
        if [[ -z $ban_custom_ssl_secret ]]; then
            info "${cr_name}-ban-custom-ssl-secret is not found."
        else
            info "Found ${cr_name}-ban-custom-ssl-secret and the script will now delete the secret."
            ${CLI_CMD} delete secret ${cr_name}-ban-custom-ssl-secret -n $CP4BA_SERVICES_NS
        fi


        # shutdown CP4BA operators and show tips for [NEXT ACTION]
        if [[ ! ("$cp4ba_original_csv_ver_for_upgrade_script" == "$CP4BA_RELEASE_BASE_MAJOR_VERSION"*) ]]; then
            info "Shutdown CP4BA Operators before upgrade CP4BA capabilities."
            shutdown_operator $TEMP_OPERATOR_PROJECT_NAME
            printf "\n"
            if [[ $CONTENT_CR_EXIST == "Yes" || (" ${EXISTING_PATTERN_ARR[@]} " =~ "content") || ((" ${EXISTING_PATTERN_ARR[@]} " =~ "workflow") && (! " ${EXISTING_PATTERN_ARR[@]} " =~ "workflow-process-service")) || (" ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing") || (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "baw_authoring") || (" ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "ae_data_persistence") ]]; then
                if [[ $UPGRADE_MODE == "shared2dedicated" ]]; then
                    echo -e "\x1B[33;5m[ATTENTION]: \x1B[0m${RED_TEXT}You DO NOT need to run \"./cp4a-pre-upgrade-and-post-upgrade-optional.sh\" in \"pre-upgrade\" mode when migrate IBM Cloud Pak foundational services from \"Cluster-scoped\" to \"Namespace-scoped\"!${RESET_TEXT}"
                    prompt_press_any_key_to_continue
                fi
            fi
            printf "\n"
            ## -- https://jsw.ibm.com/browse/DBACLD-174848 - <To fix the incorrect script path while running the deployment script in the upgrade mode>
            CUR_DIR=$(realpath "$(dirname "${BASH_SOURCE[0]}")")
            echo "${YELLOW_TEXT}[ATTENTION]: ${RESET_TEXT}"
            echo "${YELLOW_TEXT}  - All CP4BA operators have already been shut down by the script.${RESET_TEXT}"
            echo "${YELLOW_TEXT}  - All CP4BA operators will start up automatically when running [upgradeDeploymentStatus] mode of the cp4a-deployment.sh script.${RESET_TEXT}"
            printf "\n"
            echo "${YELLOW_TEXT}[NEXT ACTIONS]:${RESET_TEXT}"
            step_num=1
            echo "  - STEP ${step_num} ${YELLOW_TEXT}(Optional)${RESET_TEXT}: You can run ${GREEN_TEXT}\"${CUR_DIR}/cp4a-deployment.sh -m upgradeOperatorStatus -n $TARGET_PROJECT_NAME\"${RESET_TEXT} to check whether the upgrade of the CP4BA operator and its dependencies is successful."
            step_num=$((step_num + 1))

            if [[ $css_flag == "true" || " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "css" ]]; then
                echo "  - STEP ${step_num} ${RED_TEXT}(Required)${RESET_TEXT}: You have Content Search Services (CSS) installed. Make sure you stop the IBM Content Search Services index dispatcher. Refer to the FileNet P8 Platform Documentation for more details."
                echo "    ${YELLOW_TEXT}* Stopping the IBM Content Search Services index dispatcher.${RESET_TEXT}"
                echo "      1. Log in to the Administration Console for Content Platform Engine."
                echo "      2. In the navigation pane, select the domain icon."
                echo "      3. In the edit pane, click the Text Search Subsystem tab and clear the Enable indexing check box."
                echo "      4. Click Save to apply your changes."
                step_num=$((step_num + 1))
            fi
            echo "  - STEP ${step_num} ${RED_TEXT}(Required)${RESET_TEXT}: You need to run ${GREEN_TEXT}\"${CUR_DIR}/cp4a-deployment.sh -m upgradeDeployment -n $TARGET_PROJECT_NAME\"${RESET_TEXT} to upgrade CP4BA deployment."
            echo "    ${RED_TEXT}[ATTENTION]: ${RESET_TEXT}${YELLOW_TEXT}When you run the [upgradeDeployment] mode of the cp4a-deployment.sh script, the updated custom resource (CR) must be manually applied that all required additional actions can be completed before the upgrade process begins. Refer to the Knowledge Center: \"Updating the custom resource for each capability in your deployment\" topic to complete the REQUIRED steps for the installed pattern(s).${RESET_TEXT}"
            step_num=$((step_num + 1))
            echo "  - STEP ${step_num} ${RED_TEXT}(Required)${RESET_TEXT}: You can run ${GREEN_TEXT}\"${CUR_DIR}/cp4a-deployment.sh -m upgradeDeploymentStatus -n $TARGET_PROJECT_NAME\"${RESET_TEXT} to check whether the upgrade of the CP4BA deployment was successful."
        else
            # for upgrading IFIX by IFIX
            printf "\n"
            echo "${YELLOW_TEXT}[NEXT ACTIONS]:${RESET_TEXT}"
            step_num=1
            echo "  - STEP ${step_num} ${YELLOW_TEXT}(Optional)${RESET_TEXT}: You can run ${GREEN_TEXT}\"${CUR_DIR}/cp4a-deployment.sh -m upgradeOperatorStatus -n $TARGET_PROJECT_NAME\"${RESET_TEXT} to check whether the upgrade of the CP4BA operator and its dependencies was successful."
            printf "\n"
            step_num=$((step_num + 1))
            echo "  - STEP ${step_num} ${RED_TEXT}(Required)${RESET_TEXT}: You can run ${GREEN_TEXT}\"${CUR_DIR}/cp4a-deployment.sh -m upgradeDeploymentStatus -n $TARGET_PROJECT_NAME\"${RESET_TEXT} to check whether the upgrade of the CP4BA deployment was successful."
        fi
    fi

fi
############## End - Migration CPfs mode and upgrade CP4BA Operators ##############

if [ "$RUNTIME_MODE" == "upgradeOperatorStatus" ]; then
    project_name=$TARGET_PROJECT_NAME

    # check current cp4ba version, to verify which version of the CP4BA operator is currently running
    # check_cp4ba_operator_version $TARGET_PROJECT_NAME $ALLOW_DIRECT_UPGRADE

    # Check whether the CP4BA is separation of operators and operands.
    
    check_cp4ba_separate_operand $TARGET_PROJECT_NAME

    UPGRADE_DEPLOYMENT_FOLDER=${CUR_DIR}/cp4ba-upgrade/project/$CP4BA_SERVICES_NS
    UPGRADE_DEPLOYMENT_CR=${UPGRADE_DEPLOYMENT_FOLDER}/custom_resource
    UPGRADE_DEPLOYMENT_CONTENT_CR_TMP=${UPGRADE_DEPLOYMENT_CR}/.content_tmp.yaml
    UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP=${UPGRADE_DEPLOYMENT_CR}/.icp4acluster_tmp.yaml
    mkdir -p $UPGRADE_DEPLOYMENT_CR >/dev/null 2>&1

    if [[ $SEPARATE_OPERAND_FLAG == "Yes" ]]; then
        source ${CUR_DIR}/helper/upgrade/upgrade_merge_yaml.sh $CP4BA_SERVICES_NS $ALLOW_DIRECT_UPGRADE
    else
        source ${CUR_DIR}/helper/upgrade/upgrade_merge_yaml.sh $TARGET_PROJECT_NAME $ALLOW_DIRECT_UPGRADE
    fi

    #Get the name of the CP4BA operator's CSV in the target namespace, to identify if the operator is deployed in the specific project namespace.
    cp4a_operator_csv_name_target_ns=$(${CLI_CMD} get csv -n $TARGET_PROJECT_NAME --no-headers --ignore-not-found | grep "IBM Cloud Pak for Business Automation" | awk '{print $1}')

    #Get the name of the CP4BA operator's CSV in the all-namespaces, to check if the operator is deployed in the all-namespaces mode.
    cp4a_operator_csv_name_allnamespace_ns=$(${CLI_CMD} get csv -n $ALL_NAMESPACE_NAME --no-headers --ignore-not-found | grep "IBM Cloud Pak for Business Automation" | awk '{print $1}')

    #determine where the CP4BA operator is deployed, in specific ns or all namespace.
    if [[ -z $cp4a_operator_csv_name_allnamespace_ns && (! -z $cp4a_operator_csv_name_target_ns) ]]; then
        #when CP4BA operator is deployed only in the target project namespace.
        success "The IBM Cloud Pak for Business Automation Operator was found deployed in the project \"$TARGET_PROJECT_NAME\"."
        ALL_NAMESPACE_FLAG="No"
        TEMP_OPERATOR_PROJECT_NAME=$TARGET_PROJECT_NAME
    elif [[ (! -z $cp4a_operator_csv_name_allnamespace_ns) && (! -z $cp4a_operator_csv_name_target_ns) ]]; then
        #when CP4BA operator is deployed in all-namespaces mode.
        success "The IBM Cloud Pak for Business Automation Operator was found deployed as AllNamespace mode in the project \"$ALL_NAMESPACE_NAME\"."
        ALL_NAMESPACE_FLAG="Yes"
        #since all-ns, set project name to the default all-namespaces operators project.
        project_name="openshift-operators"
        TEMP_OPERATOR_PROJECT_NAME="openshift-operators"
    fi

    # Get value of cp4ba_original_csv_ver_for_upgrade_script, for upgrade script to know the original version of CP4BA.
    ibm_cp4ba_shared_info_cm=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS)
    ibm_cp4ba_content_shared_info_cm=$(${CLI_CMD} get configmap ibm-cp4ba-content-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS)
    if [[ ! -z $ibm_cp4ba_shared_info_cm ]]; then
        tmp_csv_val=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info -n $CP4BA_SERVICES_NS -o jsonpath='{.data.cp4ba_original_csv_ver_for_upgrade_script}')
        if [[ ! -z $tmp_csv_val ]]; then
            cp4ba_original_csv_ver_for_upgrade_script=$tmp_csv_val
        fi
    fi
        
    if [[ ! -z $ibm_cp4ba_content_shared_info_cm ]]; then
        tmp_csv_val=$(${CLI_CMD} get configmap ibm-cp4ba-content-shared-info -n $CP4BA_SERVICES_NS -o jsonpath='{.data.cp4ba_original_csv_ver_for_upgrade_script}')
        if [[ ! -z $tmp_csv_val ]]; then
            cp4ba_original_csv_ver_for_upgrade_script=$tmp_csv_val
        fi
    fi

    #Check if the CRD for 'contents.icp4a.ibm.com' exists, this is to manage and interact with content CR
    ${CLI_CMD} get crd |grep contents.icp4a.ibm.com >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        content_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        #Check if a content CR name was successfully retrieved, this is to ensure that we only proceed if a content CR exists in the namespace.
        if [ ! -z $content_cr_name ]; then
            cr_type="content"
            cp4ba_cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
            owner_ref=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)
            if [[ ${owner_ref} == "ICP4ACluster" ]]; then
                echo
            else
                ${CLI_CMD} get $cr_type $content_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_CONTENT_CR_TMP}
                bai_flag=`cat $UPGRADE_DEPLOYMENT_CONTENT_CR_TMP | ${YQ_CMD} r - spec.content_optional_components.bai`
                bai_flag=$(echo $bai_flag | tr '[:upper:]' '[:lower:]')
                CONTENT_CR_EXIST="Yes"
                cp4ba_root_ca_secret_name=`cat $UPGRADE_DEPLOYMENT_CONTENT_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.root_ca_secret`
            fi
        fi
    fi

    #Get the name of the 'icp4acluster' CR in the specified namespace, this is to interact with a specific instance of the 'icp4acluster' CR, we need its name.
    #This step ensures we can perform operations on this resource.
    icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
    if [ ! -z $icp4acluster_cr_name ]; then
        cr_type="icp4acluster"
        cp4ba_cr_metaname=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
        ${CLI_CMD} get $cr_type $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}

        cp4ba_root_ca_secret_name=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.root_ca_secret`
        convert_olm_cr "${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}"
        if [[ $olm_cr_flag == "No" ]]; then
            # Get EXISTING_PATTERN_ARR/EXISTING_OPT_COMPONENT_ARR
            existing_pattern_list=""
            existing_opt_component_list=""

            EXISTING_PATTERN_ARR=()
            EXISTING_OPT_COMPONENT_ARR=()
            existing_pattern_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_patterns`
            existing_opt_component_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_optional_components`

            OIFS=$IFS
            IFS=',' read -r -a EXISTING_PATTERN_ARR <<< "$existing_pattern_list"
            IFS=',' read -r -a EXISTING_OPT_COMPONENT_ARR <<< "$existing_opt_component_list"
            IFS=$OIFS
        fi
    fi
    if [[ -z $content_cr_name && -z $icp4acluster_cr_name ]]; then
        fail "Not found any CP4BA custom resource in cluster in the project \"$CP4BA_SERVICES_NS\"."
        exit 1
    fi

    info "Checking CP4BA operators upgrade done or not"
    #verify whether the upgrade process for the operators is complete, the script verifies the current status of the operators before proceeding.
    check_operator_status $TEMP_OPERATOR_PROJECT_NAME "full" "channel"
    if [[ " ${CHECK_CP4BA_OPERATOR_RESULT[@]} " =~ "FAIL" ]]; then
        fail "Failed to upgrade CP4BA operators"
    else
        success "CP4BA operators upgraded successfully!"
        if [[ ! ("$cp4ba_original_csv_ver_for_upgrade_script" == "$CP4BA_RELEASE_BASE_MAJOR_VERSION"*) ]]; then
            info "All CP4BA operators are shutting down before upgrade Zen/IM/CP4BA capabilities!"
            shutdown_operator $TEMP_OPERATOR_PROJECT_NAME
        fi
        printf "\n"
        echo "${YELLOW_TEXT}[NEXT ACTION]${RESET_TEXT}: "
        #check if the original CSV version is not matching a specific pattern (version "25.0.")
        #check ensures that the shutdown operation is only performed if the version is not the major version, i.e we are not doing a ifix ifix upgrade The version check helps in controlling upgrade.
        CUR_DIR=$(realpath "$(dirname "${BASH_SOURCE[0]}")")
        if [[ ! ("$cp4ba_original_csv_ver_for_upgrade_script" == "$CP4BA_RELEASE_BASE_MAJOR_VERSION"*) ]]; then
            echo "${YELLOW_TEXT}* Run the script in [upgradeDeployment] mode to upgrade the CP4BA deployment when upgrade CP4BA to $CP4BA_RELEASE_BASE.${RESET_TEXT}"
            echo "${GREEN_TEXT}# ${CUR_DIR}/cp4a-deployment.sh -m upgradeDeployment -n $TARGET_PROJECT_NAME${RESET_TEXT}"
        fi
        printf "\n"
        echo "${YELLOW_TEXT}* Run the script in [upgradeDeploymentStatus] mode directly when upgrade CP4BA from $CP4BA_RELEASE_BASE IFix to IFix.${RESET_TEXT}"
        echo "${GREEN_TEXT}# ${CUR_DIR}/cp4a-deployment.sh -m upgradeDeploymentStatus -n $TARGET_PROJECT_NAME${RESET_TEXT}"
    fi
fi

### Running with -m upgradeDeployment mode.
if [ "$RUNTIME_MODE" == "upgradeDeployment" ]; then
    project_name=$TARGET_PROJECT_NAME

    # Check whether the CP4BA is separation of operators and operands.
    check_cp4ba_separate_operand $TARGET_PROJECT_NAME

    #fetch service namespace, which is to determine if the operator is deployed in this specific namespace for upgrade
    cp4a_operator_csv_name_target_ns=$(${CLI_CMD} get csv -n $TARGET_PROJECT_NAME --no-headers --ignore-not-found | grep "IBM Cloud Pak for Business Automation" | awk '{print $1}')
    #fetch service namespace, determine if the operator is deployed in AllNamespace mode, which means it is available across all namespaces.
    cp4a_operator_csv_name_allnamespace_ns=$(${CLI_CMD} get csv -n $ALL_NAMESPACE_NAME --no-headers --ignore-not-found | grep "IBM Cloud Pak for Business Automation" | awk '{print $1}')

    #Check if the operator is NOT deployed in AllNamespace mode and is deployed in the targeted namespace
    if [[ -z $cp4a_operator_csv_name_allnamespace_ns && (! -z $cp4a_operator_csv_name_target_ns) ]]; then
        success "Found IBM Cloud Pak for Business Automation Operator deployed in the project \"$TARGET_PROJECT_NAME\"."
        ALL_NAMESPACE_FLAG="No"
        TEMP_OPERATOR_PROJECT_NAME=$TARGET_PROJECT_NAME
    elif [[ (! -z $cp4a_operator_csv_name_allnamespace_ns) && (! -z $cp4a_operator_csv_name_target_ns) ]]; then
        success "Found IBM Cloud Pak for Business Automation Operator deployed as AllNamespace mode in the project \"$ALL_NAMESPACE_NAME\"."
        ALL_NAMESPACE_FLAG="Yes"
        project_name="openshift-operators"
        TEMP_OPERATOR_PROJECT_NAME="openshift-operators"
    fi

    # Get value of cp4ba_original_csv_ver_for_upgrade_script, which stores csv and used in setup of environment
    ibm_cp4ba_shared_info_cm=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS)
    ibm_cp4ba_content_shared_info_cm=$(${CLI_CMD} get configmap ibm-cp4ba-content-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS)
    if [[ ! -z $ibm_cp4ba_shared_info_cm ]]; then
        tmp_csv_val=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info -n $CP4BA_SERVICES_NS -o jsonpath='{.data.cp4ba_original_csv_ver_for_upgrade_script}')
        if [[ ! -z $tmp_csv_val ]]; then
            cp4ba_original_csv_ver_for_upgrade_script=$tmp_csv_val
        fi
    fi
        
    if [[ ! -z $ibm_cp4ba_content_shared_info_cm ]]; then
        tmp_csv_val=$(${CLI_CMD} get configmap ibm-cp4ba-content-shared-info -n $CP4BA_SERVICES_NS -o jsonpath='{.data.cp4ba_original_csv_ver_for_upgrade_script}')
        if [[ ! -z $tmp_csv_val ]]; then
            cp4ba_original_csv_ver_for_upgrade_script=$tmp_csv_val
        fi
    fi
    
    #exit if CSV version is 25.0.*, upgradedeployment is not required if the original CSV version is 25.0.*
    if [[ "$cp4ba_original_csv_ver_for_upgrade_script" == "$CP4BA_RELEASE_BASE_MAJOR_VERSION"* ]]; then
        warning "DO NOT NEED to run [upgradeDeployment] mode for upgrading from ${CP4BA_RELEASE_BASE}GA/${CP4BA_RELEASE_BASE}.X to ${CP4BA_RELEASE_BASE}.X"
        echo "Exiting ..."
        exit 1
    fi

    #check whether operands (resources managed by the operator) are deployed separately from the main CP4BA project
    if [[ $SEPARATE_OPERAND_FLAG == "Yes" ]]; then
        source ${CUR_DIR}/helper/upgrade/upgrade_merge_yaml.sh $CP4BA_SERVICES_NS $ALLOW_DIRECT_UPGRADE
    else
        source ${CUR_DIR}/helper/upgrade/upgrade_merge_yaml.sh $TARGET_PROJECT_NAME $ALLOW_DIRECT_UPGRADE
    fi

    ${CLI_CMD} get crd |grep contents.icp4a.ibm.com >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        # Get the name of the 'content' custom resource in the namespace, to check specific 'content' resource that we need
        content_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        if [ ! -z $content_cr_name ]; then
            # info "Retrieving existing CP4BA Content (Kind: content.icp4a.ibm.com) Custom Resource"
            cr_type="content"
            cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
            owner_ref=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)
            if [[ ${owner_ref} != "ICP4ACluster" ]]; then
                # Get the application version from the custom resource specification, to check if the cr version is up-to-date or need an upgrade.
                cr_version=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.appVersion)
                if [[ $cr_version == "${CP4BA_RELEASE_BASE}" ]]; then
                    warning "The release version of content custom resource \"$content_cr_name\" is already \"$cr_version\". "
                    printf "\n"
                    while true; do
                        printf "\x1B[1mDo you want to continue running the  upgrade? (Yes/No, default: No): \x1B[0m"
                        read -rp "" ans
                        case "$ans" in
                        "y"|"Y"|"yes"|"Yes"|"YES")
                            RERUN_UPGRADE_DEPLOYMENT="Yes"
                            break
                            ;;
                        "n"|"N"|"no"|"No"|"NO"|"")
                            echo "Exiting..."
                            exit 1
                            ;;
                        *)
                            echo -e "Answer must be \"Yes\" or \"No\"\n"
                            ;;
                        esac
                    done
                else #DBACLD-160277: Setting the original version of the content CR to the current version of the content CR
                    export original_cr_version=$cr_version
                fi
            fi
        fi
    fi

    #Retrieve the name of the 'icp4acluster' custom resource in the namespace,to check its version and decide if an upgrade is necessary
    icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
    # Check if a CR name was retrieved, to check 'icp4acluster' custom resource exist in the specified namespace
    if [ ! -z $icp4acluster_cr_name ]; then
        cr_version=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.appVersion)
        if [[ $cr_version == "${CP4BA_RELEASE_BASE}" ]]; then
            warning "The release version of icp4acluster custom resource \"$icp4acluster_cr_name\" is already \"$cr_version\"."
            printf "\n"
            while true; do
                printf "\x1B[1mDo you want to continue running the upgrade? (Yes/No, default: No): \x1B[0m"
                read -rp "" ans
                case "$ans" in
                "y"|"Y"|"yes"|"Yes"|"YES")
                    RERUN_UPGRADE_DEPLOYMENT="Yes"
                    break
                    ;;
                "n"|"N"|"no"|"No"|"NO"|"")
                    echo "Exiting..."
                    exit 1
                    ;;
                *)
                    echo -e "Answer must be \"Yes\" or \"No\"\n"
                    ;;
                esac
            done
        else #DBACLD-160277: Setting the original version of the content CR to the current version of the content CR
            export original_cr_version=$cr_version
        fi
    fi
    # info "Starting to upgrade CP4BA Deployment..."
    # info "Incomming..."

    # trap 'startup_operator $TARGET_PROJECT_NAME' EXIT
    # info "Checking CP4BA operator and dependencies ready or not"
    # check_operator_status $TARGET_PROJECT_NAME
    # if [[ " ${CHECK_CP4BA_OPERATOR_RESULT[@]} " =~ "FAIL" ]]; then
    #     fail "CP4BA or dependency operaotrs is NOT ready all!"
    #     exit 1
    # else
    #     info "The CP4BA and dependency operaotrs is ready for upgrade CP4BA deployment!"
    # fi
    # create_upgrade_property
    # if [[ -z $UPGRADE_MODE ]]; then
    #     cs_dedicated=$(${CLI_CMD} get cm -n ${COMMON_SERVICES_CM_NAMESPACE}  | grep ${COMMON_SERVICES_CM_DEDICATED_NAME} | awk '{print $1}')

    #     cs_shared=$(${CLI_CMD} get cm -n ${COMMON_SERVICES_CM_NAMESPACE}  | grep ${COMMON_SERVICES_CM_SHARED_NAME} | awk '{print $1}')

    #     if [[ "$cs_dedicated" != "" || "$cs_shared" != ""  ]] ; then
    #         control_namespace=$(${CLI_CMD} get cm ${COMMON_SERVICES_CM_DEDICATED_NAME} --no-headers --ignore-not-found -n ${COMMON_SERVICES_CM_NAMESPACE} -o jsonpath='{ .data.common-service-maps\.yaml }' | grep  'controlNamespace' | cut -d':' -f2 )
    #         control_namespace=$(sed -e 's/^"//' -e 's/"$//' <<<"$control_namespace")
    #         control_namespace=$(sed "s/ //g" <<< $control_namespace)
    #     fi

    #     # For shared to shared, the common-service-maps be created under kube-public also.
    #     # So the script need to check structure of common-service-maps to decide this is shared or dedicated
    #     if [[ ("$cs_dedicated" != "" && "$cs_shared" == "") || (! -z $control_namespace)  ]]; then
    #         UPGRADE_MODE="dedicated2dedicated"
    #     elif [[ $ALL_NAMESPACE_FLAG == "No" ]]; then
    #         # Dedicde upgrade mode by customer
    #         select_upgrade_mode
    #     elif [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
    #         UPGRADE_MODE="shared2shared"
    #         ENABLE_PRIVATE_CATALOG=0
    #         info "Found the IBM Cloud Pak foundational services/IBM Cloud Pak for Business Automation are in all namespaces, IBM Cloud Pak foundational services only can be migrated from \"Cluster-scoped\" to \"Cluster-scoped\"!"
    #     fi
    # fi

    # if [[ $UPGRADE_MODE == "shared2shared" ]]; then
    #     ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TEMP_OPERATOR_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #     ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"ibm-common-services\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    # elif [[ $UPGRADE_MODE == "shared2dedicated" || $UPGRADE_MODE == "dedicated2dedicated" ]]; then
    #     ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #     ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    # fi

    # if [[ "$cs_dedicated" != "" && "$cs_shared" == ""  ]]; then
    #     UPGRADE_MODE="dedicated2dedicated"
    #     ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #     ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    # elif [[ "$cs_dedicated" != "" && "$cs_shared" != "" ]]; then
    #     ${CLI_CMD} get cm ${COMMON_SERVICES_CM_DEDICATED_NAME} --no-headers --ignore-not-found -n ${COMMON_SERVICES_CM_NAMESPACE} -o jsonpath='{ .data.common-service-maps\.yaml }' > /tmp/common-service-maps.yaml
    #     common_service_namespace=`cat /tmp/common-service-maps.yaml | ${YQ_CMD} r - namespaceMapping.[0].map-to-common-service-namespace`
    #     common_service_flag=`cat /tmp/common-service-maps.yaml | ${YQ_CMD} r - namespaceMapping.[1].map-to-common-service-namespace`
    #     if [[ -z $common_service_flag && $common_service_namespace == "ibm-common-services" ]]; then
    #         UPGRADE_MODE="shared2shared"
    #         ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"ibm-common-services\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #         if [[ $ALL_NAMESPACE_FLAG == "Yes" ]]; then
    #             ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$ALL_NAMESPACE_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #         elif [[ $ALL_NAMESPACE_FLAG == "No" ]]; then
    #             ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #         fi
    #     elif [[ $common_service_flag != ""  ]]; then
    #         UPGRADE_MODE="shared2dedicated"
    #         ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #         ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #     elif [[ $common_service_flag == ""  ]]; then
    #         UPGRADE_MODE="dedicated2dedicated"
    #         ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #         ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #     fi
    # elif  [[ "$cs_dedicated" == "" && "$cs_shared" != "" && $ALL_NAMESPACE_FLAG == "No" ]]; then
    #     # Dedicde upgrade mode by customer
    #     select_upgrade_mode
    #     if [[ $UPGRADE_MODE == "shared2shared" ]]; then
    #         ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TEMP_OPERATOR_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #         ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"ibm-common-services\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #     elif [[ $UPGRADE_MODE == "shared2dedicated" ]]; then
    #         ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #         ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"$TARGET_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #     fi
    # elif  [[ "$cs_dedicated" == "" && "$cs_shared" != "" && $ALL_NAMESPACE_FLAG == "Yes" ]]; then
    #     UPGRADE_MODE="shared2shared"
    #     ${SED_COMMAND} "s|CS_OPERATOR_NAMESPACE=\"\"|CS_OPERATOR_NAMESPACE=\"$TEMP_OPERATOR_PROJECT_NAME\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    #     ${SED_COMMAND} "s|CS_SERVICES_NAMESPACE=\"\"|CS_SERVICES_NAMESPACE=\"ibm-common-services\"|g" ${UPGRADE_DEPLOYMENT_PROPERTY_FILE}
    # fi


    # $TARGET_PROJECT_NAME for cp4ba deployment, $TEMP_OPERATOR_PROJECT_NAME for cp4ba operators
    upgrade_deployment $CP4BA_SERVICES_NS $TEMP_OPERATOR_PROJECT_NAME $ALLOW_DIRECT_UPGRADE

    echo "${YELLOW_TEXT}[TIPS]${RESET_TEXT}"
    echo "* When running the script in [upgradeDeploymentStatus] mode, it will detect whether the Zen/IM ready or not."
    echo "* After Zen/IM is ready, the script will automatically start all CP4BA operators."
    printf "\n"
    echo "If the script runs in [upgradeDeploymentStatus] mode to check the Zen/IM timeout, you could check the status by running the below command."
    msgB "To manually check the zenService version: "
    echo "  # ${CLI_CMD} get zenService $(${CLI_CMD} get zenService --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS |awk '{print $1}') --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.status.currentVersion}'"
    printf "\n"
    msgB "To manually check zenService status and progress: "
    echo "  # ${CLI_CMD} get zenService $(${CLI_CMD} get zenService --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS |awk '{print $1}') --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.status.zenStatus}'"
    echo "  # ${CLI_CMD} get zenService $(${CLI_CMD} get zenService --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS |awk '{print $1}') --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.status.progress}'"

    # if [[  " ${EXISTING_OPT_COMPONENT_ARR[@]} " =~ "bai" || "${bai_flag}" == "true" ]]; then
    #     printf "\n"
    #     echo -e "\x1B[33;5m[ATTENTION]: \x1B[0m${YELLOW_TEXT}AFTER UPGRADING IBM CLOUD PAK FOR BUSINESS AUTOMATION (CP4BA) DEPLOYMENT SUCCESSFULLY, YOU NEED TO REMOVE${RESET_TEXT} ${RED_TEXT}\"recovery_path\"${RESET_TEXT} ${YELLOW_TEXT}FROM CUSTOM RESOURCE UNDER${RESET_TEXT} ${RED_TEXT}\"bai_configuration\"${RESET_TEXT} ${YELLOW_TEXT}MANUALLY IF EXISTING.${RESET_TEXT}"
    # fi

    # if [[ $CONTENT_CR_EXIST == "Yes" || " ${EXISTING_PATTERN_ARR[@]}" =~ "workflow-runtime" || " ${EXISTING_PATTERN_ARR[@]}" =~ "workflow-authoring" || " ${EXISTING_PATTERN_ARR[@]}" =~ "content" || " ${EXISTING_PATTERN_ARR[@]}" =~ "document_processing" || "${EXISTING_OPT_COMPONENT_ARR[@]}" =~ "ae_data_persistence" ]]; then
    #     printf "\n"
    #     echo -e "\x1B[33;5m[ATTENTION]: \x1B[0m${YELLOW_TEXT}AFTER UPGRADING IBM CLOUD PAK FOR BUSINESS AUTOMATION (CP4BA) DEPLOYMENT SUCCESSFULLY, YOU NEED TO RUN${RESET_TEXT} ${RED_TEXT}\"./cp4a-pre-upgrade-and-post-upgrade-optional.sh post-upgrade\"${RESET_TEXT}, ${YELLOW_TEXT}AND THEN CLEAN BROWSER COOKIE BEFORE LOGIN${RESET_TEXT} ${RED_TEXT}[NOTES: DO NOT need to run it when upgrade CP4BA from 23.0.2.X to 24.0.0 (migration IBM Cloud Pak foundational services from Cluster-scoped -> Cluster-scoped or Namespace-scoped -> Namespace-scoped)]${RESET_TEXT}"
    # fi
fi
### End of running with -m upgradeDeployment mode.

### Beginning of running with -m upgradeDeploymentStatus mode
if [[ "$RUNTIME_MODE" == "upgradeDeploymentStatus" ]]; then
    UPGRADE_DEPLOYMENT_FOLDER=${CUR_DIR}/cp4ba-upgrade/project/$TARGET_PROJECT_NAME
    TEMP_CP_CONSOLE_FILE=${UPGRADE_DEPLOYMENT_FOLDER}/original-cp-console.yaml
    TEMP_CP_CONSOLE_FILE_ID_PROVIDER=${UPGRADE_DEPLOYMENT_FOLDER}/id-provider-cp-console.yaml
    TEMP_CP_CONSOLE_FILE_ID_MGMT=${UPGRADE_DEPLOYMENT_FOLDER}/id-mgmt-cp-console.yaml
    source ${CUR_DIR}/helper/upgrade/upgrade_merge_yaml.sh $TARGET_PROJECT_NAME $ALLOW_DIRECT_UPGRADE

    CP_CONSOLE='cp-console'
    ID_PROVIDER_ROUTE_NAME='cp-console-iam-provider'
    ID_PROVIDER_PATH='/idprovider/'
    ID_MGMT_ROUTE_NAME='cp-console-iam-idmgmt'
    ID_MGMT_PATH='/idmgmt/'

    function get_default_cp_console_route() {

        if [ ! -d $UPGRADE_DEPLOYMENT_FOLDER ]; then
            mkdir $UPGRADE_DEPLOYMENT_FOLDER
        fi

        rm -fr $TEMP_CP_CONSOLE_FILE
        local tmp_cp_console=$( ${CLI_CMD} get route $CP_CONSOLE --no-headers --ignore-not-found  -n $TARGET_PROJECT_NAME_CS | awk '{print $1}' )
        if [ -n $tmp_cp_console  ]; then
            # echo -e "${BLUE}Creating backup yaml for route $CP_CONSOLE${COLOR_OFF}"
            ${CLI_CMD} get route $CP_CONSOLE -o yaml -n $TARGET_PROJECT_NAME_CS > $TEMP_CP_CONSOLE_FILE
            CP_CONSOLE_HOST=$(${YQ_CMD} r $TEMP_CP_CONSOLE_FILE spec.host )
            ID_MGMT_CP_CONSOLE=$( echo "id-mgmt-${CP_CONSOLE_HOST}" | sed   "s/-$TARGET_PROJECT_NAME_CS//g" )
            ID_PROVIDER_CP_CONSOLE=$( echo "id-provider-${CP_CONSOLE_HOST}" | sed  "s/-$TARGET_PROJECT_NAME_CS//g")
            cp $TEMP_CP_CONSOLE_FILE $TEMP_CP_CONSOLE_FILE_ID_PROVIDER
            cp $TEMP_CP_CONSOLE_FILE $TEMP_CP_CONSOLE_FILE_ID_MGMT
        else
            warning "Could not find the route \"$CP_CONSOLE\" in the project \"$TARGET_PROJECT_NAME_CS\""
            return 1
        fi
        return 0
    }

    function create_custom_idprovider_route() {

        if [ -a ${TEMP_CP_CONSOLE_FILE_ID_PROVIDER} ]; then

            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER metadata.name "$ID_PROVIDER_ROUTE_NAME"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER spec.path "$ID_PROVIDER_PATH"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER spec.host "$ID_PROVIDER_CP_CONSOLE"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER metadata.labels.path "idprovider"


            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER metadata.ownerReferences
            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER metadata.uid
            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER metadata.resourceVersion
            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER status
            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER metadata.creationTimestamp

            if [[ "$1" == "platform-identity-provider" ]]; then
            ${SED_COMMAND} "s/-$TARGET_PROJECT_NAME_CS//g" $TEMP_CP_CONSOLE_FILE_ID_PROVIDER
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER spec.to.name "platform-identity-provider"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER spec.port.targetPort "4300"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_PROVIDER 'metadata.annotations."haproxy.router.openshift.io/rewrite-target"' '/'
            fi

            # echo -e "Creating new route named $ID_PROVIDER_ROUTE_NAME"
            ${CLI_CMD} apply -f $TEMP_CP_CONSOLE_FILE_ID_PROVIDER -n $TARGET_PROJECT_NAME_CS >/dev/null 2>&1
            if [ $? -ne 0 ]; then
                warning "Failed to create route named $ID_PROVIDER_ROUTE_NAME in the project \"$TARGET_PROJECT_NAME_CS\"!"
                exit 1
            fi
        else
            fail "File not found: ${TEMP_CP_CONSOLE_FILE_ID_PROVIDER}"
            return -1
        fi

        return 0

    }

    function create_custom_idmgmt_route() {

        if [ -a ${TEMP_CP_CONSOLE_FILE_ID_MGMT} ]; then

            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_MGMT metadata.name "$ID_MGMT_ROUTE_NAME"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_MGMT metadata.labels.path "idmgmt"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_MGMT spec.path "$ID_MGMT_PATH"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_MGMT spec.host "$ID_MGMT_CP_CONSOLE"

            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_MGMT metadata.ownerReferences
            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_MGMT metadata.uid
            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_MGMT metadata.resourceVersion
            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_MGMT status
            ${YQ_CMD} d -i $TEMP_CP_CONSOLE_FILE_ID_MGMT metadata.creationTimestamp

            if [[ "$1" == "platform-identity-management" ]]; then
            ${SED_COMMAND} "s/-$TARGET_PROJECT_NAME_CS//g" $TEMP_CP_CONSOLE_FILE_ID_MGMT
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_MGMT spec.to.name "platform-identity-management"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_MGMT spec.port.targetPort "4500"
            ${YQ_CMD} w -i $TEMP_CP_CONSOLE_FILE_ID_MGMT 'metadata.annotations."haproxy.router.openshift.io/rewrite-target"' '/'

            fi

            # echo -e "Creating new route named $ID_MGMT_ROUTE_NAME"
            ${CLI_CMD} apply -f $TEMP_CP_CONSOLE_FILE_ID_MGMT -n $TARGET_PROJECT_NAME_CS >/dev/null 2>&1
            if [ $? -ne 0 ]; then
                warning "Failed to create route named $ID_MGMT_ROUTE_NAME in the project \"$TARGET_PROJECT_NAME_CS\"!"
                exit 1
            fi
        else
            fail "File not found: ${TEMP_CP_CONSOLE_FILE_ID_MGMT}"
            return -1
        fi
        return 0
    }

    project_name=$TARGET_PROJECT_NAME
    # Check whether the CP4BA is separation of operators and operands.
    check_cp4ba_separate_operand $TARGET_PROJECT_NAME

    cp4a_operator_csv_name_target_ns=$(${CLI_CMD} get csv -n $TARGET_PROJECT_NAME --no-headers --ignore-not-found | grep "IBM Cloud Pak for Business Automation" | awk '{print $1}')
    cp4a_operator_csv_name_allnamespace_ns=$(${CLI_CMD} get csv -n $ALL_NAMESPACE_NAME --no-headers --ignore-not-found | grep "IBM Cloud Pak for Business Automation" | awk '{print $1}')

    if [[ -z $cp4a_operator_csv_name_allnamespace_ns && (! -z $cp4a_operator_csv_name_target_ns) ]]; then
        success "The IBM Cloud Pak for Business Automation Operator was found deployed in the project \"$TARGET_PROJECT_NAME\"."
        ALL_NAMESPACE_FLAG="No"
        TEMP_OPERATOR_PROJECT_NAME=$TARGET_PROJECT_NAME
    elif [[ (! -z $cp4a_operator_csv_name_allnamespace_ns) && (! -z $cp4a_operator_csv_name_target_ns) ]]; then
        success "The IBM Cloud Pak for Business Automation Operator was found deployed as AllNamespace mode in the project \"$ALL_NAMESPACE_NAME\"."
        ALL_NAMESPACE_FLAG="Yes"
        project_name="openshift-operators"
        TEMP_OPERATOR_PROJECT_NAME="openshift-operators"
    fi

    # Get value of cp4ba_original_csv_ver_for_upgrade_script
    ibm_cp4ba_shared_info_cm=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS)
    ibm_cp4ba_content_shared_info_cm=$(${CLI_CMD} get configmap ibm-cp4ba-content-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS)
    if [[ ! -z $ibm_cp4ba_shared_info_cm ]]; then
        tmp_csv_val=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info -n $CP4BA_SERVICES_NS -o jsonpath='{.data.cp4ba_original_csv_ver_for_upgrade_script}')
        if [[ ! -z $tmp_csv_val ]]; then
            cp4ba_original_csv_ver_for_upgrade_script=$tmp_csv_val
        fi
    fi
        
    if [[ ! -z $ibm_cp4ba_content_shared_info_cm ]]; then
        tmp_csv_val=$(${CLI_CMD} get configmap ibm-cp4ba-content-shared-info -n $CP4BA_SERVICES_NS -o jsonpath='{.data.cp4ba_original_csv_ver_for_upgrade_script}')
        if [[ ! -z $tmp_csv_val ]]; then
            cp4ba_original_csv_ver_for_upgrade_script=$tmp_csv_val
        fi
    fi

    # Function to remove the old operands that will conflict with the new version of Events Operator
    # Function defined in common.sh
    # https://jsw.ibm.com/browse/DBACLD-178674
    delete_cpfs_operand_requests "$TARGET_PROJECT_NAME"

    ${CLI_CMD} get crd | grep contents.icp4a.ibm.com >/dev/null 2>&1
    if [ $? -eq 0 ]; then
        content_cr_name=$(${CLI_CMD} get content -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
        if [[ ! -z $content_cr_name ]]; then
            # info "Retrieving existing CP4BA Content (Kind: content.icp4a.ibm.com) Custom Resource"
            cr_type="content"
            cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
            owner_ref=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)
            if [[ "$owner_ref" != "ICP4ACluster" ]]; then
                CONTENT_CR_EXIST="Yes"
                info "Scaling up \"IBM CP4BA FileNet Content Manager\" operator"
                ${CLI_CMD} scale --replicas=1 deployment ibm-content-operator -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
                if [ $? -eq 0 ]; then
                    sleep 1
                else
                    fail "Failed to scale up \"IBM CP4BA FileNet Content Manager\" operator"
                fi

                info "Scaling up \"IBM CP4BA Foundation\" operator"
                ${CLI_CMD} scale --replicas=1 deployment icp4a-foundation-operator -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
                if [ $? -eq 0 ]; then
                    sleep 1
                else
                    fail "Failed to scale up \"IBM CP4BA Foundation\" operator"
                fi
                cr_version=$(${CLI_CMD} get content $content_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.appVersion)
                if [[ $cr_version != "${CP4BA_RELEASE_BASE}" ]]; then
                    fail "The release version: \"$cr_version\" in content custom resource \"$content_cr_name\" is incorrect. Apply the new version of the CR first."
                    exit 1
                fi
            else
                CONTENT_CR_EXIST="No"
            fi
        fi
    fi

    icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}')
    if [ ! -z $icp4acluster_cr_name ]; then
        info "Scaling up \"IBM Cloud Pak for Business Automation (CP4BA) multi-pattern\" operator"
        ${CLI_CMD} scale --replicas=1 deployment ibm-cp4a-operator -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            sleep 1
        else
            fail "Failed to scale up \"IBM Cloud Pak for Business Automation (CP4BA) multi-pattern\" operator"
        fi

        info "Scaling up \"IBM CP4BA FileNet Content Manager\" operator"
        ${CLI_CMD} scale --replicas=1 deployment ibm-content-operator -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            sleep 1
        else
            fail "Failed to scale up \"IBM CP4BA FileNet Content Manager\" operator"
        fi

        info "Scaling up \"IBM CP4BA Foundation\" operator"
        ${CLI_CMD} scale --replicas=1 deployment icp4a-foundation-operator -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
        if [ $? -eq 0 ]; then
            sleep 1
        else
            fail "Failed to scale up \"IBM CP4BA Foundation\" operator"
        fi

        cr_version=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - spec.appVersion)
        if [[ $cr_version != "${CP4BA_RELEASE_BASE}" ]]; then
            fail "The release version: \"$cr_version\" in icp4acluster custom resource \"$icp4acluster_cr_name\" is incorrect. Apply the new version of the CR first."
            exit 1
        fi
    fi
    # info "Scaling up \"IBM CP4BA FileNet Content Manager\" operator"
    # ${CLI_CMD} scale --replicas=1 deployment ibm-content-operator -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
    # if [ $? -eq 0 ]; then
    #     sleep 1
    #     echo "Done!"
    # else
    #     fail "Failed to scale up \"IBM CP4BA FileNet Content Manager\" operator"
    # fi
    # info "Scaling up \"IBM Cloud Pak for Business Automation (CP4BA) multi-pattern\" operator"
    # ${CLI_CMD} scale --replicas=1 deployment ibm-cp4a-operator -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
    # if [ $? -eq 0 ]; then
    #     sleep 1
    #     echo "Done!"
    # else
    #     fail "Failed to scale up \"IBM Cloud Pak for Business Automation (CP4BA) multi-pattern\" operator"
    # fi

    # info "Scaling up \"IBM CP4BA Foundation\" operator"
    # ${CLI_CMD} scale --replicas=1 deployment icp4a-foundation-operator -n $TEMP_OPERATOR_PROJECT_NAME >/dev/null 2>&1
    # if [ $? -eq 0 ]; then
    #     sleep 1
    #     echo "Done!"
    # else
    #     fail "Failed to scale up \"IBM CP4BA Foundation\" operator"
    # fi

    while true; do
        clear
        isReady_cp4ba=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.data.cp4ba_operator_of_last_reconcile}')
        isReady_foundation=$(${CLI_CMD} get configmap ibm-cp4ba-shared-info --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.data.foundation_operator_of_last_reconcile}')
        if [[ -z "$isReady_cp4ba" && -z "$isReady_foundation" ]]; then
            CP4BA_DEPLOYMENT_STATUS="Getting Upgrade Status ..."
            printf '%s %s\n' "$(date)" "[refresh interval: 30s]"
            echo -en "[Press Ctrl+C to exit] \t\t"
            printHeaderMessage "CP4BA Upgrade Status"
            echo -en "${GREEN_TEXT}$CP4BA_DEPLOYMENT_STATUS${RESET_TEXT}"
            sleep 30
        else
            break
        fi
    done

   # check for zenStatus and currentverison for zen

    zen_service_name=$(${CLI_CMD} get zenService --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS |awk '{print $1}')
    if [[ ! -z "$zen_service_name" ]]; then
        clear
        maxRetry=360
        for ((retry=0;retry<=${maxRetry};retry++)); do
            # As workaround for https://github.ibm.com/IBMPrivateCloud/roadmap/issues/64207
            # update secret postgresql-operator-controller-manager-config in <cp4ba> namespace and/or ibm-common-services namespace and add this annotation ibm-bts/skip-updates: "true"
            if ${CLI_CMD} get secret -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | grep postgresql-operator-controller-manager-config >/dev/null 2>&1; then
                ${CLI_CMD} patch secret postgresql-operator-controller-manager-config -n $CP4BA_SERVICES_NS -p '{"metadata": {"annotations": {"ibm-bts/skip-updates": "true"}}}' >/dev/null 2>&1
            fi

            zenservice_version=$(${CLI_CMD} get zenService $zen_service_name --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.status.currentVersion}')
            isCompleted=$(${CLI_CMD} get zenService $zen_service_name --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.status.zenStatus}')
            isProgressDone=$(${CLI_CMD} get zenService $zen_service_name --no-headers --ignore-not-found -n $CP4BA_SERVICES_NS -o jsonpath='{.status.progress}')

            if [[ "$isCompleted" != "Completed" || "$isProgressDone" != "100%" || "$zenservice_version" != "${ZEN_OPERATOR_VERSION//v/}" ]]; then
                clear
                CP4BA_DEPLOYMENT_STATUS="Waiting for the zenService to be ready (could take up to 120 minutes) before upgrade the CP4BA capabilities..."
                printf '%s %s\n' "$(date)" "[refresh interval: 60s]"
                echo -en "[Press Ctrl+C to exit] \t\t"
                printf "\n"
                echo "${YELLOW_TEXT}$CP4BA_DEPLOYMENT_STATUS${RESET_TEXT}"
                printHeaderMessage "CP4BA Upgrade Status"
                if [[ "$zenservice_version" == "${ZEN_OPERATOR_VERSION//v/}" ]]; then
                    echo "zenService Version (Expected - ${ZEN_OPERATOR_VERSION//v/})       : ${GREEN_TEXT}$zenservice_version${RESET_TEXT}"
                else
                    echo "zenService Version (Expected - ${ZEN_OPERATOR_VERSION//v/})       : ${RED_TEXT}$zenservice_version${RESET_TEXT}"
                fi
                if [[ "$isCompleted" == "Completed" && "$zenservice_version" == "${ZEN_OPERATOR_VERSION//v/}" ]]; then
                    echo "zenService Status (Expected - Completed)    : ${GREEN_TEXT}$isCompleted${RESET_TEXT}"
                else
                    echo "zenService Status (Expected - Completed)    : ${RED_TEXT}$isCompleted${RESET_TEXT}"
                fi

                if [[ "$isProgressDone" == "100%" && "$zenservice_version" == "${ZEN_OPERATOR_VERSION//v/}" ]]; then
                    echo "zenService Progress (Expected - 100%)       : ${GREEN_TEXT}$isProgressDone${RESET_TEXT}"
                else
                    echo "zenService Progress (Expected - 100%)       : ${RED_TEXT}$isProgressDone${RESET_TEXT}"
                fi
                sleep 60
            elif [[ "$isCompleted" == "Completed" && "$isProgressDone" == "100%" && "$zenservice_version" == "${ZEN_OPERATOR_VERSION//v/}" ]]; then
                break
            elif [[ $retry -eq ${maxRetry} ]]; then
                printf "\n"
                warning "Timeout waiting for the Zen Service to start"
                echo -e "\x1B[1mCheck the status of the Zen Service\x1B[0m"
                printf "\n"
                exit 1
            fi
        done
        clear
        # success "The Zen Service (${ZEN_OPERATOR_VERSION//v/}) is ready for CP4BA"
        CP4BA_DEPLOYMENT_STATUS="The Zen Service (${ZEN_OPERATOR_VERSION//v/}) is ready for CP4BA"
        printf '%s %s\n' "$(date)" "[refresh interval: 30s]"
        echo -en "[Press Ctrl+C to exit] \t\t"
        printf "\n"
        echo "${YELLOW_TEXT}$CP4BA_DEPLOYMENT_STATUS${RESET_TEXT}"
        info "Starting all CP4BA Operators to upgrade CP4BA capabilities"
        printHeaderMessage "CP4BA Upgrade Status"
        if [[ "$zenservice_version" == "${ZEN_OPERATOR_VERSION//v/}" ]]; then
            echo "zenService Version        : ${GREEN_TEXT}$zenservice_version${RESET_TEXT}"
        else
            echo "zenService Version        : ${RED_TEXT}$zenservice_version${RESET_TEXT}"
        fi
        if [[ "$isCompleted" == "Completed" ]]; then
            echo "zenService Status         : ${GREEN_TEXT}$isCompleted${RESET_TEXT}"
        else
            echo "zenService Status         : ${RED_TEXT}$isCompleted${RESET_TEXT}"
        fi

        if [[ "$isProgressDone" == "100%" && "$zenservice_version" == "${ZEN_OPERATOR_VERSION//v/}" ]]; then
            echo "zenService Progress       : ${GREEN_TEXT}$isProgressDone${RESET_TEXT}"
        else
            echo "zenService Progress       : ${RED_TEXT}$isProgressDone${RESET_TEXT}"
        fi

        if [[ ! ("$cp4ba_original_csv_ver_for_upgrade_script" == "24."*) && "$ALLOW_DIRECT_UPGRADE" == 1 ]]; then
            ## Create tow route after zenService ready
            TARGET_PROJECT_NAME_CS=$(${CLI_CMD} get route --no-headers --ignore-not-found  -A |grep  cp-console-iam-provider|awk '{print $1}')
            if [[ -z $TARGET_PROJECT_NAME_CS ]]; then
                warning "cp-console-iam-provider not found in the cluster. continuing..."
            else
                get_default_cp_console_route
                res=$?
                if [[ ${res} == "0" ]]; then
                    create_custom_idprovider_route "platform-identity-provider"
                    create_custom_idmgmt_route "platform-identity-management"
                fi
            fi

            # start all cp4ba operators after zen/im ready
            startup_operator $TEMP_OPERATOR_PROJECT_NAME "silent"
            sleep 10

            ## Apply workaround for https://jsw.ibm.com/browse/DBACLD-137719 before start migration CPfs
            ## scale up ibm-bts-operator-controller-manager in ibm-common-services project after zenService ready
            if [[ $ALL_NAMESPACE_FLAG == "yes" ]]; then
                bts_operator_name=$(${CLI_CMD} get deployment ibm-bts-operator-controller-manager --no-headers --ignore-not-found -n ibm-common-services -o name)
                if [[ ! -z $bts_operator_name ]]; then
                    ${CLI_CMD} scale --replicas=1 deployment ibm-bts-operator-controller-manager -n ibm-common-services >/dev/null 2>&1
                    if [[ $? -ne 0 ]]; then
                        warning "Failed to scale up ibm-bts-operator-controller-manager operator in the project \"ibm-common-services\". Scale up the ibm-bts-operator-controller-manager operator manually."
                    fi
                fi
            fi
        # elif [[ "$is_ifix_to_ifix_upgrade" == "false" ]]; then # Startup operator pods in the target project
        else
            startup_operator $TARGET_PROJECT_NAME "silent"
        fi
    else
        fail "ZenService not found in the project \"$CP4BA_SERVICES_NS\", exiting..."
        echo "****************************************************************************"
        exit 1
    fi

    # show_cp4ba_upgrade_status
    while true
    do
        printf '%s\n' "$(clear; show_cp4ba_upgrade_status)"
        sleep 30
    done
fi
### End of running with -m upgradeDeploymentStatus mode
if [ "$RUNTIME_MODE" == "upgradePostconfig" ]; then
    project_name=$TARGET_PROJECT_NAME

    UPGRADE_DEPLOYMENT_FOLDER=${CUR_DIR}/cp4ba-upgrade/project/$project_name
    UPGRADE_DEPLOYMENT_PROPERTY_FILE=${UPGRADE_DEPLOYMENT_FOLDER}/cp4ba_upgrade.property

    UPGRADE_DEPLOYMENT_CR=${UPGRADE_DEPLOYMENT_FOLDER}/custom_resource
    UPGRADE_DEPLOYMENT_CR_BAK=${UPGRADE_DEPLOYMENT_CR}/backup

    UPGRADE_DEPLOYMENT_CONTENT_CR=${UPGRADE_DEPLOYMENT_CR}/content.yaml
    UPGRADE_DEPLOYMENT_CONTENT_CR_TMP=${UPGRADE_DEPLOYMENT_CR}/.content_tmp.yaml
    UPGRADE_DEPLOYMENT_CONTENT_CR_BAK=${UPGRADE_DEPLOYMENT_CR_BAK}/content_cr_backup.yaml

    UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR=${UPGRADE_DEPLOYMENT_CR}/icp4acluster.yaml
    UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP=${UPGRADE_DEPLOYMENT_CR}/.icp4acluster_tmp.yaml
    UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_BAK=${UPGRADE_DEPLOYMENT_CR_BAK}/icp4acluster_cr_backup.yaml

    mkdir -p ${UPGRADE_DEPLOYMENT_CR} >/dev/null 2>&1
    mkdir -p ${UPGRADE_DEPLOYMENT_CR_BAK} >/dev/null 2>&1

    source ${CUR_DIR}/helper/upgrade/upgrade_merge_yaml.sh $TARGET_PROJECT_NAME $ALLOW_DIRECT_UPGRADE

    info "Starting to execute script for post CP4BA upgrade"
    # Retrieve existing WfPSRuntime CR
    exist_wfps_cr_array=($(${CLI_CMD} get WfPSRuntime -n $CP4BA_SERVICES_NS --no-headers --ignore-not-found | awk '{print $1}'))
    if [ ! -z $exist_wfps_cr_array ]; then
        for item in "${exist_wfps_cr_array[@]}"
        do
            info "Retrieving existing IBM CP4BA Workflow Process Service (Kind: WfPSRuntime.icp4a.ibm.com) Custom Resource: \"${item}\""
            cr_type="WfPSRuntime"
            cr_metaname=$(${CLI_CMD} get $cr_type ${item} -n $CP4BA_SERVICES_NS -o yaml | ${YQ_CMD} r - metadata.name)
            UPGRADE_DEPLOYMENT_WFPS_CR=${UPGRADE_DEPLOYMENT_CR}/wfps_${cr_metaname}.yaml
            UPGRADE_DEPLOYMENT_WFPS_CR_TMP=${UPGRADE_DEPLOYMENT_CR}/.wfps_${cr_metaname}_tmp.yaml
            UPGRADE_DEPLOYMENT_WFPS_CR_BAK=${UPGRADE_DEPLOYMENT_CR_BAK}/wfps_cr_${cr_metaname}_backup.yaml

            ${CLI_CMD} get $cr_type ${item} -n $CP4BA_SERVICES_NS -o yaml > ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP}

            # Backup existing WfPSRuntime CR
            mkdir -p ${UPGRADE_DEPLOYMENT_CR_BAK}
            ${COPY_CMD} -rf ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} ${UPGRADE_DEPLOYMENT_WFPS_CR_BAK}

            info "Merging existing IBM CP4BA Workflow Process Service custom resource: \"${item}\" with new version ($CP4BA_RELEASE_BASE)"
            # Delete unnecessary section in CR
            ${YQ_CMD} d -i ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} status
            ${YQ_CMD} d -i ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} metadata.annotations
            ${YQ_CMD} d -i ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} metadata.creationTimestamp
            ${YQ_CMD} d -i ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} metadata.generation
            ${YQ_CMD} d -i ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} metadata.resourceVersion
            ${YQ_CMD} d -i ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} metadata.uid

            # replace release/appVersion
            # ${SED_COMMAND} "s|release: .*|release: ${CP4BA_RELEASE_BASE}|g" ${UPGRADE_DEPLOYMENT_PFS_CR_TMP}
            ${SED_COMMAND} "s|appVersion: .*|appVersion: ${CP4BA_RELEASE_BASE}|g" ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP}

            # # change failureThreshold/periodSeconds for WfPS after upgrade
            # ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} spec.node.probe.startupProbe.failureThreshold 80
            # ${YQ_CMD} w -i ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} spec.node.probe.startupProbe.periodSeconds 5

            ${SED_COMMAND} "s|'\"|\"|g" ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP}
            ${SED_COMMAND} "s|\"'|\"|g" ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP}


            success "Completed to merge existing IBM CP4BA Workflow Process Service custom resource with new version ($CP4BA_RELEASE_BASE)"
            ${COPY_CMD} -rf ${UPGRADE_DEPLOYMENT_WFPS_CR_TMP} ${UPGRADE_DEPLOYMENT_WFPS_CR}

            info "Apply the new version ($CP4BA_RELEASE_BASE) of IBM CP4BA Workflow Process Service custom resource"
            ${CLI_CMD} annotate WfPSRuntime ${item} kubectl.kubernetes.io/last-applied-configuration- -n $CP4BA_SERVICES_NS >/dev/null 2>&1
            sleep 3
            ${CLI_CMD} apply -f ${UPGRADE_DEPLOYMENT_WFPS_CR} -n $CP4BA_SERVICES_NS >/dev/null 2>&1
            if [ $? -ne 0 ]; then
                fail "IBM CP4BA Workflow Process Service custom resource update failed"
                exit 1
            else
                echo "Done!"

                printf "\n"
                echo "${YELLOW_TEXT}[NEXT ACTION]${RESET_TEXT}:"
                msgB "Run \"cp4a-deployment.sh -m upgradeDeploymentStatus -n $CP4BA_SERVICES_NS\" to get overview upgrade status for IBM CP4BA Workflow Process Service"
            fi
        done
    fi

    # # Retrieve existing Content CR for remove route cp-console-iam-provider/cp-console-iam-idmgmt
    # content_cr_name=$(${CLI_CMD} get content -n $project_name --no-headers --ignore-not-found | awk '{print $1}')
    # if [ ! -z $content_cr_name ]; then
    #     info "Retrieving existing CP4BA Content (Kind: content.icp4a.ibm.com) Custom Resource"
    #     cr_type="content"
    #     cr_metaname=$(${CLI_CMD} get content $content_cr_name -n $project_name -o yaml | ${YQ_CMD} r - metadata.name)
    #     owner_ref=$(${CLI_CMD} get content $content_cr_name -n $project_name -o yaml | ${YQ_CMD} r - metadata.ownerReferences.[0].kind)
    #     if [[ ${owner_ref} != "ICP4ACluster" ]]; then
    #         iam_idprovider=$(${CLI_CMD} get route -n $project_name -o 'custom-columns=NAME:.metadata.name' --no-headers --ignore-not-found | grep cp-console-iam-provider)
    #         iam_idmgmt=$(${CLI_CMD} get route -n $project_name -o 'custom-columns=NAME:.metadata.name' --no-headers --ignore-not-found | grep cp-console-iam-idmgmt)
    #         if [[ ! -z $iam_idprovider ]]; then
    #             info "Remove \"cp-console-iam-provider\" route from project \"$project_name\"."
    #             ${CLI_CMD} delete route $iam_idprovider -n $project_name >/dev/null 2>&1
    #         fi
    #         if [[ ! -z $iam_idmgmt ]]; then
    #             info "Remove \"cp-console-iam-idmgmt\" route from project \"$project_name\"."
    #             ${CLI_CMD} delete route $iam_idmgmt -n $project_name >/dev/null 2>&1
    #         fi
    #     fi
    # fi
    # Retrieve existing ICP4ACluster CR for ADP post upgrade
    icp4acluster_cr_name=$(${CLI_CMD} get icp4acluster -n $project_name --no-headers --ignore-not-found | awk '{print $1}')
    if [ ! -z $icp4acluster_cr_name ]; then
        info "Retrieving existing CP4BA ICP4ACluster (Kind: icp4acluster.icp4a.ibm.com) Custom Resource"
        cr_type="icp4acluster"
        cr_metaname=$(${CLI_CMD} get icp4acluster $icp4acluster_cr_name -n $project_name -o yaml | ${YQ_CMD} r - metadata.name)
        ${CLI_CMD} get $cr_type $icp4acluster_cr_name -n $project_name -o yaml > ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}

        # Backup existing icp4acluster CR
        mkdir -p ${UPGRADE_DEPLOYMENT_CR_BAK}
        ${COPY_CMD} -rf ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP} ${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_BAK}

        convert_olm_cr "${UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP}"
        if [[ $olm_cr_flag == "No" ]]; then
            # Get EXISTING_PATTERN_ARR/EXISTING_OPT_COMPONENT_ARR
            existing_pattern_list=""
            existing_opt_component_list=""

            EXISTING_PATTERN_ARR=()
            EXISTING_OPT_COMPONENT_ARR=()
            existing_pattern_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_deployment_patterns`
            existing_opt_component_list=`cat $UPGRADE_DEPLOYMENT_ICP4ACLUSTER_CR_TMP | ${YQ_CMD} r - spec.shared_configuration.sc_optional_components`

            OIFS=$IFS
            IFS=',' read -r -a EXISTING_PATTERN_ARR <<< "$existing_pattern_list"
            IFS=',' read -r -a EXISTING_OPT_COMPONENT_ARR <<< "$existing_opt_component_list"
            IFS=$OIFS
        fi

        # Remove cp-console-iam-provider/cp-console-iam-idmgmt
        if [[ (" ${EXISTING_PATTERN_ARR[@]} " =~ "document_processing") || (" ${EXISTING_PATTERN_ARR[@]} " =~ "content") || ("${EXISTING_OPT_COMPONENT_ARR[@]}" =~ "ae_data_persistence") || ("${EXISTING_OPT_COMPONENT_ARR[@]}" =~ "baw_authoring") ]]; then
            iam_idprovider=$(${CLI_CMD} get route -n $project_name -o 'custom-columns=NAME:.metadata.name' --no-headers --ignore-not-found | grep cp-console-iam-provider)
            iam_idmgmt=$(${CLI_CMD} get route -n $project_name -o 'custom-columns=NAME:.metadata.name' --no-headers --ignore-not-found | grep cp-console-iam-idmgmt)
            if [[ ! -z $iam_idprovider ]]; then
                info "Remove \"cp-console-iam-provider\" route from project \"$project_name\"."
                ${CLI_CMD} delete route $iam_idprovider -n $project_name >/dev/null 2>&1
            fi
            if [[ ! -z $iam_idmgmt ]]; then
                info "Remove \"cp-console-iam-idmgmt\" route from project \"$project_name\"."
                ${CLI_CMD} delete route $iam_idmgmt -n $project_name >/dev/null 2>&1
            fi
        fi
    fi
    success "Completed to execute script for post CP4BA upgrade"
fi
